#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <assert.h>
#include <linux/if_ether.h>
#include <asm/byteorder.h>
#include <stdlib.h>
#include <pthread.h>
#include "multi_ap.h"
#include "cmdu_message.h"
#include "cmdu_tlv_parse.h"
#include "cmdu.h"
#include "os.h"
#include "byteorder.h"
#include "debug.h"
#include "_1905_lib_io.h"
#include "file_io.h"
#include "message_wait_queue.h"
#include "topology.h"
#include "eloop.h"
#include "mapfilter_if.h"
#include "ethernet_layer.h"

extern unsigned char tlv_temp[];
extern unsigned char dev_send_1905_buf[];
extern int _1905_read_dev_send_1905(struct p1905_managerd_ctx* ctx,
	char* name, unsigned char *almac, unsigned short* _type,
	unsigned short *tlv_len, unsigned char *pay_load);
extern int set_opt_not_forward_dest(unsigned char *inputmac);

struct global_oper_class
{
	unsigned char opclass;			/* regulatory class */
	unsigned char channel_num;
	unsigned char channel_set[13];	/* max 13 channels, use 0 as terminator */
};

/*Global operating classes*/
 struct global_oper_class oper_class[] = {
	{0, 0, {0}},			/* Invlid entry */
	{81, 13, {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13}},
	{82, 1, {14}},
	{83, 9, {1, 2, 3, 4, 5, 6, 7, 8, 9}},
	{84, 9, {5, 6, 7, 8, 9, 10, 11, 12, 13}},
	{94, 2, {133, 137}},
	{95, 4, {132, 134, 136, 138}},
	{96, 8, {131, 132, 133, 134, 135, 136, 137, 138}},
	{101, 2, {21, 25}},
	{102, 5, {11, 13, 15, 17, 19}},
	{103, 10, {1, 2, 3, 4, 5, 6, 7, 8, 9, 10}},
	{104, 2, {184, 192}},
	{105, 2, {188, 196}},
	{106, 2, {191, 195}},
	{107, 5, {189, 191, 193, 195, 197}},
	{108, 10, {188, 189, 190, 191, 192, 193, 194, 195, 196, 197}},
	{109, 4, {184, 188, 192, 196}},
	{110, 7, {183, 184, 185, 186, 187, 188, 189}},
	{111, 8, {182, 183, 184, 185, 186, 187, 188, 189}},
	{112, 3, {8, 12, 16}},
	{113, 5, {7, 8, 9, 10, 11}},
	{114, 6, {6, 7, 8, 9, 10, 11}},
	{115, 5, {36, 40, 44, 48}},
	{116, 2, {36, 44}},
	{117, 2, {40, 48}},
	{118, 4, {52, 56, 60, 64}},
	{119, 2, {52, 60}},
	{120, 2, {56, 64}},
	{121, 12, {100, 104, 108, 112, 116, 120, 124, 128, 132, 136, 140, 144}},
	{122, 6, {100, 108, 116, 124, 132, 140}},
	{123, 6, {104, 112, 120, 128, 136, 144}},
	{124, 4, {149, 153, 157, 161}},
	{125, 6, {149, 153, 157, 161, 165, 169}},
	{126, 2, {149, 157}},
	{127, 2, {153, 161}},
	{128, 6, {42, 58, 106, 122, 138, 155}},
	{129, 2, {50, 114}},
	{130, 6, {42, 58, 106, 122, 138, 155}},
	{0, 0, {0}}			/* end */
};

unsigned char opclass_2g[] = {
	81, 82, 83, 84, 101, 102, 103, 112, 113, 114, 0
};

unsigned char opclass_5gh[] = {
	94, 95, 96, 104, 105, 106, 107, 108, 109, 110, 111, 121, 122,
	123, 124, 125, 126, 127, 0
};

unsigned char opclass_5gl[] = {
	115, 116, 117, 118, 119, 120, 0
};

int triger_autoconfiguration(struct p1905_managerd_ctx *ctx)
{
	int i = 0;

	if(ctx->current_autoconfig_info.radio_index != -1)
	{
		debug(DEBUG_ERROR, "another radio's autoconfig is ongoing, wait it done\n");
		return -1;
	}

	for(i = 0; i < ctx->radio_number; i++)
	{
		if(ctx->rinfo[i].trrigerd_autoconf == 1)
		{
			ctx->current_autoconfig_info.radio_index = i;
			ctx->enrolle_state = wait_4_send_m1;
			debug(DEBUG_OFF, "radio(%02x:%02x:%02x:%02x:%02x:%02x) will be the alive one\n",
				PRINT_MAC(ctx->rinfo[i].identifier));
			eloop_register_timeout(0, 0, ap_autoconfig_enrolle_step, (void *)ctx, NULL);
			break;
		}
	}
	if(i >= ctx->radio_number)
	{
		debug(DEBUG_OFF, "auto configuration done of all radio\n");
		return 2;
	}
	else
	{
		debug(DEBUG_OFF, "trigger radio(%02x:%02x:%02x:%02x:%02x:%02x) to do autoconfiguration\n",
			PRINT_MAC(ctx->rinfo[i].identifier));
		return 0;
	}
}
void auto_configuration_done(struct p1905_managerd_ctx *ctx)
{
	unsigned char radio_index = 0;

	if (ctx->current_autoconfig_info.radio_index != -1) {
		radio_index = (unsigned char)ctx->current_autoconfig_info.radio_index;
	} else {
		debug(DEBUG_ERROR, "error radio_index == -1!!!\n");
		return;
	}

	/*set to 0 to stop sm*/
	set_radio_autoconf_trriger(ctx, ctx->rinfo[radio_index].identifier, 0);
	ctx->rinfo[radio_index].config_status = MAP_CONF_CONFED;
	ctx->current_autoconfig_info.radio_index = -1;

	triger_autoconfiguration(ctx);
}

int fill_send_tlv(struct p1905_managerd_ctx* ctx, unsigned char* buf, unsigned short len)
{
	if (len >= sizeof(ctx->send_tlv)) {
		debug(DEBUG_ERROR, "send tlv len(%d) > ctx->send_tlv len", len);
		return -1;
	}

	if (ctx->send_tlv_len != 0) {
		debug(DEBUG_ERROR, "receive new send tlv, replace old one(%lu)\n", ctx->send_tlv_cookie);
	}
	ctx->send_tlv_cookie = os_random();
	debug(DEBUG_TRACE, "send_tlv_cookie = %ld, len=%d\n", ctx->send_tlv_cookie, len);
	memcpy(ctx->send_tlv, buf, len);
	ctx->send_tlv_len = len;
	return 0;
}

int reset_send_tlv(struct p1905_managerd_ctx *ctx)
{
	ctx->send_tlv_len = 0;
	return 0;
}

int reset_stored_tlv(struct p1905_managerd_ctx* ctx)
{
	ctx->revd_tlv_len = 0;
	return 0;
}

int store_revd_tlvs(struct p1905_managerd_ctx* ctx, unsigned char *revd_buf, unsigned short len)
{
	if((ctx->revd_tlv_len + len) > 10240)
		return -1;
	else{
			memcpy(ctx->revd_tlv + ctx->revd_tlv_len, revd_buf, len);
			ctx->revd_tlv_len += len;
			return 0;
		}
}
int get_tlv_len_by_tlvtype(unsigned char *ptlvtype)
{
	unsigned short tlvlen;

	/*jump to tlv len field*/
	tlvlen = *(unsigned short *)(ptlvtype + 1);
	/*endian change*/
	tlvlen = be2cpu16(tlvlen);
	/*add byte of tlytyle & tlvlength*/
	tlvlen += 3;

	return tlvlen;
}

int wapp_register_unsolicited_event(struct p1905_managerd_ctx* ctx, unsigned char *buf,
	int len, unsigned short event)
{
	struct cmd_to_wapp *cmd = NULL;
	char recv_buf[8];
	size_t send_pkt_len = 0;
	size_t recv_pkt_len = 0;
	struct timeval tv;
	tv.tv_sec = 5;
	tv.tv_usec = 0;

	cmd = (struct cmd_to_wapp *)buf;
	cmd->type = SET_REGITER_EVENT;
	cmd->role = !ctx->role;

	memcpy(cmd->body, &event, sizeof(event));
	send_pkt_len = sizeof(struct cmd_to_wapp) + sizeof(event);
	recv_pkt_len = sizeof(recv_buf);
	if(0 > wapp_usr_intf_send_command(ctx, (char *)buf, send_pkt_len)) {
		debug(DEBUG_ERROR, "wapp_usr_intf_send_command fail\n");
		return wapp_utils_error;
	}

	if(wapp_usr_intf_ctrl_pending(ctx->wapp_ctrl, &tv) == 1)
	{
		if(wapp_usr_intf_ctrl_recv(ctx->wapp_ctrl, recv_buf, &recv_pkt_len))
		{
			debug(DEBUG_ERROR, "wapp_usr_intf_ctrl_recv %d fail\n", event);
			goto fail;
		}
		if(!memcmp(recv_buf, "OK", 2))
		{
			debug(DEBUG_OFF, "register event(%d) success\n", event);
			goto suc;
		}
		else
		{
			debug(DEBUG_ERROR, "register event(%d) fail\n", event);
			goto fail;
		}
	}
	else
	{
		debug(DEBUG_ERROR, "error when register event %d\n", event);
		goto fail;
	}

fail:
	return wapp_utils_error;
suc:
	return 0;
}
/**
  *this function is to get all the capability of ap; as one radio unique identifier belongs to one
  *specified band, so here use band info on behave of radio unique identifier
  */
void wapp_get_ap_capinfo(struct p1905_managerd_ctx *ctx, unsigned char radio_index)
{
	wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_AP_CAPABILITY, LIB_AP_CAPABILITY,
		NULL, NULL, NULL, 0);
	/*get cap for the specific radio*/
	wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_RADIO_BASIC_CAP, LIB_RADIO_BASIC_CAP,
		ctx->rinfo[radio_index].identifier, NULL, NULL, 0);
	wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_AP_HT_CAPABILITY, LIB_AP_HT_CAPABILITY,
		ctx->rinfo[radio_index].identifier, NULL, NULL, 0);
	wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_AP_VHT_CAPABILITY, LIB_AP_VHT_CAPABILITY,
		ctx->rinfo[radio_index].identifier, NULL, NULL, 0);
	wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_OPERATIONAL_BSS, LIB_OPERBSS_REPORT,
		ctx->rinfo[radio_index].identifier, NULL, NULL, 0);
}

void wapp_get_channel_repinfo(struct p1905_managerd_ctx *ctx)
{
	int radio_index = 0;

	for(radio_index = 0; radio_index < ctx->radio_number; radio_index++) {
		wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_CHANNEL_PREFERENCE,
			LIB_CHANNLE_PREFERENCE, ctx->rinfo[radio_index].identifier,
			NULL, NULL, 0);
		wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_RA_OP_RESTRICTION,
			LIB_RADIO_OPERATION_RESTRICTION, ctx->rinfo[radio_index].identifier,
			NULL, NULL, 0);
	}
}

int notify_wapp_search_controller_done(struct p1905_managerd_ctx* ctx)
{
	unsigned char buf[256];
	struct cmd_to_wapp* cmd = NULL;
	struct bh_link_info *info = NULL;
	int i = 0;

	memset(buf, 0, sizeof(buf));
	cmd = (struct cmd_to_wapp*)buf;

	cmd->type = WAPP_USER_MAP_CONTROLLER_FOUND;
	cmd->role= !ctx->role;
	cmd->band = ctx->cinfo.supported_freq;

	info = (struct bh_link_info *)cmd->body;

	for(i = 0; i < ctx->itf_number; i++)
	{
		if(!memcmp(ctx->cinfo.local_ifname, ctx->itf[i].if_name, IFNAMSIZ))
		{
			if(ctx->itf[i].media_type == IEEE802_3_GROUP)
			{
				info->type = 0;
			}
			else if(ctx->itf[i].media_type & IEEE802_11_GROUP)
			{
				info->type = 1;
			}
			memcpy(info->mac_addr, ctx->itf[i].mac_addr, ETH_ALEN);
			memcpy(info->ifname, ctx->itf[i].if_name, IFNAMSIZ);
			break;
		}
	}
	if (0 > wapp_usr_intf_send_command(ctx, (char *)buf, sizeof(struct cmd_to_wapp) + sizeof(struct bh_link_info))) {
		debug(DEBUG_ERROR, "send cmd(0x%04x) fail\n", cmd->type);
		return wapp_utils_error;
	}
	return wapp_utils_success;
}

void wapp_get_ap_metrics_info(struct p1905_managerd_ctx *ctx)
{
	struct bss_db *bss = NULL;

	SLIST_FOREACH(bss, &ctx->metric_entry.metrics_query_head, bss_entry)
    {
		wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_AP_METRICS_INFO, LIB_AP_METRICS_INFO,
			bss->bssid, NULL, NULL, 0);
	}
}

void wapp_get_assoc_sta_traffic_stats(struct p1905_managerd_ctx *ctx)
{
	struct metric_policy_db *policy = NULL;

	SLIST_FOREACH(policy, &ctx->map_policy.mpolicy.policy_head, policy_entry)
    {
		if (policy->sta_stats_inclusion) {
			wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_ASSOC_STA_TRAFFIC_STATS,
				LIB_ALL_ASSOC_STA_TRAFFIC_STATS, NULL, NULL,  (void *)policy->identifier, ETH_ALEN);
		}
	}
}

void wapp_get_all_assoc_sta_link_metrics(struct p1905_managerd_ctx *ctx)
{
	struct metric_policy_db *policy = NULL;

	SLIST_FOREACH(policy, &ctx->map_policy.mpolicy.policy_head, policy_entry)
    {
		if (policy->sta_metrics_inclusion) {
			wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_ASSOC_STA_LINK_METRICS,
				LIB_ALL_ASSOC_STA_LINK_METRICS, NULL, NULL,  (void *)policy->identifier, ETH_ALEN);
		}
	}
}

int wapp_get_info_by_msgtype(struct p1905_managerd_ctx *ctx,
	unsigned short msgtype, unsigned short waitmsgtype, unsigned char *bssid,
	unsigned char *stamac, void *data, int datalen)
{
	struct cmd_to_wapp *cmd = NULL;
	unsigned char buf[MAX_PKT_LEN] = {0};
	unsigned char *recv_buf = NULL;
	int send_pkt_len = 0, recv_pkt_len = 0;

	debug(DEBUG_TRACE, "get msg(0x%04x) from wapp\n", msgtype);

	cmd = (struct cmd_to_wapp *)buf;
	cmd->type = msgtype;
	cmd->role = !ctx->role;

	if (bssid)
		memcpy(cmd->bssAddr, bssid, 6);
	if (stamac)
		memcpy(cmd->staAddr, stamac, 6);
	if (data)
		memcpy(cmd->body, data, datalen);

	send_pkt_len = sizeof(struct cmd_to_wapp) + datalen;
	if(0 > wapp_usr_intf_send_command(ctx, (char *)buf, send_pkt_len)) {
		debug(DEBUG_ERROR, "send msgtype(0x%04x) fail\n", msgtype);
		return wapp_utils_error;
	}

	recv_buf = buf;
	recv_pkt_len = wapp_wait_parse_spec_event(ctx, (char *)recv_buf, MAX_PKT_LEN, waitmsgtype, 2, 0);
	if (0 >= recv_pkt_len) {
		debug(DEBUG_ERROR, "receive waitmsgtype(0x%04x) fail\n", waitmsgtype);
		return wapp_utils_error;
	}

	return wapp_utils_success;
}


int delete_exist_radio_basic_capability(
	struct p1905_managerd_ctx *ctx, unsigned char* identifier)
{
	 struct radio_basic_capability_db *basic_cap = NULL;
	 struct basic_cap_db *bcap = NULL, *bcap_tmp = NULL;
	 int i = 0;

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.basic_cap_head)))
    {
        SLIST_FOREACH(basic_cap, &(ctx->ap_cap_entry.basic_cap_head), bcap_entry)
        {
            if(!memcmp(basic_cap->identifier, identifier, ETH_ALEN))
            {
                if(!SLIST_EMPTY(&(basic_cap->bcap_head)))
                {
                    bcap = SLIST_FIRST(&(basic_cap->bcap_head));
                    while(bcap)
                    {
                    	debug(DEBUG_TRACE, "delete_exist struct basic_cap_db\n");
						debug(DEBUG_TRACE, "opclass:%d, max_tx_pwr=%d, non_operch_num=%d\n",
							bcap->op_class, bcap->max_tx_pwr, bcap->non_operch_num);
						debug(DEBUG_TRACE, "non_operch_list: \n");
						for (i = 0; i < bcap->non_operch_num; i++)
							debug(DEBUG_TRACE, "%d ", bcap->non_operch_list[i]);
						debug(DEBUG_TRACE, "\n");
                    	//hex_dump("delete_exist struct basic_cap_db", bcap, 16);
                        bcap_tmp = SLIST_NEXT(bcap, basic_cap_entry);
                        SLIST_REMOVE(&(basic_cap->bcap_head), bcap,
                                    basic_cap_db, basic_cap_entry);
                        free(bcap);
                        bcap = bcap_tmp;
                    }
                }
				debug(DEBUG_TRACE, "delete_exist struct radio_basic_capability_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(basic_cap->identifier));
				debug(DEBUG_TRACE, "max_bss_num:%d, band:%d, op_class_num:%d\n",
					basic_cap->max_bss_num, basic_cap->band, basic_cap->op_class_num);
				//hex_dump("delete_exist struct radio_basic_capability_db", basic_cap, 9);
                SLIST_REMOVE(&(ctx->ap_cap_entry.basic_cap_head), basic_cap,\
                            radio_basic_capability_db, bcap_entry);
                free(basic_cap);
                break;
            }
        }
    }

	return wapp_utils_success;
}

int insert_new_radio_basic_capability(
	struct p1905_managerd_ctx *ctx, struct ap_radio_basic_cap *bcap)
{
	struct radio_basic_cap *basic_cap = NULL;
	struct radio_basic_capability_db *bcap_db = NULL;
	struct basic_cap_db *cap = NULL;
	int op_class_num = 0;
	int i = 0, j = 0;
	unsigned char band_cap = 0;

	bcap_db = (struct radio_basic_capability_db *)malloc(sizeof(*bcap_db));
	if (!bcap_db) {
		debug(DEBUG_ERROR, "bcap_db fail to alloc memory!\n");
		return -1;
	}
	memcpy(bcap_db->identifier, bcap->identifier, ETH_ALEN);
	bcap_db->max_bss_num = bcap->max_bss_num;
	bcap_db->band = bcap->band;
	bcap_db->op_class_num = bcap->op_class_num;
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.basic_cap_head), bcap_db, bcap_entry);

	//hex_dump("insert struct radio_basic_capability_db", bcap_db, 9);
	debug(DEBUG_TRACE, "insert struct radio_basic_capability_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(bcap_db->identifier));
	debug(DEBUG_TRACE, "max_bss_num:%d, band:%d, op_class_num:%d\n",
		bcap_db->max_bss_num, bcap_db->band, bcap_db->op_class_num);

	SLIST_INIT(&(bcap_db->bcap_head));
	op_class_num = bcap->op_class_num;
	for(i = 0; i < op_class_num; i++) {
		basic_cap = &bcap->opcap[i];
		cap = (struct basic_cap_db *)malloc(sizeof(*cap));
		if (!cap) {
			debug(DEBUG_ERROR, "cap fail to alloc memory!\n");
			continue;
		}
		memcpy(cap, basic_cap, sizeof(*basic_cap));
		SLIST_INSERT_HEAD(&(bcap_db->bcap_head), cap, basic_cap_entry);

		//hex_dump("insert struct basic_cap_db", cap, 16);
		debug(DEBUG_TRACE, "insert struct basic_cap_db\n");
		debug(DEBUG_TRACE, "opclass:%d, max_tx_pwr=%d, non_operch_num=%d\n",
			cap->op_class, cap->max_tx_pwr, cap->non_operch_num);
		debug(DEBUG_TRACE, "non_operch_list: \n");
		for (j = 0; j < cap->non_operch_num; j++)
			debug(DEBUG_TRACE, "%d ", cap->non_operch_list[j]);
		debug(DEBUG_TRACE, "\n");
		band_cap |= get_bandcap(ctx, basic_cap->op_class, basic_cap->non_operch_num,
					 basic_cap->non_operch_list);
	}
	switch (band_cap) {
	case BAND_2G_CAP:
		ctx->radio_cap |= RADIO_2G_CAP;
	break;
	case BAND_5GL_CAP:
		ctx->radio_cap |= RADIO_5GL_CAP;
	break;
	case BAND_5GH_CAP:
		ctx->radio_cap |= RADIO_5GH_CAP;
	break;
	case BAND_5G_CAP:
		ctx->radio_cap |= RADIO_5G_CAP;
	break;
	default:
	break;
	}
	debug(DEBUG_TRACE, "radio_cap:%02x\n", ctx->radio_cap);

	return wapp_utils_success;

}

int find_ap_radio_basic_capability_byidentifier(
	struct p1905_managerd_ctx *ctx, unsigned char *identifier,
	struct radio_basic_capability_db **bcap)
{
	struct radio_basic_capability_db *bcap_tmp = NULL;

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.basic_cap_head))) {
		SLIST_FOREACH(bcap_tmp, &(ctx->ap_cap_entry.basic_cap_head), bcap_entry)
		{
			//hex_dump("find_ap_radio_basic_capability_byband", bcap_tmp, 9);
			if(!memcmp(bcap_tmp->identifier, identifier, ETH_ALEN)) {
				debug(DEBUG_TRACE, "find struct radio_basic_capability_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(bcap_tmp->identifier));
				debug(DEBUG_TRACE, "max_bss_num:%d, band:%d, op_class_num:%d\n",
					bcap_tmp->max_bss_num, bcap_tmp->band, bcap_tmp->op_class_num);
				if(!SLIST_EMPTY(&(bcap_tmp->bcap_head))) {
					*bcap = bcap_tmp;
					debug(DEBUG_TRACE, "find struct basic_cap_db\n");
					break;
				}
			}
		}
	}
	return wapp_utils_success;
}

int delete_exist_operational_bss(
	struct p1905_managerd_ctx *ctx, unsigned char *identifier)
{
	 struct operational_bss_db *opcap = NULL;
	 struct op_bss_db *opbss = NULL, *opbss_tmp = NULL;

	debug(DEBUG_TRACE, "delete_exist_operational_bss\n");

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.oper_bss_head)))
    {
        SLIST_FOREACH(opcap, &(ctx->ap_cap_entry.oper_bss_head), oper_bss_entry)
        {
            if(!memcmp(opcap->identifier, identifier, ETH_ALEN))
            {
                if(!SLIST_EMPTY(&(opcap->op_bss_head)))
                {
                    opbss = SLIST_FIRST(&(opcap->op_bss_head));
                    while(opbss)
                    {
                    	debug(DEBUG_TRACE, "delete_exist struct op_bss_db\n");
						debug(DEBUG_TRACE, "ssid:%s, bssid=%02x:%02x:%02x:%02x:%02x:%02x\n",
							opbss->ssid, PRINT_MAC(opbss->bssid));
                        opbss_tmp = SLIST_NEXT(opbss, op_bss_entry);
                        SLIST_REMOVE(&(opcap->op_bss_head), opbss,
                                    op_bss_db, op_bss_entry);
                        free(opbss);
                        opbss = opbss_tmp;
                    }
                }
				opcap->oper_bss_num = 0;
				debug(DEBUG_TRACE, "delete_exist struct operational_bss_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(opcap->identifier));
				debug(DEBUG_TRACE, "oper_bss_num:%d, band:%d\n",
					opcap->oper_bss_num, opcap->band);
                break;
            }
        }
    }

	return wapp_utils_success;
}

int insert_new_operational_bss(
	struct p1905_managerd_ctx *ctx, struct oper_bss_cap *opcap)
{
	struct op_bss_cap *opbss_cap = NULL;
	struct operational_bss_db *opcap_db = NULL;
	struct op_bss_db *opbss_db = NULL;
	int oper_bss_num = 0;
	int i = 0;

	SLIST_FOREACH(opcap_db, &ctx->ap_cap_entry.oper_bss_head, oper_bss_entry)
    {
    	if(!memcmp(opcap_db->identifier, opcap->identifier, ETH_ALEN)) {
			break;
    	}
    }

	if (!opcap_db) {
		opcap_db = (struct operational_bss_db *)malloc(sizeof(struct operational_bss_db));
		if (!opcap_db) {
			debug(DEBUG_ERROR, "alloc struct operational_bss_db fail\n");
			return wapp_utils_error;
		}
		memcpy(opcap_db->identifier, opcap->identifier, ETH_ALEN);
		SLIST_INSERT_HEAD(&ctx->ap_cap_entry.oper_bss_head, opcap_db, oper_bss_entry);
	}

	opcap_db->oper_bss_num = opcap->oper_bss_num;

	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(opcap->identifier));
	debug(DEBUG_TRACE, "oper_bss_num:%d\n",opcap->oper_bss_num);

	SLIST_INIT(&(opcap_db->op_bss_head));
	oper_bss_num = opcap_db->oper_bss_num;
	for(i = 0; i < oper_bss_num; i++) {
		opbss_cap = &opcap->cap[i];
		opbss_db = (struct op_bss_db *)malloc(sizeof(*opbss_db));
		if (!opbss_db) {
			debug(DEBUG_ERROR, "alloc struct op_bss_db fail\n");
			return wapp_utils_error;
		}
		memcpy(opbss_db->bssid, opbss_cap->bssid, ETH_ALEN);
		opbss_db->ssid_len = opbss_cap->ssid_len;
		memcpy(opbss_db->ssid, opbss_cap->ssid, opbss_db->ssid_len);
		SLIST_INSERT_HEAD(&opcap_db->op_bss_head, opbss_db, op_bss_entry);

		debug(DEBUG_TRACE, "insert struct op_bss_db\n");
		debug(DEBUG_TRACE, "ssid(len=%d):%s, bssid=%02x:%02x:%02x:%02x:%02x:%02x\n",
				opbss_db->ssid_len, opbss_db->ssid, PRINT_MAC(opbss_db->bssid));
	}

	return wapp_utils_success;

}

void delete_legacy_assoc_clients_db(struct p1905_managerd_ctx *ctx)
{
	struct associated_clients_db *assoc_cli_db = NULL, *assoc_cli_db_tmp = NULL;
	struct operational_bss_db *opcap_db = NULL;
	struct op_bss_db *bss_info = NULL;
	struct clients_db *cli_db = NULL, *cli_db_tmp = NULL;
	char bss_find = 0;

	assoc_cli_db = SLIST_FIRST(&ctx->ap_cap_entry.assoc_clients_head);
	while (assoc_cli_db) {
		bss_find = 0;
		assoc_cli_db_tmp = SLIST_NEXT(assoc_cli_db, assoc_clients_entry);
		SLIST_FOREACH(opcap_db, &ctx->ap_cap_entry.oper_bss_head, oper_bss_entry) {
			SLIST_FOREACH(bss_info, &opcap_db->op_bss_head, op_bss_entry) {
				if (!os_memcmp(bss_info->bssid, assoc_cli_db->bssid, ETH_ALEN)) {
					bss_find = 1;
					break;
				}
			}
			if (bss_find)
				break;
		}
		if (!bss_find) {
			SLIST_REMOVE(&ctx->ap_cap_entry.assoc_clients_head, assoc_cli_db,
			associated_clients_db, assoc_clients_entry);
			debug(DEBUG_OFF, "free assoc_cli_db(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(assoc_cli_db->bssid));
			cli_db = SLIST_FIRST(&assoc_cli_db->clients_head);
			while (cli_db != NULL) {
				cli_db_tmp = SLIST_NEXT(cli_db, clients_entry);
				debug(DEBUG_TRACE, "free cli_db mac[%02x:%02x:%02x:%02x:%02x:%02x]\n",
					PRINT_MAC(cli_db->mac));
				os_free(cli_db);
				cli_db = cli_db_tmp;
			}
			os_free(assoc_cli_db);
		}
		assoc_cli_db = assoc_cli_db_tmp;
	}
}

int update_assoc_sta_info(struct p1905_managerd_ctx *ctx,
	struct map_client_association_event *cinfo)
{
	struct associated_clients_db *bss_db = NULL;
	struct associated_clients_db *bss_db_tmp = NULL;
	struct clients_db *sta_db = NULL;

	SLIST_FOREACH(bss_db, &(ctx->ap_cap_entry.assoc_clients_head), assoc_clients_entry)
	{
		if(!memcmp(bss_db->bssid, cinfo->bssid, ETH_ALEN))
			break;
	}
	if(bss_db == NULL)
	{
		debug(DEBUG_TRACE, "bss(%02x:%02x:%02x:%02x:%02x:%02x) not exist, create new one\n",
			PRINT_MAC(cinfo->bssid));
		bss_db = (struct associated_clients_db*)malloc(sizeof(struct associated_clients_db));
		if(bss_db == NULL)
		{
			debug(DEBUG_ERROR, "alloc memory fail\n");
			return -1;
		}
		memset(bss_db, 0, sizeof(struct associated_clients_db));
		memcpy(bss_db->bssid, cinfo->bssid, ETH_ALEN);
		SLIST_INIT(&(bss_db->clients_head));
		SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.assoc_clients_head), bss_db, assoc_clients_entry);
	}
	bss_db_tmp = bss_db;
	bss_db = NULL;

	SLIST_FOREACH(bss_db, &(ctx->ap_cap_entry.assoc_clients_head), assoc_clients_entry)
	{
		if (os_memcmp(bss_db->bssid, cinfo->bssid, ETH_ALEN))
			continue;
		SLIST_FOREACH(sta_db, &(bss_db->clients_head), clients_entry)
		{
			if(!memcmp(sta_db->mac, cinfo->sta_mac, ETH_ALEN))
			{
				/*free all stainfo for assoc req frame len may be different*/
				SLIST_REMOVE(&(bss_db->clients_head), sta_db, clients_db, clients_entry);
				free(sta_db);
				bss_db->assco_clients_num--;
				debug(DEBUG_TRACE, "delete sta(%02x:%02x:%02x:%02x:%02x:%02x) in bss(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(cinfo->sta_mac), PRINT_MAC(bss_db->bssid));
				break;
			}
		}
	}

	if(cinfo->assoc_evt == 0x80) /*station connect to bss*/
	{
		sta_db = (struct clients_db *)malloc(sizeof(struct clients_db) + cinfo->assoc_req_len);
		if (!sta_db) {
			debug(DEBUG_ERROR, "fail to allock memory for sta_db!\n");
			return -1;
		}
		memcpy(sta_db->mac, cinfo->sta_mac, ETH_ALEN);
		sta_db->time = cinfo->assoc_time;
		sta_db->frame_len = cinfo->assoc_req_len;
		memcpy(sta_db->frame, cinfo->assoc_req, cinfo->assoc_req_len);

		debug(DEBUG_TRACE, "insert sta(%02x:%02x:%02x:%02x:%02x:%02x) in bss(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(cinfo->sta_mac), PRINT_MAC(cinfo->bssid));

		SLIST_INSERT_HEAD(&(bss_db_tmp->clients_head), sta_db, clients_entry);
		bss_db_tmp->assco_clients_num++;
	}

	SLIST_FOREACH(bss_db, &ctx->ap_cap_entry.assoc_clients_head, assoc_clients_entry) {
		if (!os_memcmp(bss_db->bssid, cinfo->bssid, ETH_ALEN) && bss_db->assco_clients_num == 0) {
			SLIST_REMOVE(&ctx->ap_cap_entry.assoc_clients_head, bss_db, associated_clients_db, assoc_clients_entry);
			debug(DEBUG_OFF, "free bss_db(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(bss_db->bssid));
			os_free(bss_db);
			bss_db = NULL;
			break;
		}
	}

	return 0;
}

/**
  * @service: 1 for controller only; 2 for agent only; 3 for both controller and agent
  */
unsigned short append_supported_service_tlv(
        unsigned char *pkt, unsigned char service)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf = SUPPORTED_SERVICE_TLV_TYPE;
    temp_buf +=1;
	/**
	  * need communicate with WAPP to get supported service
	  * or just write hard code to Multi-AP Agent
	  */
	if (service == 0) {
		*(unsigned short *)(temp_buf) = cpu2be16(SUPPORTED_SERVICE_LENGTH);
    	temp_buf += 2;
		*temp_buf++ = 1;
    	*temp_buf = SERVICE_CONTROLLER;
    	total_length = SUPPORTED_SERVICE_LENGTH + 3;
	} else if (service == 1) {
		*(unsigned short *)(temp_buf) = cpu2be16(SUPPORTED_SERVICE_LENGTH);
    	temp_buf += 2;
		*temp_buf++ = 1;
    	*temp_buf = SERVICE_AGENT;
    	total_length = SUPPORTED_SERVICE_LENGTH + 3;
	} else if (service == 2) {
		*(unsigned short *)(temp_buf) = cpu2be16(SUPPORTED_SERVICE2_LENGTH);
    	temp_buf += 2;
		*temp_buf++ = 2;
    	*temp_buf++ = SERVICE_CONTROLLER;
		*temp_buf = SERVICE_AGENT;
    	total_length = SUPPORTED_SERVICE2_LENGTH + 3;
	} else {
		debug(DEBUG_ERROR, "unvalid service\n");
	}

    return total_length;
}

unsigned short append_searched_service_tlv(
        unsigned char *pkt)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf = SEARCHED_SERVICE_TLV_TYPE;
    temp_buf +=1;

	*(unsigned short *)(temp_buf) = cpu2be16(SEARCHED_SERVICE_LENGTH);
    temp_buf += 2;
	*temp_buf++ = 1;
    *temp_buf = SERVICE_CONTROLLER;

    total_length = SEARCHED_SERVICE_LENGTH + 3;

    return total_length;
}
/**
  * @identifier: 6 byte of Radio Unique Identifier of a radio of the Agent
  */
unsigned short append_ap_radio_identifier_tlv(
        unsigned char *pkt, unsigned char *identifier)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf = AP_RADIO_IDENTIFIER_TYPE;
    temp_buf +=1;

	*(unsigned short *)(temp_buf) = cpu2be16(AP_RADIO_IDENTIFIER_LENGTH);
    temp_buf += 2;
	memcpy(temp_buf,identifier,ETH_ALEN);

    total_length = AP_RADIO_IDENTIFIER_LENGTH + 3;

    return total_length;
}

unsigned short append_ap_radio_basic_capability_tlv(struct p1905_managerd_ctx* ctx, unsigned char *pkt,
	struct radio_basic_capability_db *bcap)
{
	unsigned short total_length = 0, tmp_len = 0;
    unsigned char *temp_buf;
	struct basic_cap_db *cap = NULL;
	unsigned char radio_index = 0;

	temp_buf = pkt;
	*temp_buf = AP_RADIO_BASIC_CAPABILITY_TYPE;
    temp_buf += 1;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, bcap->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	if (ctx->current_autoconfig_info.radio_index != -1) {
		radio_index = (unsigned char)ctx->current_autoconfig_info.radio_index;
	}
	temp_buf[0] = ctx->current_autoconfig_info.radio_index != -1 ?
		ctx->rinfo[radio_index].bss_number : bcap->max_bss_num;
	temp_buf += 1;
	total_length += 1;

	temp_buf[0] = bcap->op_class_num;
	temp_buf += 1;
	total_length += 1;

	SLIST_FOREACH(cap, &(bcap->bcap_head), basic_cap_entry)
	{
		tmp_len = 3 + cap->non_operch_num;
		memcpy(temp_buf, cap, tmp_len);
		temp_buf += tmp_len;
		total_length += tmp_len;
	}

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}


/**
  * @cap1(0 or 1): cap of unassociated sta link metrics reporting on the channels its BSSs are
  * currently operating on
  * @cap2(0 or 1): cap of unassociated sta link metrics reporting on the channels its BSSs are
  * not currently operating on
  * @cap3(0 or 1): cap of Agent-initiated RSSI-based Steering
  */
unsigned short append_ap_capability_tlv(
        unsigned char *pkt, struct ap_capability *ap_cap)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf = AP_CAPABILITY_TYPE;
    temp_buf +=1;

	*(unsigned short *)(temp_buf) = cpu2be16(AP_CAPABILITY_LENGTH);
    temp_buf += 2;

	/*if big endian is correct*/
	*temp_buf = (ap_cap->sta_report_on_cop << 7) | (ap_cap->sta_report_not_cop << 6) |
		(ap_cap->rssi_steer << 5);

    total_length = AP_CAPABILITY_LENGTH + 3;

    return total_length;
}

unsigned short append_ap_ht_capability_tlv(
        unsigned char *pkt, struct ap_ht_cap_db* ht_cap)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf++ = AP_HT_CAPABILITY_TYPE;

	*(unsigned short *)(temp_buf) = cpu2be16(AP_HT_CAPABILITY_LENGTH);
    temp_buf += 2;
	memcpy(temp_buf, ht_cap->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;

	*temp_buf = (ht_cap->tx_stream << 6) | (ht_cap->rx_stream << 4) |
		(ht_cap->sgi_20 << 3) | (ht_cap->sgi_40 << 2) | (ht_cap->ht_40 << 1);

    total_length = AP_HT_CAPABILITY_LENGTH + 3;

    return total_length;
}

unsigned short append_ap_vht_capability_tlv(
        unsigned char *pkt, struct ap_vht_cap_db *vht_cap)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf = NULL;

    temp_buf = pkt;

    *temp_buf = AP_VHT_CAPABILITY_TYPE;
    temp_buf += 1;

	*(unsigned short *)(temp_buf) = cpu2be16(AP_VHT_CAPABILITY_LENGTH);
    temp_buf += 2;
	memcpy(temp_buf, vht_cap->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	*(unsigned short *)(temp_buf) = cpu2be16(vht_cap->vht_tx_mcs);
	temp_buf += 2;
	*(unsigned short *)(temp_buf) = cpu2be16(vht_cap->vht_rx_mcs);
	temp_buf += 2;
	*temp_buf = (vht_cap->tx_stream << 5) | (vht_cap->rx_stream << 2) |
		(vht_cap->sgi_80 << 1) | (vht_cap->sgi_160);
	temp_buf++;
	*temp_buf = (vht_cap->vht_8080 << 7) | (vht_cap->vht_160 << 6) |
		(vht_cap->su_beamformer << 5) | (vht_cap->mu_beamformer << 4);

    total_length = AP_VHT_CAPABILITY_LENGTH + 3;

    return total_length;
}

unsigned short append_ap_he_capability_tlv(
        unsigned char *pkt, struct ap_he_cap_db *he_cap)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf = NULL;

    temp_buf = pkt;

    *temp_buf++ = AP_HE_CAPABILITY_TYPE;

	*(unsigned short *)(temp_buf) = cpu2be16(9 + he_cap->he_mcs_len);
    temp_buf += 2;
	memcpy(temp_buf, he_cap->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;

	*temp_buf++ = he_cap->he_mcs_len;
	os_memcpy(temp_buf, he_cap->he_mcs, he_cap->he_mcs_len);
	temp_buf += he_cap->he_mcs_len;
	*temp_buf++ = (he_cap->tx_stream << 5) | (he_cap->rx_stream << 2) |
		(he_cap->he_8080 << 1) | (he_cap->he_160);
	*temp_buf++ = (he_cap->su_bf_cap << 7) | (he_cap->mu_bf_cap << 6) |
		(he_cap->ul_mu_mimo_cap << 5) | (he_cap->ul_mu_mimo_ofdma_cap << 4) |
		(he_cap->dl_mu_mimo_ofdma_cap << 3) | (he_cap->ul_ofdma_cap << 2) |
		(he_cap->dl_ofdma_cap << 1);

    total_length = 12 + he_cap->he_mcs_len;

    return total_length;
}

unsigned short append_client_steering_request_tlv(unsigned char *pkt)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;
	unsigned char BSSID[6] = {0x00,0x0C,0x43,0x26,0x60,0x98};
	unsigned char TBSSID[6] = {0x00,0x0C,0x43,0x26,0x60,0x60};
	unsigned char STA[6] = {0x38,0xBC,0x1A,0xC1,0xD3,0x40};

    temp_buf = pkt;

    *temp_buf++ = STEERING_REQUEST_TYPE;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, BSSID, 6);
	temp_buf += 6;
	total_length += 6;

	*temp_buf++ = 0xe0;
	total_length += 1;

	*temp_buf++ = 0x00;
	*temp_buf++ = 0x05;
	total_length += 2;

	/*sta num & sta mac*/
	*temp_buf++ = 1;
	total_length += 1;
	memcpy(temp_buf, STA, 6);
	temp_buf += 6;
	total_length += 6;

	/*bssid num & bssid mac, opclass, channel*/
	*temp_buf++ = 1;
	total_length += 1;
	memcpy(temp_buf, TBSSID, 6);
	temp_buf += 6;
	total_length += 6;
	*temp_buf++ = 115;
	*temp_buf++ = 36;
	total_length += 2;

	/*calculate totoal length & fill into the length field*/
    *(pkt + 1) =  (unsigned char)(((total_length - 3) >> 8) & 0xff);
    *(pkt + 2) =  (unsigned char)((total_length - 3) & 0xff);

    return total_length;
}

unsigned short append_client_steering_btm_report_tlv(unsigned char *pkt,
	struct cli_steer_btm_event *btm_evt)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;
	unsigned char zero_bssid[ETH_ALEN] = {0};

    temp_buf = pkt;

    *temp_buf++ = STEERING_BTM_REPORT_TYPE;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, btm_evt->bssid, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	memcpy(temp_buf, btm_evt->sta_mac, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = btm_evt->status;
	total_length += 1;

	if (memcmp(zero_bssid, btm_evt->tbssid, ETH_ALEN)) {
		memcpy(temp_buf, btm_evt->tbssid, ETH_ALEN);
		temp_buf += ETH_ALEN;
		total_length += ETH_ALEN;
	}

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

unsigned short append_map_policy_config_request_tlv(unsigned char *pkt)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;
	unsigned char BSSID[ETH_ALEN] = {0x00,0x0C,0x43,0x26,0x60,0x98};

	temp_buf = pkt;

	*temp_buf++ = STEERING_POLICY_TYPE;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	*temp_buf++ = 0;
	total_length += 1;

	*temp_buf++ = 0;
	total_length += 1;

	*temp_buf++ = 1;
	total_length += 1;
	memcpy(temp_buf, BSSID, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;
	*temp_buf++ = 0x01;
	*temp_buf++ = 0xff;
	*temp_buf++ = 20;
	total_length += 3;

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}

unsigned short append_cli_assoc_cntrl_request_tlv(unsigned char *pkt)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;
	unsigned char BSSID[ETH_ALEN] = {0x00,0x0C,0x43,0x26,0x60,0x98};
	unsigned char STA[ETH_ALEN] = {0x38,0xBC,0x1A,0xC1,0xD3,0x40};

	temp_buf = pkt;

	*temp_buf++ = CLI_ASSOC_CONTROL_REQUEST_TYPE;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, BSSID, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = 0;
	total_length += 1;

	*temp_buf++ = 0;
	*temp_buf++ = 30;
	total_length += 2;

	*temp_buf++ = 1;
	total_length += 1;

	memcpy(temp_buf, STA, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;


	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}

unsigned short append_channel_preference_tlv(unsigned char *pkt,
	struct ch_prefer_db *prefer_db)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;
	struct prefer_info_db *prefer_info = NULL;

	if (!prefer_db) {
		debug(DEBUG_ERROR, "NULL pointer prefer_db!\n");
		return 0;
	}

	temp_buf = pkt;

  /*firstly shift to tlv value*/
	temp_buf += 3;
	total_length += 3;

	memcpy(temp_buf, prefer_db->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = prefer_db->op_class_num;
	total_length += 1;
	if (!SLIST_EMPTY(&prefer_db->prefer_info_head)) {
		SLIST_FOREACH(prefer_info, &prefer_db->prefer_info_head, prefer_info_entry)
		{
			/*operating class*/
			*temp_buf++ = prefer_info->op_class;
			total_length += 1;
			/*channel num*/
			*temp_buf++ = prefer_info->ch_num;
			total_length += 1;
			/*channel list*/
			if (prefer_info->ch_num) {
				memcpy(temp_buf, prefer_info->ch_list, prefer_info->ch_num);
				temp_buf += prefer_info->ch_num;
				total_length += prefer_info->ch_num;
			}
			/*preference & reason*/
			*temp_buf++ = (prefer_info->perference << 4) | prefer_info->reason;
			total_length += 1;
		}
	} else {
		debug(DEBUG_ERROR, "error! prefer_info_head==NULL\n");
		return 0;
	}

	/*if really has the p1905.1 neighbor device, append type & length */
	*pkt = CH_PREFERENCE_TYPE;
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}

unsigned short append_transmit_power_limit_tlv(unsigned char *pkt,
	unsigned char *identifier, signed char tx_power_limit)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;

	temp_buf = pkt;

	*temp_buf++ = TRANSMIT_POWER_LIMIT_TYPE;
	total_length += 1;

	*(unsigned short *)(temp_buf) = cpu2be16(TRANSMIT_POWER_LIMIT_LENGTH);
	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = tx_power_limit;
	total_length += 1;

	return total_length;
}

unsigned short append_radio_operation_restriction_tlv(unsigned char *pkt,
	struct oper_restrict_db *restriction_db)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;
	struct restrict_db *restrict_info = NULL;
	int i = 0;

	if (!restriction_db) {
		debug(DEBUG_ERROR, "NULL pointer restriction_db!\n");
		return 0;
	}

	temp_buf = pkt;

  /*firstly shift to tlv value*/
	temp_buf += 3;
	total_length += 3;

	memcpy(temp_buf, restriction_db->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = restriction_db->op_class_num;
	total_length += 1;

	if (!SLIST_EMPTY(&restriction_db->restrict_head)) {
		SLIST_FOREACH(restrict_info, &restriction_db->restrict_head, restrict_entry)
		{
			*temp_buf++ = restrict_info->op_class;
			total_length += 1;
			*temp_buf++ = restrict_info->ch_num;
			total_length += 1;
			for (i = 0; i < restrict_info->ch_num; i++) {
				*temp_buf++ = restrict_info->ch_list[i];
				*temp_buf++ = restrict_info->min_fre_sep[i];
				total_length += 2;
			}
		}
	} else {
		debug(DEBUG_ERROR, "error! restrict_head==NULL\n");
		return 0;
	}

	/*if really has the p1905.1 neighbor device, append type & length */
	*pkt = RADIO_OPERATION_RESTRICTION_TYPE;
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}


unsigned short append_channel_selection_response_tlv(unsigned char *pkt,
	struct ch_stat_info *stat)
{
	unsigned short total_length = 0;
    unsigned char *temp_buf;

	temp_buf = pkt;

	*temp_buf = CH_SELECTION_RESPONSE_TYPE;
    temp_buf += 1;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, stat->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf = stat->code;
	temp_buf += 1;
	total_length += 1;

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}

unsigned short append_operating_channel_report_tlv(unsigned char *pkt,
	struct ch_rep_info *rep_info)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;

	temp_buf = pkt;

	*temp_buf = OPERATING_CHANNEL_REPORT_TYPE;
	temp_buf += 1;
	total_length += 1;

	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, rep_info->identifier, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf = 1;
	temp_buf += 1;
	total_length += 1;

	*temp_buf = rep_info->op_class;
	temp_buf += 1;
	total_length += 1;

	*temp_buf = rep_info->channel;
	temp_buf += 1;
	total_length += 1;

	*temp_buf = rep_info->tx_power;
	temp_buf += 1;
	total_length += 1;

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}


unsigned short channel_selection_response_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct ch_stat_info *stat = NULL;
	int i = 0;

	msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
		for (i = 0; i < ctx->ch_stat->ch_rsp_num; i++) {
			stat = ctx->ch_stat->info + i;
			length = append_channel_selection_response_tlv(tlv_temp_buf, stat);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}
		free(ctx->ch_stat);
		ctx->ch_stat = NULL;
	}
	else
	{
		length = append_send_tlv(tlv_temp_buf, ctx);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CHANNLE_SELECTION_RESPONSE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short channel_selection_request_message(unsigned char *buf,
	struct agent_list_db *agent, struct p1905_managerd_ctx *ctx)

{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct ch_prefer_db *chcap = NULL;

	msg_hdr = (cmdu_message_header*)buf;
	if(!ctx->send_tlv_len)
	{
		SLIST_FOREACH(chcap, &agent->ch_prefer_head, ch_prefer_entry)
		{
			length = append_channel_preference_tlv(tlv_temp_buf, chcap);
			total_tlvs_length += length;
			tlv_temp_buf += length;
			if (chcap->tx_power_limit) {
				length = append_transmit_power_limit_tlv(tlv_temp_buf,
					chcap->identifier, chcap->tx_power_limit);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
		}
	}
	else
	{
		length = append_send_tlv(tlv_temp_buf, ctx);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CHANNLE_SELECTION_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short combined_infrastructure_metrics_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx, unsigned char *dalmac)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct agent_list_db *agent = NULL, *nagent = NULL;
	struct tx_link_metric_db *tx_link_metric = NULL;
	struct rx_link_metric_db *rx_link_metric = NULL;
	struct mrsp_db *mrsp = NULL;
	unsigned char neighbor_almac[ETH_ALEN] = {0}, zero_almac[ETH_ALEN] = {0};

	msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{

		find_agent_info(ctx, dalmac, &agent);
		if (!agent) {
			debug(DEBUG_ERROR, "error! no agent info exist\n");
			goto end;
		}
		SLIST_FOREACH(tx_link_metric, &agent->tx_metrics_head, tx_link_metric_entry)
		{
			if (!memcmp(tx_link_metric->almac, agent->almac, ETH_ALEN)) {
				memcpy(neighbor_almac, tx_link_metric->nalmac, ETH_ALEN);
				break;
			}
		}
		if (!tx_link_metric) {
			SLIST_FOREACH(rx_link_metric, &agent->rx_metrics_head, rx_link_metric_entry)
			{
				if (!memcmp(rx_link_metric->almac, agent->almac, ETH_ALEN)) {
					memcpy(neighbor_almac, rx_link_metric->nalmac, ETH_ALEN);
					break;
				}
			}
		}

		if (memcmp(neighbor_almac, zero_almac, ETH_ALEN)) {
			find_agent_info(ctx, neighbor_almac, &nagent);
			if (!nagent) {
				debug(DEBUG_ERROR, "error! no neighbor agent info exist\n");
				goto end;
			}
			SLIST_FOREACH(mrsp, &agent->metrics_rsp_head, mrsp_entry)
			{
				length = append_ap_metrics_tlv(tlv_temp_buf, mrsp);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
			SLIST_FOREACH(mrsp, &nagent->metrics_rsp_head, mrsp_entry)
			{
				length = append_ap_metrics_tlv(tlv_temp_buf, mrsp);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
			SLIST_FOREACH(tx_link_metric, &agent->tx_metrics_head, tx_link_metric_entry)
			{
				length = append_tx_link_metric_tlv(tlv_temp_buf, tx_link_metric);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
			SLIST_FOREACH(tx_link_metric, &nagent->tx_metrics_head, tx_link_metric_entry)
			{
				length = append_tx_link_metric_tlv(tlv_temp_buf, tx_link_metric);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
			SLIST_FOREACH(rx_link_metric, &agent->rx_metrics_head, rx_link_metric_entry)
			{
				length = append_rx_link_metric_tlv(tlv_temp_buf, rx_link_metric);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}

			SLIST_FOREACH(rx_link_metric, &nagent->rx_metrics_head, rx_link_metric_entry)
			{
				length = append_rx_link_metric_tlv(tlv_temp_buf, rx_link_metric);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}

		}
	}
	else
	{
		length = append_send_tlv(tlv_temp_buf, ctx);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(COMBINED_INFRASTRUCTURE_METRICS);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

end:
	return total_tlvs_length;

}

unsigned short client_steering_request_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CLIENT_STEERING_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short client_steering_btm_report_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_client_steering_btm_report_tlv(tlv_temp_buf, &ctx->sinfo.sbtm_evt);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CLIENT_STEERING_BTM_REPORT);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short client_steering_completed_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CLIENT_STEERING_COMPLETED);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short map_policy_config_request_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(MAP_POLICY_CONFIG_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short client_association_control_request_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CLIENT_ASSOC_CONTROL_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short channel_preference_report_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
	struct ch_prefer_db *prefer_db = NULL;
	struct oper_restrict_db *restriction_db = NULL;

    msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
		if(!SLIST_EMPTY(&ctx->ap_cap_entry.ch_prefer_head)) {
			SLIST_FOREACH(prefer_db, &ctx->ap_cap_entry.ch_prefer_head, ch_prefer_entry)
			{
				length = append_channel_preference_tlv(tlv_temp_buf, prefer_db);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
		}
		if(!SLIST_EMPTY(&ctx->ap_cap_entry.oper_restrict_head)) {
			SLIST_FOREACH(restriction_db, &ctx->ap_cap_entry.oper_restrict_head, oper_restrict_entry)
			{
				length = append_radio_operation_restriction_tlv(tlv_temp_buf, restriction_db);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
		}
	}
	else
	{
		length = append_send_tlv(tlv_temp_buf, ctx);
   		total_tlvs_length += length;
    	tlv_temp_buf += length;
	}
    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(CHANNLE_PREFERENCE_REPORT);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned short operating_channel_report_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	int i = 0;

	msg_hdr = (cmdu_message_header*)buf;

	for (i = 0; i < ctx->ch_rep->ch_rep_num; i++) {
		if (ctx->ch_rep->info[i].channel) {
			length = append_operating_channel_report_tlv(tlv_temp_buf, &ctx->ch_rep->info[i]);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}
	}
	free(ctx->ch_rep);
	ctx->ch_rep = NULL;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(OPERATING_CHANNEL_REPORT);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short channel_preference_query_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{

	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CHANNLE_PREFERENCE_QUERY);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short ap_capability_query_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(AP_CAPABILITY_QUERY);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned short ap_capability_report_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;
	struct ap_ht_cap_db* pdb_ht = NULL;
	struct ap_vht_cap_db* pdb_vht = NULL;
	struct ap_he_cap_db* pdb_he = NULL;
	struct radio_basic_capability_db *pdb_racap = NULL;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_ap_capability_tlv(tlv_temp_buf, &ctx->ap_cap_entry.ap_cap);
    total_tlvs_length += length;
    tlv_temp_buf += length;


	/*One or more AP HT Capabilities TLV from spec 17.1.7*/
	SLIST_FOREACH(pdb_racap, &(ctx->ap_cap_entry.basic_cap_head), bcap_entry)
	{
		debug(DEBUG_TRACE, "pdb_racap->identifier=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(pdb_racap->identifier));
		length = append_ap_radio_basic_capability_tlv(ctx, tlv_temp_buf, pdb_racap);
		total_tlvs_length += length;
		tlv_temp_buf += length;

	}
	/*Zero or more AP HT Capabilities TLV from spec 17.1.7*/
	SLIST_FOREACH(pdb_ht, &(ctx->ap_cap_entry.ap_ht_cap_head), ap_ht_cap_entry)
	{
		debug(DEBUG_TRACE, "pdb_ht->identifier=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(pdb_ht->identifier));
		length = append_ap_ht_capability_tlv(tlv_temp_buf, pdb_ht);
		total_tlvs_length += length;
		tlv_temp_buf += length;

	}
	/*Zero or more AP VHT Capabilities TLV from spec 17.1.7*/
	SLIST_FOREACH(pdb_vht, &(ctx->ap_cap_entry.ap_vht_cap_head), ap_vht_cap_entry)
	{
		debug(DEBUG_TRACE, "pdb_vht->identifier=%02x:%02x:%02x:%02x:%02x:%02x\n", PRINT_MAC(pdb_vht->identifier));
		length = append_ap_vht_capability_tlv(tlv_temp_buf, pdb_vht);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}

	SLIST_FOREACH(pdb_he, &(ctx->ap_cap_entry.ap_he_cap_head), ap_he_cap_entry)
	{
		debug(DEBUG_TRACE, "pdb_vht->identifier=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(pdb_he->identifier));
		length = append_ap_he_capability_tlv(tlv_temp_buf, pdb_he);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}

	#ifdef MAP_R2
	/*one channel scan capability tlv*/
	length = append_channel_scan_capability_tlv(tlv_temp_buf, &(ctx->ap_cap_entry.ch_scan_cap_head));
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/* One CAC Capabilities TLV */
	length = append_cac_capability_tlv(tlv_temp_buf, &(ctx->ap_cap_entry.radio_cac_cap));
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/* One R2 AP Capability TLV  */
	length = append_r2_cap_tlv(tlv_temp_buf, &(ctx->ap_cap_entry.ap_r2_cap));
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/* One Metric Collection Interval TLV */
	length = append_metric_collection_interval_tlv(tlv_temp_buf, ctx->metric_entry.metric_collection_interval);
	total_tlvs_length += length;
	tlv_temp_buf += length;
#endif // #ifdef MAP_R2

    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(AP_CAPABILITY_REPORT);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned short cli_capability_report_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    cmdu_message_header *msg_hdr;
    unsigned short length = 0;
    unsigned short total_tlvs_length =0;

    msg_hdr = (cmdu_message_header*)buf;

    length = append_cli_info_tlv(tlv_temp_buf, &ctx->sinfo.cinfo);
    total_tlvs_length += length;
    tlv_temp_buf += length;
	length = append_cli_capability_report_tlv(ctx, tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;
    length = append_end_of_tlv(tlv_temp_buf);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    /*tlvs size is less than (46(minimun ethernet frame payload size)
     *-8(cmdu header size)) ==>padding
     */
    if(total_tlvs_length < MIN_TLVS_LENGTH)
    {
	    memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
	    total_tlvs_length = MIN_TLVS_LENGTH;
    }

    //0x00: for this version of the specification
    //0x01~0xFF: Reserved Values
    msg_hdr->message_version = 0x0;

    //set reserve field to 0
    msg_hdr->reserved_field_0 = 0x0;
    msg_hdr->message_type = cpu2be16(CLIENT_CAPABILITY_REPORT);
    msg_hdr->relay_indicator = 0x0;
    //set reserve field to 0
    msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

unsigned short cli_capability_query_message(
        unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct client_info *cinfo = NULL;

	msg_hdr = (cmdu_message_header*)buf;
	if(ctx->send_tlv_len == 0)
	{
		return total_tlvs_length;
	}
	else
	{
		cinfo = (struct client_info*)ctx->send_tlv;
		reset_send_tlv(ctx);
	}
	length = append_cli_info_tlv(tlv_temp_buf, cinfo);
    total_tlvs_length += length;
    tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CLIENT_CAPABILITY_QUERY);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

int _1905_ack_message(unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	/*no content*/

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(e_1905_ack);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}


int high_layer_data_message(unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(e_higher_layer_data);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}

int dev_send_1905_msg(unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	memcpy(tlv_temp_buf, ctx->pcmdu_tx_buf, ctx->cmdu_tx_buf_len);
	length = ctx->cmdu_tx_buf_len;
	total_tlvs_length += length;
    tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(ctx->cmdu_tx_msg_type);
	if (ctx->cmdu_tx_msg_type == AP_AUTOCONFIG_RENEW || ctx->cmdu_tx_msg_type == TOPOLOGY_NOTIFICATION) {
		msg_hdr->relay_indicator = 0x1;
		debug(DEBUG_TRACE, "renew msg or notification msg,relay_indicator will be turn on\n");
	} else {
		msg_hdr->relay_indicator = 0x0;
	}
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

    return total_tlvs_length;
}


int parse_supported_service_tlv(
	unsigned char *buf, unsigned char *service) {

	unsigned char *temp_buf;
    unsigned short length = 0;
    temp_buf = buf;

    if((*temp_buf) == SUPPORTED_SERVICE_TLV_TYPE)
        temp_buf++;
    else
    {
        return -1;
    }

    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf+=2;

    if (length == 2) {
		temp_buf++;
		*service = *temp_buf;
		debug(DEBUG_TRACE, "support service %s\n", (*service) ? "agent":"controller");
    } else if (length == 3) {
   		temp_buf++;
		*service = 2;
		debug(DEBUG_TRACE, "support service agent and controller\n");
    } else {
        return -1;
    }

    return (length+3);

}

int get_bandcap(struct p1905_managerd_ctx *ctx, unsigned char opclass,
	unsigned char non_opch_num, unsigned char *non_opch)
{
	unsigned char chnum = 0;
	int i = 0, j = 0, k = 0;
	struct global_oper_class *opcls = oper_class;
	unsigned char *chset = NULL;
	unsigned char cap = BAND_INVALID_CAP;

	do {
		if (opcls[i].opclass == opclass) {
			chnum = opcls[i].channel_num;
			chset = opcls[i].channel_set;
			if (non_opch_num == chnum) {
				debug(DEBUG_TRACE, "opclass(%d) all channel not used\n", opclass);
				hex_dump("non_opch", non_opch, non_opch_num);
				cap = BAND_INVALID_CAP;
			} else {
				switch (opclass) {
				case 81:
				case 82:
				case 83:
				case 84:
				case 101:
				case 102:
				case 103:
				case 112:
				case 113:
				case 114:
					cap = BAND_2G_CAP;
				break;
				case 94:
				case 95:
				case 96:
				case 104:
				case 105:
				case 106:
				case 107:
				case 108:
				case 109:
				case 110:
				case 111:
				case 121:
				case 122:
				case 123:
				case 124:
				case 125:
				case 126:
				case 127:
					cap = BAND_5GH_CAP;
				break;
				case 115:
				case 116:
				case 117:
				case 118:
				case 119:
				case 120:
					cap = BAND_5GL_CAP;
				break;
				case 128:
				case 129:
				case 130:
					hex_dump("chset",chset,chnum);
					hex_dump("non_opch",non_opch,non_opch_num);
					for (j = 0; j < chnum; j++) {
						for (k = 0; k < non_opch_num; k++) {
							if (chset[j] == non_opch[k])
								break;
						}
						if ((k == non_opch_num) && (!ctx->MAP_Cer)) {
							if (chset[j] < 100)
								cap |= BAND_5GL_CAP;
							else
								cap |= BAND_5GH_CAP;
							debug(DEBUG_TRACE, "opclass=%d channel(%d) support cap=%d\n", opclass, chset[j], cap);
						}
					}
				break;
				default:
					debug(DEBUG_ERROR, "unknown opclass %d\n", opclass);
				break;

				}
				debug(DEBUG_TRACE, "opclass=%d band_cap=%d\n", opclass, cap);
			}
			break;
		}
		i++;
	} while (opcls[i].opclass != 0);

	return cap;
}


int parse_ap_radio_basic_cap_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf,
	unsigned char *identifier, unsigned char* max_bss_num, unsigned char *band_cap)
{
	unsigned char *temp_buf = NULL;
    unsigned short length = 0;
	unsigned char opnum = 0, i = 0, non_opch_num = 0;
	unsigned char opclass = 0;
	unsigned char cap = 0;

    temp_buf = buf;

    if((*temp_buf) == AP_RADIO_BASIC_CAPABILITY_TYPE)
        temp_buf++;
    else
        return -1;

    length = *(unsigned short *)temp_buf;
   	length = be2cpu16(length);
    temp_buf+=2;

	/*here only parse identifier of ap_radio_basic_cap_tlv*/
	memcpy(identifier, temp_buf, 6);
	temp_buf += 6;
	*max_bss_num = *temp_buf++;
	/*Number of operating classes*/
	opnum = *temp_buf++;
	*band_cap = 0;
	for (i = 0; i < opnum; i++) {
		opclass =  *temp_buf;
		temp_buf += 2;
		non_opch_num = *temp_buf++;
		cap = get_bandcap(ctx, opclass, non_opch_num, temp_buf);
		*band_cap |= cap;
		temp_buf += non_opch_num;
	}
	debug(DEBUG_TRACE, "bandcap=%d\n", *band_cap);
	debug(DEBUG_TRACE, "radio_id(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(identifier));

    return (length+3);
}

#ifdef MAP_R2
void delete_exist_radio_identifier(struct p1905_managerd_ctx *ctx)
{
	struct radio_identifier_db *p_tmp = NULL, *p_next = NULL;

	if(!SLIST_EMPTY(&ctx->metric_entry.radio_identifier_head))
	{
		p_tmp = SLIST_FIRST(&ctx->metric_entry.radio_identifier_head);
		while (p_tmp)
		{
			debug(DEBUG_TRACE, "remove radio identifier(%02x:%02x:%02x:%02x:%02x:%02x) \n", PRINT_MAC(p_tmp->identifier));

			p_next = SLIST_NEXT(p_tmp, radio_identifier_entry);

			SLIST_REMOVE(&ctx->metric_entry.radio_identifier_head, p_tmp,
				radio_identifier_db, radio_identifier_entry);
			free(p_tmp);
			p_tmp = p_next;
		}
	}
}

void add_new_radio_identifier(struct p1905_managerd_ctx *ctx, unsigned char *identifier)
{
	struct radio_identifier_db *p_new = NULL;

	p_new = (struct radio_identifier_db *)malloc(sizeof(struct radio_identifier_db));

	if (!p_new) {
		return;
	}

	memcpy(p_new->identifier, identifier, ETH_ALEN);
	debug(DEBUG_TRACE, "add radio identifier(%02x:%02x:%02x:%02x:%02x:%02x) \n", PRINT_MAC(p_new->identifier));

	SLIST_INSERT_HEAD(&ctx->metric_entry.radio_identifier_head, p_new, radio_identifier_entry);
}
#endif // #ifdef MAP_R2

int parse_ap_radio_identifier_tlv(unsigned char *buf, struct p1905_managerd_ctx *ctx
#ifdef MAP_R2
, unsigned char need_store
#endif // #ifdef MAP_R2
)
{
	unsigned char *temp_buf;
	unsigned short length = 0;
	temp_buf = buf;

	if((*temp_buf) == AP_RADIO_IDENTIFIER_TYPE)
		temp_buf++;
	else
		return -1;

	length = ETH_ALEN;
	temp_buf+=2;
#ifdef MAP_R2
	if (need_store)
		add_new_radio_identifier(ctx, temp_buf);
#endif // #ifdef MAP_R2
	return (length+3);
}

int parse_client_assoc_event_tlv(unsigned char *buf, unsigned char *mac,
	unsigned char *bssid, unsigned char *event)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == CLIENT_ASSOCIATION_EVENT_TYPE)
		temp_buf++;
	else
		return -1;

	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);
	if (length != 13) {
		debug(DEBUG_ERROR, "length(%d) error\n", length);
		return -1;
	}
	temp_buf+=2;

	debug(DEBUG_TRACE, "client(%02x:%02x:%02x:%02x:%02x:%02x) ", PRINT_MAC(temp_buf));
	os_memcpy(mac, temp_buf, ETH_ALEN);
	temp_buf += ETH_ALEN;
	os_memcpy(bssid, temp_buf, ETH_ALEN);
	temp_buf += ETH_ALEN;
	*event = *temp_buf;

	debug(DEBUG_TRACE, "sta(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(mac));
	debug(DEBUG_TRACE, "%s the ", (*event) ? "join" : "left");
	debug(DEBUG_TRACE, "bss(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(bssid));

	return (length+3);

}

int parse_client_info_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx, unsigned char *bssid, unsigned char *sta_mac)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == CLIENT_INFO_TYPE)
		temp_buf++;
	else
		return -1;

	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);
	if (length != 12) {
		debug(DEBUG_ERROR, "length(%d) error\n", length);
		return -1;
	}
	temp_buf+=2;

	debug(DEBUG_TRACE, "BSSID (%02x:%02x:%02x:%02x:%02x:%02x) ", PRINT_MAC(temp_buf));
	memcpy(bssid, temp_buf, ETH_ALEN);
	temp_buf += ETH_ALEN;
	debug(DEBUG_TRACE, "client (%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(temp_buf));
	memcpy(sta_mac, temp_buf, ETH_ALEN);
	return (length+3);

}


int append_operational_bss_tlv(unsigned char *pkt,
	struct list_head_oper_bss *opbss_head)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;
	int radio_num = 0;
	unsigned char exist = 0;
	struct op_bss_db *bss_info = NULL;
	struct operational_bss_db *operbss_info = NULL;

	temp_buf = pkt;

  /*firstly shift to tlv value*/
	temp_buf += 3;
	total_length += 3;

	temp_buf += 1;
	total_length += 1;

	if(!SLIST_EMPTY(opbss_head)) {
		SLIST_FOREACH(operbss_info, opbss_head, oper_bss_entry)
		{
			exist = 1;
			radio_num++;
			memcpy(temp_buf, operbss_info->identifier, ETH_ALEN);
			debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
						PRINT_MAC(operbss_info->identifier));
			temp_buf += ETH_ALEN;
			total_length += ETH_ALEN;

			*temp_buf = operbss_info->oper_bss_num;
			temp_buf += 1;
			total_length += 1;
			if (!SLIST_EMPTY(&operbss_info->op_bss_head)) {
				SLIST_FOREACH(bss_info, &operbss_info->op_bss_head, op_bss_entry)
				{
					debug(DEBUG_TRACE, "BSSID(%02x:%02x:%02x:%02x:%02x:%02x)\n",
						PRINT_MAC(bss_info->bssid));
					memcpy(temp_buf, bss_info->bssid, ETH_ALEN);
					temp_buf += ETH_ALEN;
					total_length += ETH_ALEN;
					*temp_buf = bss_info->ssid_len;
					temp_buf += 1;
					total_length += 1;
					memcpy(temp_buf, bss_info->ssid, bss_info->ssid_len);
					temp_buf += bss_info->ssid_len;
					total_length += bss_info->ssid_len;
				}
			} else {
				debug(DEBUG_TRACE, "interface of this radio are all off\n");
			}
		 }
	 }

	/*if really has the p1905.1 neighbor device, append type & length */
	if(exist) {
		(*pkt) = AP_OPERATIONAL_BSS_TYPE;
		*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);
		*(pkt + 3) = radio_num;
		debug(DEBUG_TRACE, "%s radio_num=%d\n",__func__,radio_num);
		return total_length;
	} else {
		return 0;
	}

}

int append_associated_clients_tlv(unsigned char *pkt,
	struct list_head_assoc_clients *asscli_head, unsigned int cnt)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf = NULL;
	int bss_num = 0, cli_num = 0;
	unsigned char bss_exist = 0;
	struct clients_db *clis_info = NULL;
	struct associated_clients_db *assc_clis_info = NULL;
	unsigned short time = 0;

	temp_buf = pkt;

  /*firstly shift to tlv value*/
	temp_buf += 3;
	total_length += 3;

	temp_buf += 1;
	total_length += 1;

	if(!SLIST_EMPTY(asscli_head)) {
		SLIST_FOREACH(assc_clis_info, asscli_head, assoc_clients_entry)
		{
			bss_exist = 1;
			bss_num++;
			memcpy(temp_buf, assc_clis_info->bssid, ETH_ALEN);
			temp_buf += ETH_ALEN;
			total_length += ETH_ALEN;
			debug(DEBUG_TRACE, "assco_clients_num=%d\n", assc_clis_info->assco_clients_num);
			*(unsigned short *)(temp_buf) = cpu2be16(assc_clis_info->assco_clients_num);
			temp_buf += 2;
			total_length += 2;

			if (!SLIST_EMPTY(&assc_clis_info->clients_head)) {
				cli_num = 0;
				SLIST_FOREACH(clis_info, &assc_clis_info->clients_head, clients_entry)
				{
					cli_num++;
					memcpy(temp_buf, clis_info->mac, ETH_ALEN);
					temp_buf += ETH_ALEN;
					total_length += ETH_ALEN;

					if ((cnt - clis_info->time) > 65534)
						time = 65535;
					else
						time = (unsigned short)(cnt - clis_info->time);
					debug(DEBUG_TRACE, "time=%d\n", time);
					*(unsigned short *)(temp_buf) = cpu2be16(time);
					temp_buf += 2;
					total_length += 2;
				}
				if (cli_num != assc_clis_info->assco_clients_num) {
					debug(DEBUG_ERROR, "error! assco_clients_num mismatch");
					debug(DEBUG_ERROR, "cli_num=%d, assco_clients_num=%d\n", cli_num,
						assc_clis_info->assco_clients_num);
					return 0;
				}
			} else {
				debug(DEBUG_TRACE, "no station assoc %02x:%02x:%02x:%02x:%02x:%02x\n", PRINT_MAC(assc_clis_info->bssid));
				continue;
			}
		 }
	 }

	/*if really has the p1905.1 neighbor device, append type & length */
	if(bss_exist) {
		(*pkt) = AP_ASSOCIATED_CLIENTS_TYPE;
		*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);
		*(pkt + 3) = bss_num;
		return total_length;
	} else {
		return 0;
	}
}
#ifdef MAP_R2
int append_map_version_tlv(unsigned char *pkt, unsigned char version)
{
	unsigned char *temp_buf = NULL;
	unsigned short total_length = 0;

	temp_buf = pkt;
	total_length = 4;
	(*temp_buf) = MULTI_AP_VERSION_TYPE;
	*(unsigned short *)(temp_buf+1) = cpu2be16(total_length-3);
	*(temp_buf+3) = version;

	return total_length;
}

int append_r2_cap_tlv(unsigned char *pkt, struct ap_r2_capability *r2_cap)
{
	unsigned char *temp_buf = NULL;
	unsigned short total_length = 0;

	temp_buf = pkt;
	(*temp_buf) = R2_AP_CAPABILITY_TYPE;
	temp_buf += 3;

	/*
	1. Set the Max Number Service Prioritization Rules in the R2 AP Capability TLV to 6 or greater.
	2. If the Multi-AP R2 Agent Advanced Service Prioritization Capability fields are included,
		the Multi-AP R2 Agent shall set the Max Number Advance Service Prioritization Rules
		in the R2 AP Capability TLV to 0 or greater.
	3. Set the Max Number of Permitted Destination MAC Addresses field to 1 or greater and
		shall support Packet Filtering using up to that number of Permitted Destination MAC Addresses.
	4. Set the Byte Counter Units field to 0x01 (KiB (kibibytes)).
	5. Set the Max Total Number of VIDs field to 2 or greater.
	6. Set the Number of Ethernet Interfaces that can be configured as Edge field to 0 or
		greater and add an Interface ID field for each logical Ethernet interface that
		can be configured as Edge
	*/

	*temp_buf = r2_cap->max_total_num_sp_rules;
	temp_buf += 2;	/*sizeof(max_total_num_sp_rules) + sizeof(reserved1)*/

	/* Byte Counter Units: bits 7-6
	**	0: bytes
	**	1: kibibytes (KiB)
	**	2: mebibytes (MiB)
	**	3: reserved
	*/
	*temp_buf = 0;
	if (r2_cap->byte_counter_units == 1)
		*temp_buf |= 0x40;
	else if (r2_cap->byte_counter_units == 2)
		*temp_buf |= 0x80;

	if (r2_cap->basic_sp_flag == 1)
		*temp_buf |= 0x20;
	if (r2_cap->enhanced_sp_flag == 1)
		*temp_buf |= 0x10;

	temp_buf += 1;

	*temp_buf++ = r2_cap->max_total_num_vid;

	total_length = (unsigned short)(temp_buf - pkt);

	*(unsigned short *)(pkt + 1) = cpu2be16(total_length - 3);

	return total_length;
}

int append_metric_collection_interval_tlv(unsigned char *pkt, unsigned int interval)
{
	unsigned char *temp_buf = NULL;
	unsigned short total_length = 0;

	temp_buf = pkt;
	total_length = sizeof(unsigned int) + 3;
	(*temp_buf) = METRIC_COLLECTION_INTERVAL_TYPE;
	*(unsigned short *)(temp_buf+1) = cpu2be16(sizeof(unsigned int));
	temp_buf += 3;
	*(unsigned int *)(temp_buf) = cpu2be32(interval);

	return total_length;
}

int append_default_8021Q_tlv(unsigned char *pkt, unsigned char *al_mac,
	unsigned char bss_num, struct set_config_bss_info *bss_info)
{
	unsigned char *temp_buf = NULL;
	unsigned short total_length = 0;
	unsigned char i = 0;


	for (i = 0; i < MAX_SET_BSS_INFO_NUM; i ++) {
		if (memcmp(al_mac, bss_info[i].mac, ETH_ALEN))
			continue;

		if (bss_info[i].pvid)
			break;
	}

	if (i >= bss_num)
		return 0;

	temp_buf = pkt;
	*temp_buf++ = DEFAULT_8021Q_SETTING_TYPE;
	*(unsigned short *)temp_buf = cpu2be16(3);
	temp_buf += 2;

	*(unsigned short *)temp_buf = cpu2be16(bss_info[i].pvid);
	temp_buf +=2;

	*temp_buf++ = (bss_info[i].pcp << 5) & 0xE0;

	total_length = temp_buf - pkt;
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

	return total_length;
}


int append_traffic_separation_tlv(unsigned char *pkt, unsigned char *al_mac,
	unsigned char bss_num, struct set_config_bss_info *bss_info)
{
	unsigned char *temp_buf = pkt;
	unsigned short total_length = 0;
	unsigned char i = 0, ssid_num = 0;

	debug(DEBUG_OFF, "al_mac %02x:%02x:%02x:%02x:%02x:%02x\n", PRINT_MAC(al_mac));

	temp_buf += 3;

	/* skip ssid num 1 byte */
	temp_buf ++;

	for (i = 0; i < MAX_SET_BSS_INFO_NUM; i ++) {
		debug(DEBUG_OFF, "bss[%d] mac %02x:%02x:%02x:%02x:%02x:%02x, vid %d, ssid %s, ssid_len %d\n"
			, i, PRINT_MAC(bss_info[i].mac), bss_info[i].vid, bss_info[i].ssid, bss_info[i].ssid_len);
		if (memcmp(al_mac, bss_info[i].mac, ETH_ALEN))
			continue;

		if (!bss_info[i].vid)
			continue;

		ssid_num ++;

		/* ssid len 1 byte */
		*temp_buf++ = bss_info[i].ssid_len;

		/* ssid */
		memcpy(temp_buf, bss_info[i].ssid, bss_info[i].ssid_len);
		temp_buf += bss_info[i].ssid_len;

		/* vlan id */
		*(unsigned short *)temp_buf = cpu2be16(bss_info[i].vid);
		temp_buf += 2;
	}

	if (ssid_num) {
		total_length = temp_buf - pkt;

		*pkt = TRAFFIC_SEPARATION_POLICY_TYPE;

		/* ts tlv len */
		*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

		/* ssid num */
		*(pkt+3) = ssid_num;


	}
	/* fix cross-vendor test issue with Broadcom. Broadcom will re-onboarding after policy config
	** Broadcom will remove there policy after receiving M2 with ssid num == 0
	** only use policy config req to remove policy
	** so if ssid num ==0, we don't need to add this tlv in M2
	*/
	else
		debug(DEBUG_OFF, "no ssid need to be configured, don't need to add TS tlv in M2\n");


	return total_length;
}



int append_ap_radio_advan_tlv(unsigned char *pkt, struct ap_radio_advan_cap *radio_adv_capab)
{
	unsigned char *temp_buf = NULL;
	unsigned short total_length = 0;
	struct ap_radio_advan_cap *radio_advan_capability;

	temp_buf = pkt;
	total_length = 10;
	(*temp_buf) = AP_RADIO_ADVANCED_CAPABILITIES_TYPE;
	*(unsigned short *)(temp_buf+1) = cpu2be16(total_length-3);
	radio_advan_capability = (struct ap_radio_advan_cap *)(temp_buf + 3);
	os_memcpy(radio_advan_capability->radioid,radio_adv_capab->radioid,6);
	radio_advan_capability->flag=radio_adv_capab->flag;

	return total_length;
}

#endif // #ifdef MAP_R2

int append_client_assoc_event_tlv(unsigned char *pkt,
		struct map_client_association_event *assoc_evt)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf = NULL;

	*pkt++ = CLIENT_ASSOCIATION_EVENT_TYPE;
	total_length += 1;

	*(unsigned short *)(pkt) = cpu2be16(CLIENT_ASSOCIATION_EVENT_LENGTH);
	pkt += 2;
	total_length += 2;

	temp_buf = pkt;
	memcpy(temp_buf, assoc_evt->sta_mac, ETH_ALEN);
	temp_buf += ETH_ALEN;
	memcpy(temp_buf, assoc_evt->bssid, ETH_ALEN);
	temp_buf += ETH_ALEN;
	*temp_buf = assoc_evt->assoc_evt;

	total_length += CLIENT_ASSOCIATION_EVENT_LENGTH;
	return total_length;
}

int append_cli_info_tlv(unsigned char *pkt,
		struct client_info *info)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf = NULL;

	*pkt = CLIENT_INFO_TYPE;
	total_length += 1;

	*(unsigned short *)(pkt+1) = cpu2be16(CLIENT_INFO_LENGTH);
	total_length += 2;

	temp_buf = pkt + 3;

	memcpy(temp_buf, info->bssid, ETH_ALEN);
	temp_buf += ETH_ALEN;
	memcpy(temp_buf, info->sta_mac, ETH_ALEN);
	temp_buf += ETH_ALEN;

	total_length += CLIENT_INFO_LENGTH;
	return total_length;
}

int append_cli_capability_report_tlv(struct p1905_managerd_ctx* ctx, unsigned char *pkt)
{
	unsigned short total_length = 0;
	unsigned short length = 1;
	unsigned char *temp_buf = NULL;
	struct associated_clients_db* pbss_db = NULL;
	struct clients_db* sta_db = NULL;

	*pkt = CLIENT_CAPABILITY_REPORT_TYPE;
	total_length += 1;

	SLIST_FOREACH(pbss_db, &(ctx->ap_cap_entry.assoc_clients_head), assoc_clients_entry)
	{
		SLIST_FOREACH(sta_db, &(pbss_db->clients_head), clients_entry)
		{
			if(!memcmp(sta_db->mac, ctx->sinfo.cinfo.sta_mac, ETH_ALEN))
			{
				break;
			}
		}
		if(sta_db != NULL)
		{
			break;
		}
	}
	temp_buf = pkt + 3;
	if(!sta_db)
	{
		debug(DEBUG_TRACE, "station(%02x:%02x:%02x:%02x:%02x:%02x) not associated",
			PRINT_MAC(ctx->sinfo.cinfo.sta_mac));
		*temp_buf = 0x02;
		total_length += 1;
	}
	else
	{
		*temp_buf = 0x00;
		total_length += 1;
		temp_buf += 1;

		memcpy(temp_buf, sta_db->frame, sta_db->frame_len);
		total_length += sta_db->frame_len;
		length += sta_db->frame_len;
	}

	*(unsigned short *)(pkt+1) = cpu2be16(length);
	total_length += 2;

	return total_length;
}

int update_ap_ht_cap(struct p1905_managerd_ctx* ctx, struct ap_ht_capability *pcap)
{
	struct ap_ht_cap_db *pdb;
	SLIST_FOREACH(pdb, &(ctx->ap_cap_entry.ap_ht_cap_head), ap_ht_cap_entry)
	{
		/*exist entry, update it*/
		if(!memcmp(pdb->identifier, pcap->identifier, ETH_ALEN))
		{
			debug(DEBUG_TRACE, "update ap ht cap for existing radio(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(pcap->identifier));
			pdb->tx_stream = pcap->tx_stream;
			pdb->rx_stream = pcap->rx_stream;
			pdb->sgi_20 = pcap->sgi_20;
			pdb->sgi_40 = pcap->sgi_40;
			pdb->ht_40 = pcap->ht_40;
			return 0;
		}
	}
	/*insert new one*/
	debug(DEBUG_TRACE, "insert new ap ht cap for radio(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(pcap->identifier));
	pdb = (struct ap_ht_cap_db*)malloc(sizeof(struct ap_ht_cap_db));
	if(pdb == NULL)
	{
		debug(DEBUG_ERROR, "allocate memory fail\n");
		return -1;
	}
	memcpy(pdb->identifier, pcap->identifier, ETH_ALEN);
	pdb->tx_stream = pcap->tx_stream;
	pdb->rx_stream = pcap->rx_stream;
	pdb->sgi_20 = pcap->sgi_20;
	pdb->sgi_40 = pcap->sgi_40;
	pdb->ht_40 = pcap->ht_40;
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ap_ht_cap_head), pdb, ap_ht_cap_entry);
	return 0;
}

int update_ap_vht_cap(struct p1905_managerd_ctx* ctx, struct ap_vht_capability *pcap)
{
	struct ap_vht_cap_db *pdb;
	SLIST_FOREACH(pdb, &(ctx->ap_cap_entry.ap_vht_cap_head), ap_vht_cap_entry)
	{
		/*exist entry, update it*/
		if(!memcmp(pdb->identifier, pcap->identifier, ETH_ALEN))
		{
			debug(DEBUG_TRACE, "update ap vht cap for existing radio(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(pcap->identifier));
			pdb->vht_tx_mcs = pcap->vht_tx_mcs;
			pdb->vht_rx_mcs = pcap->vht_rx_mcs;
			pdb->mu_beamformer = pcap->mu_beamformer;
			pdb->sgi_160 = pcap->sgi_160;
			pdb->sgi_80 = pcap->sgi_80;
			pdb->vht_160 = pcap->vht_160;
			pdb->vht_8080 = pcap->vht_8080;
			pdb->tx_stream = pcap->tx_stream;
			pdb->rx_stream = pcap->rx_stream;
			pdb->su_beamformer = pcap->su_beamformer;
			return 0;
		}
	}
	/*insert new one*/
	debug(DEBUG_TRACE, "insert new ap vht cap for radio(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(pcap->identifier));
	pdb = (struct ap_vht_cap_db*)malloc(sizeof(struct ap_vht_cap_db));
	if(pdb == NULL)
	{
		debug(DEBUG_ERROR, "[%s]allocate memory fail\n", __func__);
		return -1;
	}
	memcpy(pdb->identifier, pcap->identifier, ETH_ALEN);
	pdb->vht_tx_mcs = pcap->vht_tx_mcs;
	pdb->vht_rx_mcs = pcap->vht_rx_mcs;
	pdb->mu_beamformer = pcap->mu_beamformer;
	pdb->sgi_160 = pcap->sgi_160;
	pdb->sgi_80 = pcap->sgi_80;
	pdb->vht_160 = pcap->vht_160;
	pdb->vht_8080 = pcap->vht_8080;
	pdb->tx_stream = pcap->tx_stream;
	pdb->rx_stream = pcap->rx_stream;
	pdb->su_beamformer = pcap->su_beamformer;
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ap_vht_cap_head), pdb, ap_vht_cap_entry);
	return 0;
}

int update_ap_he_cap(struct p1905_managerd_ctx* ctx, struct ap_he_capability *pcap)
{
	struct ap_he_cap_db *pdb;

	if (pcap->he_mcs_len > 12 || pcap->he_mcs_len < 2) {
		debug(DEBUG_ERROR, "invalid he_mcs_len(%d)\n", pcap->he_mcs_len);
		return -1;
	}

	SLIST_FOREACH(pdb, &(ctx->ap_cap_entry.ap_he_cap_head), ap_he_cap_entry)
	{
		/*exist entry, update it*/
		if(!memcmp(pdb->identifier, pcap->identifier, ETH_ALEN))
		{
			debug(DEBUG_TRACE, "update ap he cap for existing radio(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(pcap->identifier));
			pdb->he_mcs_len = pcap->he_mcs_len;
			os_memcpy(pdb->he_mcs, pcap->he_mcs, pdb->he_mcs_len);
			pdb->tx_stream = pcap->tx_stream;
			pdb->rx_stream = pcap->rx_stream;
			pdb->he_8080 = pcap->he_8080;
			pdb->he_160 = pcap->he_160;
			pdb->su_bf_cap = pcap->su_bf_cap;
			pdb->mu_bf_cap = pcap->mu_bf_cap;
			pdb->ul_mu_mimo_cap = pcap->ul_mu_mimo_cap;
			pdb->ul_mu_mimo_ofdma_cap = pcap->ul_mu_mimo_ofdma_cap;
			pdb->dl_mu_mimo_ofdma_cap = pcap->dl_mu_mimo_ofdma_cap;
			pdb->ul_ofdma_cap = pcap->ul_ofdma_cap;
			pdb->dl_ofdma_cap = pcap->dl_ofdma_cap;
			return 0;
		}
	}
	/*insert new one*/
	debug(DEBUG_TRACE, "insert new ap vht cap for radio(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(pcap->identifier));
	pdb = (struct ap_he_cap_db*)malloc(sizeof(struct ap_he_cap_db));
	if(pdb == NULL)
	{
		debug(DEBUG_ERROR, "allocate memory fail\n");
		return -1;
	}
	memcpy(pdb->identifier, pcap->identifier, ETH_ALEN);
	pdb->he_mcs_len = pcap->he_mcs_len;
	os_memcpy(pdb->he_mcs, pcap->he_mcs, pdb->he_mcs_len);
	pdb->tx_stream = pcap->tx_stream;
	pdb->rx_stream = pcap->rx_stream;
	pdb->he_8080 = pcap->he_8080;
	pdb->he_160 = pcap->he_160;
	pdb->su_bf_cap = pcap->su_bf_cap;
	pdb->mu_bf_cap = pcap->mu_bf_cap;
	pdb->ul_mu_mimo_cap = pcap->ul_mu_mimo_cap;
	pdb->ul_mu_mimo_ofdma_cap = pcap->ul_mu_mimo_ofdma_cap;
	pdb->dl_mu_mimo_ofdma_cap = pcap->dl_mu_mimo_ofdma_cap;
	pdb->ul_ofdma_cap = pcap->ul_ofdma_cap;
	pdb->dl_ofdma_cap = pcap->dl_ofdma_cap;
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ap_he_cap_head), pdb, ap_he_cap_entry);
	return 0;
}

int delete_exist_ch_prefer_info(struct p1905_managerd_ctx *ctx,
	unsigned char *identifier)
{
	 struct ch_prefer_db *chcap = NULL;
	 struct prefer_info_db *prefer = NULL, *prefer_tmp = NULL;
	 int i = 0;

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.ch_prefer_head)))
    {
        SLIST_FOREACH(chcap, &(ctx->ap_cap_entry.ch_prefer_head), ch_prefer_entry)
        {
            if(!memcmp(chcap->identifier, identifier, ETH_ALEN))
            {
                if(!SLIST_EMPTY(&(chcap->prefer_info_head)))
                {
                    prefer = SLIST_FIRST(&(chcap->prefer_info_head));
                    while(prefer)
                    {
                    	debug(DEBUG_TRACE, "delete_exist struct prefer_info_db\n");
						debug(DEBUG_TRACE, "opclass=%d, ch_num=%d perference=%d, reason=%d\n",
							prefer->op_class, prefer->ch_num, prefer->perference, prefer->reason);
						debug(DEBUG_TRACE, "ch_list: ");
						for (i = 0; i < prefer->ch_num; i++) {
							debugbyte(DEBUG_TRACE, "%d ", prefer->ch_list[i]);
						}
						debugbyte(DEBUG_TRACE, "\n");
                        prefer_tmp = SLIST_NEXT(prefer, prefer_info_entry);
                        SLIST_REMOVE(&(chcap->prefer_info_head), prefer,
                                    prefer_info_db, prefer_info_entry);
                        free(prefer);
                        prefer = prefer_tmp;
                    }
                }
				debug(DEBUG_TRACE, "delete_exist struct ch_prefer_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(chcap->identifier));
				debug(DEBUG_TRACE, "oper_bss_num:%d\n", chcap->op_class_num);
                SLIST_REMOVE(&(ctx->ap_cap_entry.ch_prefer_head), chcap,\
                            ch_prefer_db, ch_prefer_entry);
                free(chcap);
                break;
            }
        }
    }

	return wapp_utils_success;
}

int delete_all_ch_prefer_info(struct list_head_ch_prefer *ch_prefer_head)
{
	struct ch_prefer_db *chcap = NULL, *chcap_tmp = NULL;
	struct prefer_info_db *prefer = NULL, *prefer_tmp = NULL;
	int i = 0;

	chcap = SLIST_FIRST(ch_prefer_head);
	while (chcap != NULL) {
		chcap_tmp = SLIST_NEXT(chcap, ch_prefer_entry);
        prefer = SLIST_FIRST(&(chcap->prefer_info_head));
        while(prefer)
        {
        	debug(DEBUG_TRACE, "delete_exist struct prefer_info_db\n");
			debug(DEBUG_TRACE, "opclass=%d, ch_num=%d perference=%d, reason=%d\n",
				prefer->op_class, prefer->ch_num, prefer->perference, prefer->reason);
			debug(DEBUG_TRACE, "ch_list: ");
			for (i = 0; i < prefer->ch_num; i++) {
				debugbyte(DEBUG_TRACE, "%d ", prefer->ch_list[i]);
			}
			debugbyte(DEBUG_TRACE, "\n");
            prefer_tmp = SLIST_NEXT(prefer, prefer_info_entry);
            free(prefer);
            prefer = prefer_tmp;
        }
		debug(DEBUG_TRACE, "delete_exist struct ch_prefer_db\n");
		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(chcap->identifier));
		debug(DEBUG_TRACE, "tx_power_limit=%d, oper_bss_num:%d\n",
			chcap->tx_power_limit, chcap->op_class_num);
		free(chcap);
		chcap = chcap_tmp;
    }
	SLIST_INIT(ch_prefer_head);

	return wapp_utils_success;
}

int insert_new_channel_prefer_info(struct p1905_managerd_ctx *ctx,
	struct ch_prefer *prefer)
{
	struct ch_prefer_db *chcap = NULL;
	struct prefer_info_db *prefer_db = NULL;
	struct prefer_info *info = NULL;
	int i = 0, j = 0;

	chcap = (struct ch_prefer_db *)malloc(sizeof(struct ch_prefer_db));
	if (!chcap) {
		debug(DEBUG_ERROR, "alloc struct ch_prefer_db fail\n");
		return wapp_utils_error;
	}
	memset(chcap, 0, sizeof(struct ch_prefer_db));
	memcpy(chcap->identifier, prefer->identifier, ETH_ALEN);
	chcap->op_class_num = prefer->op_class_num;
	SLIST_INIT(&(chcap->prefer_info_head));
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ch_prefer_head), chcap, ch_prefer_entry);

	debug(DEBUG_TRACE, "insert struct ch_prefer_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(chcap->identifier));
	debug(DEBUG_TRACE, "op_class_num:%d\n", chcap->op_class_num);

	for(i = 0; i < chcap->op_class_num; i++) {
		info = &prefer->opinfo[i];
		prefer_db = (struct prefer_info_db *)malloc(sizeof(struct prefer_info_db));
		if (!prefer_db) {
			debug(DEBUG_ERROR, "alloc struct prefer_info_db fail\n");
			return wapp_utils_error;
		}
		prefer_db->op_class = info->op_class;
		prefer_db->ch_num= info->ch_num;
		memcpy(prefer_db->ch_list, info->ch_list, prefer_db->ch_num);
		prefer_db->perference= info->perference;
		prefer_db->reason= info->reason;
		SLIST_INSERT_HEAD(&(chcap->prefer_info_head), prefer_db, prefer_info_entry);

		debug(DEBUG_TRACE, "insert struct prefer_info_db\n");
		debug(DEBUG_TRACE, "opclass=%d, ch_num=%d perference=%d, reason=%d\n",
			prefer_db->op_class, prefer_db->ch_num, prefer_db->perference, prefer_db->reason);
		debug(DEBUG_TRACE, "ch_list: ");
		for (j = 0; j < prefer_db->ch_num; j++) {
			debug(DEBUG_TRACE, "%d ", prefer_db->ch_list[j]);
		}
		debug(DEBUG_TRACE, "\n");
	}

	return wapp_utils_success;

}

int insert_new_ch_prefer_info(
	struct list_head_ch_prefer *ch_prefer_head, unsigned char *buf, unsigned short len)
{
	 struct ch_prefer_db *chcap = NULL;
	 struct prefer_info_db *prefer = NULL;
	 unsigned char *pos = buf;
	 unsigned short check_len = 0, prefer_len = 0;
	 unsigned char i = 0, j = 0, op_class = 0, ch_num = 0;

	if (len < 7) {
		debug(DEBUG_ERROR, "length error less than 7\n");
		return wapp_utils_error;
	}

    SLIST_FOREACH(chcap, ch_prefer_head, ch_prefer_entry)
    {
        if(!memcmp(chcap->identifier, buf, ETH_ALEN)) {
			debug(DEBUG_TRACE, "find struct ch_prefer_db\n");
			debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(chcap->identifier));
            break;
        }
    }

	if (!chcap) {
		chcap = (struct ch_prefer_db *)malloc(sizeof(*chcap));
		if (!chcap) {
			debug(DEBUG_ERROR, "alloc struct ch_prefer_db fail\n");
			return wapp_utils_error;
		}
		memset(chcap, 0, sizeof(*chcap));
		memcpy(chcap->identifier, pos, ETH_ALEN);
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		chcap->op_class_num = *pos;
		pos += 1;
		check_len += 1;
		SLIST_INIT(&(chcap->prefer_info_head));
		SLIST_INSERT_HEAD(ch_prefer_head, chcap, ch_prefer_entry);

		debug(DEBUG_TRACE, "insert struct ch_prefer_db\n");
		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x) op_class_num=%d\n",
			PRINT_MAC(chcap->identifier), chcap->op_class_num);
	}else {
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		chcap->op_class_num = *pos;
		pos += 1;
		check_len += 1;
	}

	for(i = 0; i < chcap->op_class_num; i++) {
		op_class = *pos;
		pos += 1;
		check_len += 1;
		ch_num = *pos;
		pos += 1;
		check_len += 1;
		prefer = (struct prefer_info_db *)malloc(sizeof(struct prefer_info_db));
		if (!prefer) {
			debug(DEBUG_ERROR, "alloc struct prefer_info_db fail\n");
			return wapp_utils_error;
		}
		memset(prefer, 0 , prefer_len);
		prefer->op_class = op_class;
		prefer->ch_num = ch_num;
		if (prefer->ch_num) {
			memcpy(prefer->ch_list, pos, ch_num);
			pos += ch_num;
			check_len += ch_num;
		}
		/*bit 4~7*/
		prefer->perference = *pos >> 4;
		/*bit 0~3*/
		prefer->reason = *pos & 0x0f;
		pos += 1;
		check_len += 1;
		SLIST_INSERT_HEAD(&(chcap->prefer_info_head), prefer, prefer_info_entry);
		debug(DEBUG_TRACE, "insert struct prefer_info_db\n");
		debug(DEBUG_TRACE, "opclass=%d, ch_num=%d perference=%d, reason=%d\n",
			prefer->op_class, prefer->ch_num, prefer->perference, prefer->reason);
		debug(DEBUG_TRACE, "ch_list: ");
		for (j = 0; j < prefer->ch_num; j++) {
			debugbyte(DEBUG_TRACE, "%d ", prefer->ch_list[j]);
		}
		debugbyte(DEBUG_TRACE, "\n");
	}

	if (len != check_len) {
		debug(DEBUG_ERROR, "length mismatch len(%d) != check_len(%d)\n",
			len, check_len);
		delete_all_ch_prefer_info(ch_prefer_head);
		return wapp_utils_error;
	}

	return wapp_utils_success;

}

int delete_exist_restriction_info(struct p1905_managerd_ctx *ctx,
	unsigned char *identifier)
{
	 struct oper_restrict_db *restriction = NULL;
	 struct restrict_db *resdb = NULL, *resdb_tmp = NULL;
	 int i = 0;

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.oper_restrict_head)))
    {
        SLIST_FOREACH(restriction, &(ctx->ap_cap_entry.oper_restrict_head), oper_restrict_entry)
        {
            if(!memcmp(restriction->identifier, identifier, ETH_ALEN))
            {
                if(!SLIST_EMPTY(&(restriction->restrict_head)))
                {
                    resdb = SLIST_FIRST(&(restriction->restrict_head));
                    while(resdb)
                    {
                    	debug(DEBUG_TRACE, "delete_exist struct oper_restrict_db\n");
						debug(DEBUG_TRACE, "opclass=%d, ch_num=%d\n",
							resdb->op_class, resdb->ch_num);
						debug(DEBUG_TRACE, "ch_list: ");
						for (i = 0; i < resdb->ch_num; i++) {
							debug(DEBUG_TRACE, "%d ", resdb->ch_list[i]);
						}
						debug(DEBUG_TRACE, "\n");
						debug(DEBUG_TRACE, "min_fre_sep_list: ");
						for (i = 0; i < resdb->ch_num; i++) {
							debug(DEBUG_TRACE, "%d ", resdb->min_fre_sep[i]);
						}
						debug(DEBUG_TRACE, "\n");
                        resdb_tmp = SLIST_NEXT(resdb, restrict_entry);
                        SLIST_REMOVE(&(restriction->restrict_head), resdb,
                                    restrict_db, restrict_entry);
                        free(resdb);
                        resdb = resdb_tmp;
                    }
                }
				debug(DEBUG_TRACE, "delete_exist struct oper_restrict_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(restriction->identifier));
				debug(DEBUG_TRACE, "oper_bss_num:%d\n", restriction->op_class_num);
                SLIST_REMOVE(&(ctx->ap_cap_entry.oper_restrict_head), restriction,
                            oper_restrict_db, oper_restrict_entry);
                free(restriction);
                break;
            }
        }
    }

	return wapp_utils_success;
}

int insert_new_restriction_info(struct p1905_managerd_ctx *ctx,
	struct restriction *op_restrict)
{
	struct oper_restrict_db *restriction_db = NULL;
	struct restrict_db *resdb = NULL;
	struct restrict_info *info = NULL;
	int i = 0, j = 0;

	restriction_db = (struct oper_restrict_db *)malloc(sizeof(struct oper_restrict_db));
	if (!restriction_db) {
		debug(DEBUG_ERROR, "alloc struct oper_restrict_db fail\n");
		return wapp_utils_error;
	}
	memset(restriction_db, 0, sizeof(struct oper_restrict_db));
	memcpy(restriction_db->identifier, op_restrict->identifier, ETH_ALEN);
	restriction_db->op_class_num = op_restrict->op_class_num;
	SLIST_INIT(&(restriction_db->restrict_head));
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.oper_restrict_head),
		restriction_db, oper_restrict_entry);

	debug(DEBUG_TRACE, "insert struct oper_restrict_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(restriction_db->identifier));
	debug(DEBUG_TRACE, "op_class_num:%d\n", restriction_db->op_class_num);

	for(i = 0; i < restriction_db->op_class_num; i++) {
		info = &op_restrict->opinfo[i];
		resdb = (struct restrict_db *)malloc(sizeof(struct restrict_db));
		if (!resdb) {
			debug(DEBUG_ERROR, "alloc struct restrict_db fail\n");
			return wapp_utils_error;
		}
		resdb->op_class = info->op_class;
		resdb->ch_num= info->ch_num;
		memcpy(resdb->ch_list, info->ch_list, resdb->ch_num);
		memcpy(resdb->min_fre_sep, info->fre_separation, resdb->ch_num);
		SLIST_INSERT_HEAD(&(restriction_db->restrict_head), resdb, restrict_entry);

		debug(DEBUG_TRACE, "insert struct restrict_db\n");
		debug(DEBUG_TRACE, "opclass=%d, ch_num=%d\n",
			resdb->op_class, resdb->ch_num);
		debug(DEBUG_TRACE, "ch_list: ");
		for (j = 0; j < resdb->ch_num; j++) {
			debug(DEBUG_TRACE, "%d ", resdb->ch_list[j]);
		}
		debug(DEBUG_TRACE, "\n");
		debug(DEBUG_TRACE, "min_fre_sep_list: ");
		for (j = 0; j < resdb->ch_num; j++) {
			debug(DEBUG_TRACE, "%d ", resdb->min_fre_sep[j]);
		}
		debug(DEBUG_TRACE, "\n");
	}

	return wapp_utils_success;

}

int parse_channel_preference_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf, struct list_head_ch_prefer *ch_prefer_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == CH_PREFERENCE_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if(ctx->Certification == 1)
    {
	    //shift to tlv value field
	    temp_buf+=2;

		/*insert new channel preference info*/
		if (0 > insert_new_ch_prefer_info(ch_prefer_head, temp_buf, length)) {
			debug(DEBUG_ERROR, "insert_new_ch_prefer_info fail\n");
			return -1;
		}
	}
    return (length+3);
}


int update_tx_power_limit_info(struct p1905_managerd_ctx *ctx,
	unsigned char *identifier, signed char power)
{
	struct ch_prefer_db *chcap = NULL;

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.ch_prefer_head))) {
        SLIST_FOREACH(chcap, &(ctx->ap_cap_entry.ch_prefer_head), ch_prefer_entry)
        {
            if(!memcmp(chcap->identifier, identifier, ETH_ALEN)) {
				debug(DEBUG_TRACE, "find struct ch_prefer_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(chcap->identifier));
				chcap->tx_power_limit = power;
				debug(DEBUG_TRACE, "tx_power_limit:%d\n", chcap->tx_power_limit);
                break;
            }
        }
    }

	if (chcap == NULL) {
		debug(DEBUG_ERROR, "identifier(%02x:%02x:%02x:%02x:%02x:%02x) not exist, create new one\n",
			PRINT_MAC(identifier));
		chcap = (struct ch_prefer_db *)malloc(sizeof(*chcap));
		if(chcap == NULL) {
			debug(DEBUG_ERROR, "alloc memory fail\n");
			return wapp_utils_error;
		}
		memset(chcap, 0, sizeof(*chcap));
		memcpy(chcap->identifier, identifier, ETH_ALEN);
		chcap->tx_power_limit = power;
		debug(DEBUG_TRACE, "tx_power_limit:%d\n", chcap->tx_power_limit);
		SLIST_INIT(&(chcap->prefer_info_head));
		SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ch_prefer_head), chcap, ch_prefer_entry);
	}

	return wapp_utils_success;
}


int parse_transmit_power_limit_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == TRANSMIT_POWER_LIMIT_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if (length != 7) {
		debug(DEBUG_ERROR, "length error not eaqual 7\n");
		return -1;
	}
	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf+=2;

		if (0 > update_tx_power_limit_info(ctx, temp_buf, *(temp_buf + 6))) {
			debug(DEBUG_ERROR, "update_tx_power_limit_info fail\n");
			return -1;
		}
	}
	return (length+3);
}


int parse_channel_selection_request_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == CH_PREFERENCE_TYPE)
		{
			length = parse_channel_preference_tlv(ctx, temp_buf, &ctx->ap_cap_entry.ch_prefer_head);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error channel preference tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == TRANSMIT_POWER_LIMIT_TYPE) {

			length = parse_transmit_power_limit_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error transmit power limit tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}

	return 0;
}

unsigned char check_invalid_channel(unsigned char op_class,
	unsigned char ch_num, unsigned char *ch_list)
{
	struct global_oper_class *opcls = oper_class;
	int i = 0, j = 0, k = 0;
	unsigned char opcls_found = 0;

	/*check if the channel is in oper_class*/
	do {
		if (opcls[i].opclass == op_class) {
			opcls_found = 1;
			for (k = 0; k < ch_num; k++) {
				for (j = 0; j < opcls[i].channel_num; j++) {
					if (opcls[i].channel_set[j] == ch_list[k])
						break;
				}
				if (j == opcls[i].channel_num) {
					return 0; /*invalid channel*/
				}
			}
			break;
		}
		i++;
	} while (opcls[i].opclass != 0);

	if (opcls_found)
		return 1;
	else
		return 0;
}


unsigned char find_oper_channel(unsigned char op_class,
	unsigned char ch_num, unsigned char *ch_list)
{
	struct global_oper_class *opcls = oper_class;
	int i = 1; /* skip Invlid entry */
	int j = 0, k = 0;
	unsigned char channel = 0;

	do {
		if (opcls[i].opclass == op_class) {
			for (j = 0; j < opcls[i].channel_num; j++) {
				channel = opcls[i].channel_set[j];
				for (k = 0; k < ch_num; k++) {
					if (channel == ch_list[k])
						break;
				}
				if (k == ch_num) {
					return channel;
				}
			}
			break;
		}
		i++;
	} while (opcls[i].opclass != 0);

	return 0;
}

unsigned char find_first_oper_channel(unsigned char op_class)
{
	struct global_oper_class *opcls = oper_class;
	int i = 1; /* skip Invlid entry */
	unsigned char channel = 0;

	do {
		if (opcls[i].opclass == op_class) {
			channel = opcls[i].channel_set[0];
			return channel;
		}
		i++;
	} while (opcls[i].opclass != 0);

	return 0;
}

void update_channel_setting(struct p1905_managerd_ctx *ctx, unsigned char *almac,
	unsigned short mid)
{
	struct ch_prefer_db *chcap = NULL;
	struct prefer_info_db *prefer = NULL;
	struct ch_config_info *info = NULL;
	struct ch_stat_info *stat = NULL;
	struct ch_rep_info *rep = NULL;
	struct prefer_info_db *pinfo_db = NULL;
	int i = 0, j = 0, ch_num = 0, len = 0;
	unsigned char invalid_ch_found = 0;

    SLIST_FOREACH(chcap, &ctx->ap_cap_entry.ch_prefer_head, ch_prefer_entry)
    {
    	SLIST_FOREACH(prefer, &chcap->prefer_info_head, prefer_info_entry)
	    {
	        i++;
	    }
    }

	if (i == 0) {
		wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_OPERATING_CHANNEL_INFO,
			LIB_OPERATING_CHANNEL_INFO, NULL, NULL, NULL, 0);
		if (ctx->ch_rep) {
			ch_num = ctx->ch_rep->ch_rep_num;
			len = sizeof(struct channel_status) + ch_num * sizeof(struct ch_stat_info);
			ctx->ch_stat = (struct channel_status *)malloc(len);
			if (!ctx->ch_stat)
				return;
			memset(ctx->ch_stat, 0, len);
			ctx->ch_stat->ch_rsp_num = ch_num;
			stat = ctx->ch_stat->info;
			rep = ctx->ch_rep->info;
			for(j = 0; j < ch_num; j++) {
				memcpy(stat[j].identifier, rep[j].identifier, ETH_ALEN);
				stat[j].code = 0x00;
			}
			insert_cmdu_txq(almac, ctx->p1905_al_mac_addr,
				e_channel_selection_response,
				mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
			insert_cmdu_txq(almac, ctx->p1905_al_mac_addr,
				e_operating_channel_report,
				++ctx->mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
		}
	} else {
		len = sizeof(struct channel_status) + i * sizeof(struct ch_stat_info);
		ctx->ch_stat = (struct channel_status *)malloc(len);
		if (!ctx->ch_stat)
			return;
		memset(ctx->ch_stat, 0, len);
		ctx->ch_stat->ch_rsp_num = i;
		stat = ctx->ch_stat->info;
		len = sizeof(struct channel_setting) + i * sizeof(struct ch_config_info);
		ctx->ch_set = (struct channel_setting *)malloc(len);
		if (!ctx->ch_set)
			return;
		memset(ctx->ch_set, 0, len);
		memcpy(ctx->ch_set->almac, almac, ETH_ALEN);
		ctx->ch_set->ch_set_num = i;
		info = ctx->ch_set->chinfo;
		if(!SLIST_EMPTY(&(ctx->ap_cap_entry.ch_prefer_head))) {
	        SLIST_FOREACH(chcap, &(ctx->ap_cap_entry.ch_prefer_head), ch_prefer_entry)
	        {
	        	memcpy(stat->identifier, chcap->identifier, ETH_ALEN);
				stat->code = 0x00;
	            memcpy(info->identifier, chcap->identifier, ETH_ALEN);
				info->power = chcap->tx_power_limit;
				for(i = 0; i < ctx->itf_number; i++) {
					if(!memcmp(ctx->itf[i].mac_addr, info->identifier, ETH_ALEN)) {
						memcpy(info->ifname, ctx->itf[i].if_name, IFNAMSIZ);
						debug(DEBUG_TRACE, "find interface name %s\n", info->ifname);
						break;
					}
	       		}
				/*test code*/
				if(!SLIST_EMPTY(&(chcap->prefer_info_head))) {
					SLIST_FOREACH(pinfo_db, &(chcap->prefer_info_head), prefer_info_entry)
					{
					   if (!check_invalid_channel(pinfo_db->op_class, pinfo_db->ch_num, pinfo_db->ch_list)) {
							debug(DEBUG_TRACE, "opclass=%d\n", pinfo_db->op_class);
							hex_dump_all("invalid channel list", pinfo_db->ch_list, pinfo_db->ch_num);
						   	stat->code = 0x02;
						   	invalid_ch_found = 1;
						  	break;
					   }
					}
					if (!invalid_ch_found) {
						 SLIST_FOREACH(pinfo_db, &(chcap->prefer_info_head), prefer_info_entry)
		        		 {
		        		 	if (pinfo_db->ch_num == 0 && pinfo_db->perference == 0) {
								debug(DEBUG_TRACE, "all channels of op_class(%d) are non-operable\n",
									pinfo_db->op_class);
		        		 	} else if (pinfo_db->ch_num != 0 && pinfo_db->perference == 0) {
								/*find an operable channel from global operating class*/
								info->op_class = pinfo_db->op_class;
								debug(DEBUG_TRACE, "some channels of op_class(%d) are non-operable\n",
									pinfo_db->op_class);
								hex_dump("invalid channel list", pinfo_db->ch_list, pinfo_db->ch_num);
		        		 		info->channel = find_oper_channel(pinfo_db->op_class,
									pinfo_db->ch_num, pinfo_db->ch_list);
								break;
		        		 	} else if (pinfo_db->ch_num == 0 && pinfo_db->perference != 0) {
		        		 		/*find first channel from global operating class*/
								info->op_class = pinfo_db->op_class;
								info->channel = find_first_oper_channel(pinfo_db->op_class);
								break;
		        		 	} else {
		        		 		/*find first channel from channel list*/
								info->op_class = pinfo_db->op_class;
								info->channel = pinfo_db->ch_list[0];
								break;
		        		 	}
						 }
						 if (!info->channel)
						 	stat->code = 0x02;
					}
	            }
				debug(DEBUG_OFF, "ifname(%s), identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					info->ifname, PRINT_MAC(info->identifier));
				debug(DEBUG_OFF, "op_class=%d, channel=%d, power=%d\n",
					info->op_class, info->channel, info->power);
				info++;
				stat++;
	        }
	    }

		/*send msg to wappd*/
		wapp_set_info_by_msgtype(ctx, WAPP_USER_SET_CHANNEL_SETTING, NULL, (void *)ctx->ch_set, len);
		free(ctx->ch_set);

    	insert_cmdu_txq(almac, ctx->p1905_al_mac_addr,
			e_channel_selection_response,
			mid, ctx->itf[ctx->recent_cmdu_rx_if_number].if_name, 0);
	}

}

int update_channel_report_info(struct p1905_managerd_ctx *ctx,
	struct channel_report *rep, unsigned char len)
{
	ctx->ch_rep = (struct channel_report *)malloc(len);
	if (!ctx->ch_rep) {
		debug(DEBUG_TRACE, "alloc struct channel_report mem fail\n");
		return -1;
	}
	memcpy(ctx->ch_rep, rep, len);
	return 0;
}

int parse_steering_request_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
    unsigned char *temp_buf;
    unsigned short length = 0;
    int i =0;
	unsigned char assoc_bssid[ETH_ALEN];
	unsigned char req_mode = 0, disassoc_immmi = 0,abridged = 0;
	unsigned char sta_count = 0, bssid_count = 0, count = 0;
	unsigned short steer_window = 0, disassoc_time = 0;
	unsigned char *sta_mac_pos = NULL, *target_bssid_pos = NULL;
	struct target_bssid_info *info = NULL;
	unsigned char *target_bssid_start = NULL;



    temp_buf = buf;

    if((*temp_buf) == STEERING_REQUEST_TYPE ) {
        temp_buf++;

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		memcpy(assoc_bssid, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;

		req_mode = (*temp_buf & 0x80) ? 1 : 0;
		disassoc_immmi = (*temp_buf & 0x40) ? 1 : 0;
		abridged = (*temp_buf & 0x20) ? 1 : 0;
		temp_buf += 1;

		//if (req_mode == 0) {
			steer_window = *(unsigned short *)temp_buf;
			steer_window = be2cpu16(steer_window);
			temp_buf += 2;
		//}
		//if (disassoc_immmi) {
			disassoc_time = *(unsigned short *)temp_buf;
			disassoc_time = be2cpu16(disassoc_time);
			temp_buf += 2;
		//}

		sta_count = *temp_buf++;
		sta_mac_pos = temp_buf;
		temp_buf += sta_count * ETH_ALEN;

		if (req_mode == 1) {
			bssid_count = *temp_buf++;
		}

		count = (bssid_count > sta_count) ? bssid_count : sta_count;

		ctx->steer_req_len = sizeof(struct steer_request) +
			count * sizeof(struct target_bssid_info);
		ctx->cli_steer_req = (struct steer_request *)malloc(ctx->steer_req_len);
		if (!ctx->cli_steer_req) {
			debug(DEBUG_ERROR, "alloc cli_steer_req fail\n");
			return -1;
		}
		memset(ctx->cli_steer_req, 0, ctx->steer_req_len);
		memcpy(ctx->cli_steer_req->assoc_bssid, assoc_bssid, ETH_ALEN);
		ctx->cli_steer_req->request_mode = req_mode;
		ctx->cli_steer_req->btm_disassoc_immi = disassoc_immmi;
		ctx->cli_steer_req->btm_abridged = abridged;
		ctx->cli_steer_req->steer_window = steer_window;
		ctx->cli_steer_req->btm_disassoc_timer = disassoc_time;
		ctx->cli_steer_req->sta_count = sta_count;
		ctx->cli_steer_req->target_bssid_count = bssid_count;
		debug(DEBUG_TRACE, "%s request_mode=%d, btm_disassoc_immi=%d, "
				"btm_abridged=%d, steer_window=%d, btm_disassoc_timer=%d, "
				"sta_count=%d, target_bssid_count=%d\n", __func__,
				ctx->cli_steer_req->request_mode,
				ctx->cli_steer_req->btm_disassoc_immi,
				ctx->cli_steer_req->btm_abridged,
				ctx->cli_steer_req->steer_window,
				ctx->cli_steer_req->btm_disassoc_timer,
				ctx->cli_steer_req->sta_count,
				ctx->cli_steer_req->target_bssid_count);

		target_bssid_pos = temp_buf;
		target_bssid_start = target_bssid_pos;

		info = ctx->cli_steer_req->info;
		if (sta_count > bssid_count && bssid_count == 1) {
			/*all the specified sta roam to the same bssid*/
			for (i = 0; i < count; i++) {
				memcpy(info->target_bssid, target_bssid_pos, ETH_ALEN);
				target_bssid_pos += ETH_ALEN;
				info->op_class = *target_bssid_pos++;
				info->channel = *target_bssid_pos++;
				debug(DEBUG_TRACE, "target_bssid(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
					PRINT_MAC(info->target_bssid));
				debug(DEBUG_TRACE, "op_class=%d, channel=%d\n",info->op_class, info->channel);
				memcpy(info->sta_mac, sta_mac_pos, ETH_ALEN);
				debug(DEBUG_TRACE, "sta mac(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
					PRINT_MAC(info->sta_mac));
				sta_mac_pos += ETH_ALEN;
				target_bssid_pos = target_bssid_start;
				info++;
			}
		} else if (sta_count == bssid_count && sta_count > 0) {
			/*match sta & target bssid*/
			for (i = 0; i < count; i++) {
				memcpy(info->target_bssid, target_bssid_pos, ETH_ALEN);
				target_bssid_pos += ETH_ALEN;
				info->op_class = *target_bssid_pos++;
				info->channel = *target_bssid_pos++;
				debug(DEBUG_TRACE, "target_bssid(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
					PRINT_MAC(info->target_bssid));
				debug(DEBUG_TRACE, "op_class=%d, channel=%d\n",info->op_class, info->channel);
				memcpy(info->sta_mac, sta_mac_pos, ETH_ALEN);
				debug(DEBUG_TRACE, "sta mac(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
					PRINT_MAC(info->sta_mac));
				sta_mac_pos += ETH_ALEN;
				info++;
			}
		} else if (sta_count == 0 && bssid_count == 1) {
			/*all the associated sta roam to the same target bssid*/
			memcpy(info->target_bssid, target_bssid_pos, ETH_ALEN);
			target_bssid_pos += ETH_ALEN;
			info->op_class = *target_bssid_pos++;
			info->channel = *target_bssid_pos++;
			debug(DEBUG_TRACE, "target_bssid(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
				PRINT_MAC(info->target_bssid));
			debug(DEBUG_TRACE, "op_class=%d, channel=%d\n",info->op_class, info->channel);
			debug(DEBUG_TRACE, "sta mac(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
				PRINT_MAC(info->sta_mac));
		} else if (sta_count > 0 && bssid_count == 0) {
			/*for steering oppotunity test*/
			for (i = 0; i < count; i++) {
				debug(DEBUG_TRACE, "target_bssid(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
					PRINT_MAC(info->target_bssid));
				debug(DEBUG_TRACE, "op_class=%d, channel=%d\n",info->op_class, info->channel);
				memcpy(info->sta_mac, sta_mac_pos, ETH_ALEN);
				debug(DEBUG_TRACE, "sta mac(%d)=%02x:%02x:%02x:%02x:%02x:%02x\n",i,
					PRINT_MAC(info->sta_mac));
				sta_mac_pos += ETH_ALEN;
				info++;
			}
		} else if (sta_count == 0 && bssid_count == 0) {
			/*for steering oppotunity test*/
			debug(DEBUG_TRACE, "sta_count=bssid_count==0, no need add any target_bssid_info\n");
		} else {
			debug(DEBUG_ERROR, "unkonwn case sta_count=%d, bssid_count=%d\n",
				sta_count, bssid_count);
			return -1;
		}
	}
}
#ifdef MAP_R2
	/* R2 steering request */
    else if((*temp_buf) == R2_STEERING_REQUEST_TYPE) {

        debug(DEBUG_TRACE, "received a R2 steering request type\n");
        temp_buf++;

        //calculate tlv length
        length = *(unsigned short *)temp_buf;
    	length = be2cpu16(length);


    }
#endif
	else
		return -1;
    return (length+3);
}

int delete_exist_steering_policy(struct steer_policy *spolicy)
{
	 struct sta_db *sta = NULL, *sta_tmp = NULL;
	 struct radio_policy_db *policy = NULL, *policy_tmp = NULL;

	 sta = SLIST_FIRST(&spolicy->local_disallow_head);
	 while(sta != NULL) {
	 	debug(DEBUG_TRACE, "local_disallow_head sta_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(sta->mac));
	 	sta_tmp = SLIST_NEXT(sta, sta_entry);
		free(sta);
		sta = sta_tmp;
	}
	SLIST_INIT(&spolicy->local_disallow_head);
	spolicy->local_disallow_count = 0;

	sta = SLIST_FIRST(&spolicy->btm_disallow_head);
	while(sta != NULL) {
	 	debug(DEBUG_TRACE, "btm_disallow_head sta_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(sta->mac));
	 	sta_tmp = SLIST_NEXT(sta, sta_entry);
		free(sta);
		sta = sta_tmp;
	}
	SLIST_INIT(&spolicy->btm_disallow_head);
	spolicy->btm_disallow_count = 0;

	policy = SLIST_FIRST(&spolicy->radio_policy_head);
	while(policy != NULL) {
		debug(DEBUG_TRACE, "radio_policy_head identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(policy->identifier));
		debug(DEBUG_TRACE, "steer_policy=%d, ch_util_thres=%d, rssi_thres=%d\n",
			policy->steer_policy, policy->ch_util_thres, policy->rssi_thres);
		policy_tmp = SLIST_NEXT(policy, radio_policy_entry);
		free(policy);
		policy = policy_tmp;
	}
	SLIST_INIT(&spolicy->radio_policy_head);
	spolicy->radios = 0;

	return wapp_utils_success;
}


int parse_steering_policy_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0;
	int i =0;
	struct sta_db *sta = NULL;
	struct radio_policy_db *radio_policy = NULL;
	struct steer_policy *spolicy = &ctx->map_policy.spolicy;

	temp_buf = buf;

	if((*temp_buf) == STEERING_POLICY_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		spolicy->local_disallow_count = *temp_buf++;
		for (i = 0; i < spolicy->local_disallow_count; i++) {
			sta = (struct sta_db *)malloc(sizeof(struct sta_db));
			if (!sta) {
				debug(DEBUG_ERROR, "alloc struct sta_db for local_disallow_head fail\n");
				return -1;
			}
			memcpy(sta->mac, temp_buf, ETH_ALEN);
			temp_buf += ETH_ALEN;
			SLIST_INSERT_HEAD(&spolicy->local_disallow_head, sta, sta_entry);
		}

		spolicy->btm_disallow_count = *temp_buf++;
		for (i = 0; i < spolicy->btm_disallow_count; i++) {
			sta = (struct sta_db *)malloc(sizeof(struct sta_db));
			if (!sta) {
				debug(DEBUG_ERROR, "alloc struct sta_db for btm_disallow_head fail\n");
				return -1;
			}
			memcpy(sta->mac, temp_buf, ETH_ALEN);
			temp_buf += ETH_ALEN;
			SLIST_INSERT_HEAD(&spolicy->btm_disallow_head, sta, sta_entry);
		}

		spolicy->radios = *temp_buf++;
		for (i = 0; i < spolicy->radios; i++) {
			radio_policy = (struct radio_policy_db *)malloc(sizeof(struct radio_policy_db));
			if (!radio_policy) {
				debug(DEBUG_ERROR, "alloc struct radio_policy_db fail\n");
				return -1;
			}
			memcpy(radio_policy->identifier, temp_buf, ETH_ALEN);
			temp_buf += ETH_ALEN;
			radio_policy->steer_policy = *temp_buf++;
			radio_policy->ch_util_thres = *temp_buf++;
			radio_policy->rssi_thres = *temp_buf++;
			SLIST_INSERT_HEAD(&spolicy->radio_policy_head, radio_policy, radio_policy_entry);
		}
	}
	return (length+3);
}

int parse_cli_assoc_control_request_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned char sta_cnt = 0;
	unsigned short length = 0, len = 0;
	struct control_policy *cntrl = NULL;

	temp_buf = buf;

	if((*temp_buf) == CLI_ASSOC_CONTROL_REQUEST_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		sta_cnt = *(temp_buf + ETH_ALEN + 1 + 2);
		len = sizeof(struct control_policy) + sta_cnt * ETH_ALEN;
		ctx->steer_cntrl = (struct control_policy *)malloc(len);
		if (ctx->steer_cntrl == NULL) {
			debug(DEBUG_ERROR, "alloc memory for struct control_policy fail\n");
			return -1;
		}
		memset(ctx->steer_cntrl, 0, len);

		cntrl = ctx->steer_cntrl;
		memcpy(cntrl->bssid, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;
		cntrl->assoc_control = *temp_buf++;
		cntrl->valid_period = *(unsigned short *)temp_buf;
		cntrl->valid_period = be2cpu16(cntrl->valid_period);
		temp_buf += 2;
		cntrl->sta_list_count = *temp_buf++;
		if (cntrl->sta_list_count)
			memcpy(cntrl->sta_mac, temp_buf, cntrl->sta_list_count * ETH_ALEN);
	}
	return (length+3);
}

int parse_metric_reporting_policy_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0, i =0;
	struct metric_policy_db *metric_policy = NULL;
	struct metrics_policy *mpolicy = &ctx->map_policy.mpolicy;

	temp_buf = buf;

	if((*temp_buf) == METRIC_REPORTING_POLICY_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;

	mpolicy->report_interval = *temp_buf++;
	debug(DEBUG_TRACE, "report_interval(%d)\n", mpolicy->report_interval);
	mpolicy->radio_num = *temp_buf++;
	for (i = 0; i < mpolicy->radio_num; i++) {
		metric_policy = (struct metric_policy_db *)malloc(sizeof(struct metric_policy_db));
		if (!metric_policy) {
			debug(DEBUG_ERROR, "alloc struct metric_policy_db fail\n");
			return -1;
		}
		memcpy(metric_policy->identifier, temp_buf, ETH_ALEN);

		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(metric_policy->identifier));
		temp_buf += ETH_ALEN;
		metric_policy->rssi_thres = *temp_buf++;
		metric_policy->hysteresis_margin = *temp_buf++;
		metric_policy->ch_util_thres = *temp_buf++;
		metric_policy->sta_stats_inclusion = *temp_buf & 0x80;
		metric_policy->sta_metrics_inclusion = *temp_buf & 0x40;
		debug(DEBUG_TRACE, "rssi_thres(%d) hysteresis_margin(%d) ch_util_thres(%d) "
			"sta_stats_inclusion(%d) sta_metrics_inclusion(%d)\n",
			metric_policy->rssi_thres,
			metric_policy->hysteresis_margin,
			metric_policy->ch_util_thres,
			metric_policy->sta_stats_inclusion,
			metric_policy->sta_metrics_inclusion);
		temp_buf += 1;
		SLIST_INSERT_HEAD(&mpolicy->policy_head, metric_policy, policy_entry);
	}

	return (length+3);
}

int parse_client_steering_request_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int right_integrity = 0x1;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if((*temp_buf == STEERING_REQUEST_TYPE)
#ifdef MAP_R2
		 || (*temp_buf == R2_STEERING_REQUEST_TYPE)
#endif
		 )
		{
			integrity |= (1 << 0);

			length = parse_steering_request_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error steering request tlv\n");
				return -1;
			}

			if(ctx->Certification != 1)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}

	/*check integrity*/
	if (integrity != right_integrity) {
		debug(DEBUG_ERROR, "incomplete steering request 0x%x 0x%x\n",
			integrity, right_integrity);
		return -1;
	}

	return 0;
}

int delete_exist_metrics_policy(struct p1905_managerd_ctx *ctx, struct metrics_policy *mpolicy)
{
	struct metric_policy_db *policy = NULL, *policy_tmp = NULL;

	policy = SLIST_FIRST(&mpolicy->policy_head);
	while(policy != NULL) {
		debug(DEBUG_TRACE, "metric_policy_db identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(policy->identifier));
		debug(DEBUG_TRACE, "rssi_thres=%d, hysteresis_margin=%d, ch_util_thres=%d"
			"rssi_thres=%d, hysteresis_margin=%d\n",
			policy->rssi_thres, policy->hysteresis_margin, policy->ch_util_thres,
			policy->sta_stats_inclusion, policy->sta_metrics_inclusion);
		policy_tmp = SLIST_NEXT(policy, policy_entry);
		free(policy);
		policy = policy_tmp;
	}
	SLIST_INIT(&mpolicy->policy_head);
	mpolicy->report_interval = 0;
	mpolicy->radio_num = 0;

	return wapp_utils_success;
}


int parse_map_policy_config_request_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int all_integrity = 0xFF;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1) {
		/* Zero or one Steering Policy TLV */
		if (*temp_buf == STEERING_POLICY_TYPE) {
			integrity |= (1 << 0);

			if(ctx->Certification == 1)
			{
				/*delete the previously reserved steering policy*/
				delete_exist_steering_policy(&ctx->map_policy.spolicy);
			}

			length = parse_steering_policy_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error steering request tlv\n");
				return -1;
			}

			if(ctx->Certification == 1)
			{
				/*send the steer policy info to wapp*/
				wapp_set_steering_policy_setting(ctx, &ctx->map_policy.spolicy);
			}
			else
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}

			temp_buf += length;

		}
		/* Zero or one Metric Reporting Policy TLV */
		else if (*temp_buf == METRIC_REPORTING_POLICY_TYPE) {
			integrity |= (1 << 1);
			/*delete the previously reserved metrics policy*/
			delete_exist_metrics_policy(ctx, &ctx->map_policy.mpolicy);

			length = parse_metric_reporting_policy_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "%s error metric reporting policy tlv\n", __func__);
				return -1;
			}

			if(ctx->Certification == 1)
			{
				/*send the metrics policy info to wapp*/
				wapp_set_metrics_policy_setting(ctx, &ctx->map_policy.mpolicy);
			}
			else
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;

		}
#ifdef MAP_R2
		/* Zero or one Channel Scan Reporting Policy TLV */
		else if (*temp_buf == CHANNEL_SCAN_REPORTING_POLICY_TYPE) {
			integrity |= (1 << 2);

			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* Zero or one Default 802.1Q Settings TLV */
		else if (*temp_buf == DEFAULT_8021Q_SETTING_TYPE) {
			integrity |= (1 << 3);
			ctx->map_policy.setting.updated = 1;
			length = parse_default_802_1q_setting_tlv(temp_buf, ctx);
			debug(DEBUG_ERROR, "%s x length:%d\n", __func__, length);
			temp_buf += length;
			debug(DEBUG_ERROR, "%s xx length:%d\n", __func__, length);
		}
		/* Zero or one Traffic Separation Policy TLV */
		else if (*temp_buf == TRAFFIC_SEPARATION_POLICY_TYPE) {
			integrity |= (1 << 4);

			ctx->map_policy.tpolicy.updated = 1;
			/*delete the previously reserved traffic policy*/
			delete_exist_traffic_policy(ctx, &ctx->map_policy.tpolicy);

			length = parse_traffic_separation_policy_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "%s error trffic separation policy tlv\n", __func__);
				return -1;
			}
			temp_buf += length;
		}
#if 0
		else if (*temp_buf == PACKET_FILTERING_POLICY_TYPE) {
			integrity |= (1 << 5);
			/*delete the previously reserved packet filtering policy*/
			delete_exist_pfilter_policy(ctx, &ctx->map_policy.fpolicy);

			length = parse_packet_filtering_policy_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "%s error packet filtering policy tlv\n", __func__);
				return -1;
			}
			temp_buf += length;
		}
		/* Zero or one Ethernet Configuration Policy TLV */
		else if (*temp_buf == ETHERNET_CONFIGURATION_POLICY_TYPE) {
			integrity |= (1 << 6);
			/*delete the previously reserved eth config policy*/
			delete_exist_eth_config_policy(ctx, &ctx->map_policy.epolicy);

			length = parse_eth_config_policy_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "%s error ether config policy tlv\n", __func__);
				return -1;
			}
			temp_buf += length;
		}
#endif
		/* Zero or one Unsuccessful Association Policy TLV */
		else if (*temp_buf == UNSUCCESSFUL_ASSOCIATION_POLICY_TYPE) {
			integrity |= (1 << 7);
			//length = parse_unsuccess_assoc_policy_tlv(temp_buf, ctx);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
#endif
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if ((integrity & all_integrity) == 0) {
		debug(DEBUG_ERROR, "incomplete policy config request 0x%x 0x%x\n",
			integrity, all_integrity);
		return -1;
	}

	return 0;
}

int parse_cli_assoc_control_request_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int right_integrity = 0x1;

	temp_buf = buf;

	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == CLI_ASSOC_CONTROL_REQUEST_TYPE) {
			integrity |= (1 << 0);
			length = parse_cli_assoc_control_request_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error cli assoc control request tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if (integrity != right_integrity) {
		debug(DEBUG_ERROR, "incomplete client association control request 0x%x 0x%x\n",
			integrity, right_integrity);
		return -1;
	}

	return 0;
}

/*link metrics collection*/

void delete_ap_metrics_query_head(struct list_head_metrics_query *head)
{
	struct bss_db *bss = SLIST_FIRST(head);
	struct bss_db *bss_tmp = NULL;

	while(bss != NULL) {
		bss_tmp = SLIST_NEXT(bss, bss_entry);
		debug(DEBUG_TRACE, "free bss_db(id=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(bss->bssid));
		free(bss);
		bss = bss_tmp;
	}
	SLIST_INIT(head);
}

int parse_ap_metrics_query_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;
	int i =0;
	struct bss_db *bss = NULL;

	temp_buf = buf;

	if((*temp_buf) == AP_METRICS_QUERY_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification)
	{
		//shift to tlv value field
		temp_buf += 2;

		/*bssid num*/
		temp_buf += 1;

		ctx->metric_entry.metrics_query_cnt = (length -1) / ETH_ALEN;
		for (i = 0; i < ctx->metric_entry.metrics_query_cnt; i++) {
			bss = (struct bss_db *)malloc(sizeof(struct bss_db));
			if (!bss) {
				debug(DEBUG_ERROR, "alloc struct bss_db fail\n");
				return -1;
			}
			memcpy(bss->bssid, temp_buf, ETH_ALEN);
			temp_buf += ETH_ALEN;
			SLIST_INSERT_HEAD(&(ctx->metric_entry.metrics_query_head), \
				bss, bss_entry);
		}
	}
	return (length+3);
}

int parse_ap_metrics_query_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int all_integrity = 0xff;

	temp_buf = buf;
	reset_stored_tlv(ctx);
#ifdef MAP_R2
	delete_exist_radio_identifier(ctx);
#endif // MAP_R2
	while(1) {
		/* One AP metric query TLV */
		if (*temp_buf == AP_METRICS_QUERY_TYPE) {
			integrity |= (1 << 0);
			if(ctx->Certification)
			{
				delete_ap_metrics_query_head(&ctx->metric_entry.metrics_query_head);
			}
			length = parse_ap_metrics_query_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error ap metrics query tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
#ifdef MAP_R2
		/* Zero or more AP Radio Identifier TLVs */
		else if (*temp_buf == AP_RADIO_IDENTIFIER_TYPE) {
			integrity |= (1 << 1);

			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);

			/* need save ap radio identifier for each radio */
			length = parse_ap_radio_identifier_tlv(temp_buf, ctx
#ifdef MAP_R2
			, 1
#endif // MAP_R2
			);

			temp_buf += length;
		}
#endif
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if ((integrity & all_integrity) == 0) {
		debug(DEBUG_ERROR, "incomplete ap metrics query message 0x%x 0x%x\n",
			integrity, all_integrity);
		return -1;
	}

	return 0;
}

int parse_associated_sta_link_metrics_query_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == STA_MAC_ADDRESS_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		memcpy(ctx->metric_entry.assoc_sta, temp_buf, ETH_ALEN);
	}
	return (length+3);
}

int parse_associated_sta_link_metrics_query_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;

	reset_stored_tlv(ctx);
	while(1) {
		if (*temp_buf == STA_MAC_ADDRESS_TYPE) {
			integrity |= 0x1;
			length = parse_associated_sta_link_metrics_query_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error associated sta link metrics query tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no sta mac address tlv\n");
		return -1;
	}

	return 0;
}

int parse_associated_sta_link_metrics_rsp_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == ASSOC_STA_LINK_METRICS_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	return (length+3);
}

int parse_error_code_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == ERROR_CODE_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if (length != 7) {
		return -1;
	}
	return (length+3);
}

int parse_associated_sta_link_metrics_rsp_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0, all_integrity = 0xff;
	temp_buf = buf;

	reset_stored_tlv(ctx);
	while(1) {
		/* One or more Associated STA Link Metrics TLVs */
		if (*temp_buf == ASSOC_STA_LINK_METRICS_TYPE) {
			integrity |= (1 << 0);
			length = parse_associated_sta_link_metrics_rsp_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error associated sta link metrics query tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		/* Zero or one Error Code TLV */
		else if (*temp_buf == ERROR_CODE_TYPE) {
			integrity |= (1 << 1);
			length = parse_error_code_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error error code tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
#ifdef MAP_R2
		/* One or more Associated STA Extended Link Metrics TLVs */
		else if (*temp_buf == ASSOCIATED_STA_EXTENDED_LINK_METRIC_TYPE) {
			integrity |= (1 << 2);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
#endif
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if ((integrity & all_integrity) == 0) {
		debug(DEBUG_ERROR, "incomplete associated sta link metrics rsp message 0x%x\n",
			integrity);
		return -1;
	}

	return 0;
}


int parse_unassociated_sta_link_metrics_query_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf, struct unlink_metrics_query **unlink)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0, query_len = 0;
	struct unlink_metrics_query *unlink_query = NULL;
	unsigned char opclass = 0, num_sta = 0, num_ch = 0;
	unsigned char channels[MAX_CH_NUM] ={0};

	temp_buf = buf;

	if((*temp_buf) == UNASSOC_STA_LINK_METRICS_QUERY_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = (*(unsigned short *)temp_buf);
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		opclass = *temp_buf++;
		num_ch = *temp_buf++;
		memcpy(channels, temp_buf, num_ch);
		temp_buf += num_ch;
		num_sta = *temp_buf++;

		query_len = sizeof(struct unlink_metrics_query) + num_sta * ETH_ALEN;
		debug(DEBUG_TRACE, "query_len=%d\n",query_len);
		unlink_query = (struct unlink_metrics_query *)malloc(query_len);
		if (!unlink_query) {
			debug(DEBUG_ERROR, "alloc struct unlink_metrics_query fail\n");
			return -1;
		}

		unlink_query->oper_class = opclass;
		unlink_query->ch_num = num_ch;
		if (unlink_query->ch_num == 0) {
			debug(DEBUG_ERROR, "invalid ch num(0)\n");
			free(unlink_query);
			return -1;
		}
		memcpy(unlink_query->ch_list, channels, num_ch);
		unlink_query->sta_num = num_sta;
		if (unlink_query->sta_num == 0) {
			debug(DEBUG_ERROR, "invalid sta num(0)\n");
			free(unlink_query);
			return -1;
		}
		memcpy(unlink_query->sta_list, temp_buf, unlink_query->sta_num * ETH_ALEN);

		*unlink = unlink_query;
	}
	return (length+3);
}

int parse_unassociated_sta_link_metrics_query_message(
	struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == UNASSOC_STA_LINK_METRICS_QUERY_TYPE) {
			integrity |= 0x1;
			length = parse_unassociated_sta_link_metrics_query_tlv(ctx, temp_buf, &ctx->metric_entry.unlink_query);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error unassociated sta link metrics query tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no unassicated sta link metrics query tlv\n");
		return -1;
	}

	return 0;
}

int delete_exist_metrics_info(struct p1905_managerd_ctx *ctx,
	unsigned char *bssid)
{
	 struct mrsp_db *mrsp = NULL;
	 struct esp_db *esp = NULL, *esp_tmp = NULL;

    SLIST_FOREACH(mrsp, &(ctx->metric_entry.metrics_rsp_head), mrsp_entry)
    {
        if(!memcmp(mrsp->bssid, bssid, ETH_ALEN))
        {
        	debug(DEBUG_TRACE, "esp_cnt=%d\n", mrsp->esp_cnt);
            if(!SLIST_EMPTY(&(mrsp->esp_head)))
            {
                esp = SLIST_FIRST(&(mrsp->esp_head));
                while(esp)
                {
                	debug(DEBUG_TRACE, "delete_exist struct esp_db\n");
					debug(DEBUG_TRACE, "ac=%d, format=%d ba_win_size=%d\n",
						esp->ac, esp->format, esp->ba_win_size);
					debug(DEBUG_TRACE, "e_air_time_fraction=%d, ppdu_dur_target=%d\n",
						esp->e_air_time_fraction, esp->ppdu_dur_target);

                    esp_tmp = SLIST_NEXT(esp, esp_entry);
                    SLIST_REMOVE(&(mrsp->esp_head), esp, esp_db, esp_entry);
                    free(esp);
                    esp = esp_tmp;
                }
            }
			debug(DEBUG_TRACE, "delete_exist struct mrsp_db\n");
			debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(mrsp->bssid));
			debug(DEBUG_TRACE, "ch_uti=%d, assoc_sta_cnt=%d\n", mrsp->ch_util,
				mrsp->assoc_sta_cnt);
            SLIST_REMOVE(&(ctx->metric_entry.metrics_rsp_head), mrsp,
                        mrsp_db, mrsp_entry);
            free(mrsp);
            break;
        }
    }

	return wapp_utils_success;
}

int insert_new_metrics_info(struct p1905_managerd_ctx *ctx,
	struct ap_metrics_info *minfo)
{
	struct mrsp_db *mrsp = NULL;
	struct esp_db *esp = NULL;
	struct esp_info *info = NULL;
	int i = 0;

	mrsp = (struct mrsp_db *)malloc(sizeof(struct mrsp_db));
	if (!mrsp) {
		debug(DEBUG_ERROR, "alloc struct mrsp_db fail\n");
		return wapp_utils_error;
	}
	memset(mrsp, 0, sizeof(struct mrsp_db));
	memcpy(mrsp->bssid, minfo->bssid, ETH_ALEN);
	mrsp->ch_util = minfo->ch_util;
	mrsp->assoc_sta_cnt = minfo->assoc_sta_cnt;
	mrsp->esp_cnt = minfo->valid_esp_count;
	SLIST_INIT(&(mrsp->esp_head));
	SLIST_INSERT_HEAD(&(ctx->metric_entry.metrics_rsp_head), mrsp, mrsp_entry);

	debug(DEBUG_TRACE, "insert struct mrsp_db\n");
	debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(mrsp->bssid));
	debug(DEBUG_TRACE, "ch_uti=%d, assoc_sta_cnt=%d, esp_cnt=%d\n", mrsp->ch_util,
				mrsp->assoc_sta_cnt, mrsp->esp_cnt);

	for(i = 0; i < mrsp->esp_cnt; i++) {
		info = &minfo->esp[i];
		esp = (struct esp_db *)malloc(sizeof(struct esp_db));
		if (!esp) {
			debug(DEBUG_TRACE, "alloc struct esp_db fail\n");
			return wapp_utils_error;
		}
		esp->ac = info->ac;
		esp->format = info->format;
		esp->ba_win_size = info->ba_win_size;
		esp->e_air_time_fraction = info->e_air_time_fraction;
		esp->ppdu_dur_target = info->ppdu_dur_target;
		SLIST_INSERT_HEAD(&(mrsp->esp_head), esp, esp_entry);

		debug(DEBUG_TRACE, "insert struct esp_db\n");
		debug(DEBUG_TRACE, "ac=%d, format=%d ba_win_size=%d\n",
			esp->ac, esp->format, esp->ba_win_size);
		debug(DEBUG_TRACE, "e_air_time_fraction=%d, ppdu_dur_target=%d\n",
			esp->e_air_time_fraction, esp->ppdu_dur_target);
	}

	return wapp_utils_success;

}

unsigned short append_ap_metrics_tlv(
        unsigned char *pkt, struct mrsp_db *mrsp)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;
	struct esp_db *esp = NULL;
	unsigned char ac[4] = {1,0,3,2};
	unsigned char i = 0;
	unsigned char *esp_pos = NULL;

    temp_buf = pkt;

    *temp_buf++ = AP_METRICS_TYPE;
	total_length += 1;

    temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, mrsp->bssid, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = mrsp->ch_util;
	total_length += 1;

	*(unsigned short *)(temp_buf) = cpu2be16(mrsp->assoc_sta_cnt);
	temp_buf += 2;
	total_length += 2;

	esp_pos = temp_buf;
	temp_buf += 1;
	total_length += 1;

	*esp_pos = 0;
	for (i = 0; i < 4; i++) {
		SLIST_FOREACH(esp, &(mrsp->esp_head), esp_entry)
	    {
	    	if (esp->ac == ac[i]) {
				*esp_pos |= 1 << (7 - i);
				*temp_buf++ = (esp->ba_win_size << 5) | (esp->format << 3) | esp->ac;
				*temp_buf++ = esp->e_air_time_fraction;
				*temp_buf++ = esp->ppdu_dur_target;
				total_length += 3;
				break;
	    	}
		}
	}

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

unsigned short append_sta_traffic_stats_tlv(unsigned char *pkt, struct stats_db *stats
#ifdef MAP_R2
, unsigned char unit
#endif
)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;
	unsigned int bytes_sent = 0;
	unsigned int bytes_received = 0;

    temp_buf = pkt;

    *temp_buf++ = ASSOC_STA_TRAFFIC_STATS_TYPE;
	total_length += 1;

    temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, stats->mac, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	/* BytesSent && BytesReceived Units specified by R2 AP Capability TLV */
	/* bytes, default, mapd guarantee this unit */
	bytes_sent = stats->bytes_sent;
	bytes_received = stats->bytes_received;
#ifdef MAP_R2
	/*BytesSent*/
	if (unit == 1) {
		/* kibibytes (KiB) */
		bytes_sent = stats->bytes_sent/1024;
		bytes_received = stats->bytes_received/1024;
	}
	else if (unit == 2) {
		/* mebibytes (MiB) */
		bytes_sent = stats->bytes_sent/1024/1024;
		bytes_received = stats->bytes_received/1024/1024;
	}
#endif

	*(unsigned int *)(temp_buf) = cpu2be32(bytes_sent);
	temp_buf += 4;
	total_length += 4;

	/*BytesReceived*/
	*(unsigned int *)(temp_buf) = cpu2be32(bytes_received);
	temp_buf += 4;
	total_length += 4;

	/*PacketsSent*/
	*(unsigned int *)(temp_buf) = cpu2be32(stats->packets_sent);
	temp_buf += 4;
	total_length += 4;

	/*PacketsReceived*/
	*(unsigned int *)(temp_buf) = cpu2be32(stats->packets_received);
	temp_buf += 4;
	total_length += 4;


	/*TxPacketsErrors*/
	*(unsigned int *)(temp_buf) = cpu2be32(stats->tx_packets_errors);
	temp_buf += 4;
	total_length += 4;

	/*RxPacketsErrors*/
	*(unsigned int *)(temp_buf) = cpu2be32(stats->rx_packets_errors);
	temp_buf += 4;
	total_length += 4;

	/*RetransmissionCount*/
	*(unsigned int *)(temp_buf) = cpu2be32(stats->retransmission_count);
	temp_buf += 4;
	total_length += 4;

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

unsigned short append_sta_link_metrics_tlv(
        unsigned char *pkt, struct metrics_db *metrics)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf++ = ASSOC_STA_LINK_METRICS_TYPE;
	total_length += 1;

    temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, metrics->mac, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	/*BSSID number*/
	*temp_buf++ = 1;
	total_length += 1;

	memcpy(temp_buf, metrics->bssid, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	/*TimeDelta*/
	*(unsigned int *)(temp_buf) = cpu2be32(metrics->time_delta);
	temp_buf += 4;
	total_length += 4;

	/*Estimated MAC Date Rate in Downlink*/
	*(unsigned int *)(temp_buf) = cpu2be32(metrics->erate_downlink);
	temp_buf += 4;
	total_length += 4;

	/*Estimated MAC Date Rate in Uplink*/
	*(unsigned int *)(temp_buf) = cpu2be32(metrics->erate_uplink);
	temp_buf += 4;
	total_length += 4;

	/*Measured Uplink RSSI*/
	*temp_buf++ = metrics->rssi_uplink & 0xff;
	total_length += 1;

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

int delete_exist_traffic_stats_info(struct p1905_managerd_ctx *ctx,
	unsigned char *identifier)
{
	struct traffic_stats_db *traffic_stats = NULL;
	struct stats_db *stats = NULL, *stats_tmp = NULL;

    SLIST_FOREACH(traffic_stats, &ctx->metric_entry.traffic_stats_head, traffic_stats_entry)
    {
        if(!memcmp(traffic_stats->identifier, identifier, ETH_ALEN)) {
        	debug(DEBUG_TRACE, "sta_cnt=%d\n", traffic_stats->sta_cnt);

            stats = SLIST_FIRST(&traffic_stats->stats_head);
            while(stats) {
            	debug(DEBUG_TRACE, "delete_exist struct stats_db\n");
				debug(DEBUG_TRACE, "sta mac(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(stats->mac));
				debug(DEBUG_TRACE, "bytes_sent=%d, bytes_received=%d"
					"packets_sent=%d, packets_received=%d"
					"tx_packets_errors=%d, rx_packets_errors=%d"
					"retransmission_count=%d\n",
					stats->bytes_sent, stats->bytes_received,
					stats->packets_sent, stats->packets_received,
					stats->tx_packets_errors, stats->rx_packets_errors,
					stats->retransmission_count);

                stats_tmp = SLIST_NEXT(stats, stats_entry);
                SLIST_REMOVE(&traffic_stats->stats_head, stats, stats_db, stats_entry);
                free(stats);
                stats = stats_tmp;
            }
			debug(DEBUG_TRACE, "delete_exist struct traffic_stats_db\n");
			debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(traffic_stats->identifier));
            SLIST_REMOVE(&ctx->metric_entry.traffic_stats_head, traffic_stats,
                        traffic_stats_db, traffic_stats_entry);
            free(traffic_stats);
            break;
        }
    }

	return wapp_utils_success;
}

int insert_new_traffic_stats_info(struct p1905_managerd_ctx *ctx,
	struct sta_traffic_stats *traffic_stats)
{
	struct traffic_stats_db *tstats = NULL;
	struct stats_db *stats = NULL;
	struct stat_info *info = NULL;
	int i = 0;

	tstats = (struct traffic_stats_db *)malloc(sizeof(struct traffic_stats_db));
	if (!tstats) {
		debug(DEBUG_ERROR, "alloc struct traffic_stats_db fail\n");
		return wapp_utils_error;
	}
	memset(tstats, 0, sizeof(struct traffic_stats_db));
	memcpy(tstats->identifier, traffic_stats->identifier, ETH_ALEN);
	tstats->sta_cnt= traffic_stats->sta_cnt;
	SLIST_INIT(&tstats->stats_head);
	SLIST_INSERT_HEAD(&(ctx->metric_entry.traffic_stats_head), tstats, traffic_stats_entry);

	debug(DEBUG_TRACE, "insert struct traffic_stats_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(tstats->identifier));
	debug(DEBUG_TRACE, "sta_cnt=%d\n", tstats->sta_cnt);

	for(i = 0; i < traffic_stats->sta_cnt; i++) {
		info = &traffic_stats->stats[i];
		stats = (struct stats_db *)malloc(sizeof(struct stats_db));
		if (!stats) {
			debug(DEBUG_ERROR, "alloc struct stats_db fail\n");
			return wapp_utils_error;
		}
		memcpy(stats->mac, info->mac, ETH_ALEN);
		stats->bytes_sent = info->bytes_sent;
		stats->bytes_received = info->bytes_received;
		stats->packets_sent = info->packets_sent;
		stats->packets_received = info->packets_received;
		stats->tx_packets_errors = info->tx_packets_errors;
		stats->rx_packets_errors = info->rx_packets_errors;
		stats->retransmission_count = info->retransmission_count;
		SLIST_INSERT_HEAD(&tstats->stats_head, stats, stats_entry);

		debug(DEBUG_TRACE, "insert struct stats_db\n");
		debug(DEBUG_TRACE, "sta mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(stats->mac));
		debug(DEBUG_TRACE, "bytes_sent=%d, bytes_received=%d packets_sent=%d"
			"packets_received=%d, tx_packets_errors=%d rx_packets_errors=%d"
			"retransmission_count=%d\n",
			stats->bytes_sent, stats->bytes_received, stats->packets_sent,
			stats->packets_received, stats->tx_packets_errors, stats->rx_packets_errors,
			stats->retransmission_count);
	}

	return wapp_utils_success;

}

int delete_exist_link_metrics_info(struct p1905_managerd_ctx *ctx,
	unsigned char *identifier)
{
	struct link_metrics_db *link_metrics = NULL;
	struct metrics_db *metrics = NULL, *metrics_tmp = NULL;

    SLIST_FOREACH(link_metrics, &ctx->metric_entry.link_metrics_head, link_metrics_entry)
    {
        if(!memcmp(link_metrics->identifier, identifier, ETH_ALEN)) {
        	debug(DEBUG_TRACE, "sta_cnt=%d\n", link_metrics->sta_cnt);

            metrics = SLIST_FIRST(&link_metrics->metrics_head);
            while(metrics) {
            	debug(DEBUG_TRACE, "delete_exist struct link_metrics_db\n");
				debug(DEBUG_TRACE, "sts mac(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(metrics->mac));
				debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(metrics->bssid));
				debug(DEBUG_TRACE, "time_delta=%d, erate_downlink=%d erate_uplink=%d, rssi_uplink=%d\n",
					metrics->time_delta, metrics->erate_downlink,
					metrics->erate_uplink, metrics->rssi_uplink);

                metrics_tmp = SLIST_NEXT(metrics, metrics_entry);
                SLIST_REMOVE(&link_metrics->metrics_head, metrics, metrics_db, metrics_entry);
                free(metrics);
                metrics = metrics_tmp;
            }
			debug(DEBUG_TRACE, "delete_exist struct link_metrics_db\n");
			debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(link_metrics->identifier));
            SLIST_REMOVE(&ctx->metric_entry.link_metrics_head, link_metrics,
                        link_metrics_db, link_metrics_entry);
            free(link_metrics);
            break;
        }
    }

	return wapp_utils_success;
}

int insert_new_link_metrics_info(struct p1905_managerd_ctx *ctx,
	struct sta_link_metrics *metrics_info)
{
	struct link_metrics_db *link_metrics_ctx = NULL;
	struct metrics_db *metrics_ctx = NULL;
	struct link_metrics *info = NULL;
	int i = 0;

	link_metrics_ctx = (struct link_metrics_db *)malloc(sizeof(struct link_metrics_db));
	if (!link_metrics_ctx) {
		debug(DEBUG_ERROR, "alloc struct link_metrics_db fail\n");
		return wapp_utils_error;
	}
	memset(link_metrics_ctx, 0, sizeof(struct link_metrics_db));
	memcpy(link_metrics_ctx->identifier, metrics_info->identifier, ETH_ALEN);
	link_metrics_ctx->sta_cnt= metrics_info->sta_cnt;
	SLIST_INIT(&link_metrics_ctx->metrics_head);
	SLIST_INSERT_HEAD(&ctx->metric_entry.link_metrics_head, link_metrics_ctx, link_metrics_entry);

	debug(DEBUG_TRACE, "insert struct link_metrics_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(link_metrics_ctx->identifier));
	debug(DEBUG_TRACE, "sta_cnt=%d\n", link_metrics_ctx->sta_cnt);

	for(i = 0; i < link_metrics_ctx->sta_cnt; i++) {
		info = &metrics_info->info[i];
		metrics_ctx = (struct metrics_db *)malloc(sizeof(struct metrics_db));
		if (!metrics_ctx) {
			debug(DEBUG_ERROR, "alloc struct metrics_db fail\n");
			return wapp_utils_error;
		}
		memcpy(metrics_ctx->mac, info->mac, ETH_ALEN);
		memcpy(metrics_ctx->bssid, info->bssid, ETH_ALEN);
		metrics_ctx->time_delta = info->time_delta;
		metrics_ctx->erate_downlink = info->erate_downlink;
		metrics_ctx->erate_uplink = info->erate_uplink;
		metrics_ctx->rssi_uplink = info->rssi_uplink;
		SLIST_INSERT_HEAD(&link_metrics_ctx->metrics_head, metrics_ctx, metrics_entry);

		debug(DEBUG_TRACE, "insert struct metrics_db\n");
		debug(DEBUG_TRACE, "sta mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(metrics_ctx->mac));
		debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(metrics_ctx->bssid));
		debug(DEBUG_TRACE, "time_delta=%d, erate_downlink=%d erate_uplink=%d rssi_uplink=%d\n",
			metrics_ctx->time_delta, metrics_ctx->erate_downlink,
			metrics_ctx->erate_uplink, metrics_ctx->rssi_uplink);
	}

	return wapp_utils_success;

}

int update_one_sta_link_metrics_info(struct p1905_managerd_ctx *ctx,
	struct link_metrics *metrics)
{
	struct metrics_db *metrics_sta = &ctx->metric_entry.assoc_sta_link_metrics;

	memset(metrics_sta, 0, sizeof(struct metrics_db));
	memcpy(metrics_sta->mac, metrics->mac, ETH_ALEN);
	memcpy(metrics_sta->bssid, metrics->bssid, ETH_ALEN);
	metrics_sta->time_delta = metrics->time_delta;
	metrics_sta->erate_downlink = metrics->erate_downlink;
	metrics_sta->erate_uplink = metrics->erate_uplink;
	metrics_sta->rssi_uplink = metrics->rssi_uplink;

	debug(DEBUG_TRACE, "insert struct link_metrics_db\n");
	debug(DEBUG_TRACE, "sta mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(metrics_sta->mac));
	debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(metrics_sta->bssid));
	debug(DEBUG_TRACE, "time_delta=%d, erate_downlink=%d erate_uplink=%d rssi_uplink=%d\n",
			metrics_sta->time_delta, metrics_sta->erate_downlink,
			metrics_sta->erate_uplink, metrics_sta->rssi_uplink);

	return wapp_utils_success;

}

unsigned short ap_metrics_query_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(AP_LINK_METRICS_QUERY);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}


unsigned short ap_metrics_response_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct mrsp_db *mrsp = NULL;
	struct traffic_stats_db *traffic_stats = NULL;
	struct stats_db *stats = NULL;
	struct link_metrics_db *link_metrics = NULL;
	struct metrics_db *metrics = NULL;

	msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
		SLIST_FOREACH(mrsp, &ctx->metric_entry.metrics_rsp_head, mrsp_entry)
	    {
	    	length = append_ap_metrics_tlv(tlv_temp_buf, mrsp);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}

		SLIST_FOREACH(traffic_stats, &ctx->metric_entry.traffic_stats_head, traffic_stats_entry)
		{
			SLIST_FOREACH(stats, &traffic_stats->stats_head, stats_entry)
			{
				length = append_sta_traffic_stats_tlv(tlv_temp_buf, stats
#ifdef MAP_R2
				, ctx->ap_cap_entry.ap_r2_cap.byte_counter_units
#endif
				);
				total_tlvs_length += length;
				tlv_temp_buf += length;
			}
		}

		SLIST_FOREACH(link_metrics, &ctx->metric_entry.link_metrics_head, link_metrics_entry)
	    {
	    	SLIST_FOREACH(metrics, &link_metrics->metrics_head, metrics_entry)
	    	{
	    		length = append_sta_link_metrics_tlv(tlv_temp_buf, metrics);
				total_tlvs_length += length;
				tlv_temp_buf += length;
	    	}
		}
	}
	else
	{
		/* one AP Metrics TLV  */
		if(ctx->send_tlv_len != 0)
		{
			length = append_send_tlv(tlv_temp_buf, ctx);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}

	}
	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(AP_LINK_METRICS_RESPONSE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short associated_sta_link_metrics_query_message(
	unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(ASSOC_STA_LINK_METRICS_QUERY);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short associated_sta_link_metrics_response_message(
	unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct metrics_db *metrics = &ctx->metric_entry.assoc_sta_link_metrics;

	msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
		length = append_sta_link_metrics_tlv(tlv_temp_buf, metrics);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
	else
	{
		if(ctx->send_tlv_len != 0)
		{
			length = append_send_tlv(tlv_temp_buf, ctx);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}
	}

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(ASSOC_STA_LINK_METRICS_RESPONSE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

int delete_exist_unlink_metrics_rsp(struct unlink_metrics_info *unlink_metrics)
{
	struct unlink_metrics_db *metrics = NULL, *metrics_tmp = NULL;

	unlink_metrics->oper_class = 0;
	unlink_metrics->sta_num = 0;
	metrics = SLIST_FIRST(&unlink_metrics->unlink_metrics_head);
	while(metrics != NULL)
	{
		metrics_tmp = SLIST_NEXT(metrics, unlink_metrics_entry);
		debug(DEBUG_TRACE, "sta mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(metrics->mac));
		debug(DEBUG_TRACE, "ch=%d, time_delta=%d, uplink_rssi=%d\n",
				metrics->ch, metrics->time_delta, metrics->uplink_rssi);
		free(metrics);
		metrics = metrics_tmp;
	}
	SLIST_INIT(&unlink_metrics->unlink_metrics_head);

	return wapp_utils_success;
}

int update_unlink_metrics_rsp(struct unlink_metrics_info *unlink_metrics_ctx,
	struct unlink_metrics_rsp *unlink_metrics)
{
	struct unlink_metrics_db *metrics = NULL;
	struct unlink_rsp_sta *info = NULL;
	int i = 0;

	delete_exist_unlink_metrics_rsp(unlink_metrics_ctx);

	unlink_metrics_ctx->oper_class = unlink_metrics->oper_class;
	unlink_metrics_ctx->sta_num = unlink_metrics->sta_num;
	debug(DEBUG_TRACE, "oper_class=%d, sta_num=%d\n",
		unlink_metrics_ctx->oper_class, unlink_metrics_ctx->sta_num);

	SLIST_INIT(&unlink_metrics_ctx->unlink_metrics_head);

	for (i = 0; i < unlink_metrics->sta_num; i++) {
		metrics = (struct unlink_metrics_db *)malloc(sizeof(struct unlink_metrics_db));
		if (!metrics) {
			debug(DEBUG_TRACE, "alloc struct unlink_metrics_db fail\n");
			return wapp_utils_error;
		}
		info = &unlink_metrics->info[i];
		memcpy(metrics->mac, info->mac, ETH_ALEN);
		metrics->ch = info->ch;
		metrics->time_delta= info->time_delta;
		metrics->uplink_rssi = info->uplink_rssi;
		SLIST_INSERT_HEAD(&unlink_metrics_ctx->unlink_metrics_head, metrics, unlink_metrics_entry);
		debug(DEBUG_TRACE, "sta mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(metrics->mac));
		debug(DEBUG_TRACE, "ch=%d, time_delta=%d, uplink_rssi=%d\n",
				metrics->ch, metrics->time_delta, metrics->uplink_rssi);
	}

	return wapp_utils_success;

}

unsigned short append_sta_unlink_metrics_response_tlv(
        unsigned char *pkt, struct unlink_metrics_info *metrics)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;
	struct unlink_metrics_db *unlink_metrics = NULL;

    temp_buf = pkt;

    *temp_buf++ = UNASSOC_STA_LINK_METRICS_RSP_TYPE;
	total_length += 1;

    temp_buf += 2;
	total_length += 2;

	/*Operating Class*/
	*temp_buf++ = metrics->oper_class;
	total_length += 1;

	/*STA number*/
	*temp_buf++ = metrics->sta_num;
	total_length += 1;

	SLIST_FOREACH(unlink_metrics, &metrics->unlink_metrics_head, unlink_metrics_entry)
    {
    	/*sta mac*/
    	memcpy(temp_buf, unlink_metrics->mac, ETH_ALEN);
		temp_buf += ETH_ALEN;
		total_length += ETH_ALEN;

		/*channel*/
		*temp_buf++ = unlink_metrics->ch;
		total_length += 1;

		/*TimeDelta*/
		*(unsigned int *)(pkt+1) = cpu2be32(unlink_metrics->time_delta);
		temp_buf += 4;
		total_length += 4;

		/*Measured Uplink RSSI*/
		*temp_buf++ = unlink_metrics->uplink_rssi & 0xff;
		total_length += 1;
	}

	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

unsigned short unassociated_sta_link_metrics_query_message(
	unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(UNASSOC_STA_LINK_METRICS_QUERY);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}


unsigned short unassociated_sta_link_metrics_response_message(
	unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;
	struct unlink_metrics_info *metrics = &ctx->metric_entry.unlink_info;

	msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
		length = append_sta_unlink_metrics_response_tlv(tlv_temp_buf, metrics);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
	else
	{
		if(ctx->send_tlv_len != 0)
		{
			length = append_send_tlv(tlv_temp_buf, ctx);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}
	}

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(UNASSOC_STA_LINK_METRICS_RESPONSE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short append_beacon_metrics_response_tlv(
        unsigned char *pkt, struct beacon_metrics_rsp *beacon_rsp)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;

    temp_buf = pkt;

    *temp_buf++ = BEACON_METRICS_RESPONSE_TYPE;
	total_length += 1;

    temp_buf += 2;
	total_length += 2;

	/*MAC address of the associated STA for which the Beacon report information is requested*/
	memcpy(temp_buf, beacon_rsp->sta_mac, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	/*reserved field*/
	*temp_buf++ = beacon_rsp->reserved;
	total_length += 1;

	/*Number of measurement report elements*/
	*temp_buf++ = beacon_rsp->bcn_rpt_num;
	total_length += 1;
	memcpy(temp_buf, beacon_rsp->rpt, beacon_rsp->rpt_len);
	total_length += beacon_rsp->rpt_len;


	/*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

unsigned short beacon_metrics_response_message(
	unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	if(ctx->Certification == 1)
	{
		length = append_beacon_metrics_response_tlv(tlv_temp_buf, ctx->metric_entry.bcn_rsp);
		total_tlvs_length += length;
		tlv_temp_buf += length;

		free(ctx->metric_entry.bcn_rsp);
		ctx->metric_entry.bcn_rsp = NULL;
	}
	else
	{
		if(ctx->send_tlv_len != 0)
		{
			length = append_send_tlv(tlv_temp_buf, ctx);
			total_tlvs_length += length;
			tlv_temp_buf += length;
		}
	}

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(BEACON_METRICS_RESPONSE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short beacon_metrics_query_message(
	unsigned char *buf, struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(BEACON_METRICS_QUERY);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

int parse_beacon_metrics_query_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf,
	struct beacon_metrics_query **beacon)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0, query_len = 0;
	struct beacon_metrics_query *bcn_query = NULL;
	unsigned char bssid[ETH_ALEN] = {0};
	unsigned char sta_mac[ETH_ALEN] = {0};
	unsigned char ssid[33] = {0};
	unsigned char opclass = 0, ch = 0, rpt_detail = 0;
	unsigned char ssid_len = 0, num_chrep = 0;
	unsigned char i = 0;
	struct ap_chn_rpt *chn_rpt = NULL;

	temp_buf = buf;

	if((*temp_buf) == BEACON_METRICS_QUERY_TYPE) {
		temp_buf++;
	} else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		memcpy(sta_mac, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;

		/*opclass*/
		opclass = *temp_buf++;
		/*channel number*/
		ch = *temp_buf++;
		/*bssid*/
		memcpy(bssid, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;
		/*Reporting Detail value*/
		rpt_detail = *temp_buf++;
		/*ssid len & ssid*/
		ssid_len = *temp_buf++;
		memcpy(ssid, temp_buf, ssid_len);
		temp_buf += ssid_len;
		/*Number of AP Channel Reports*/
		num_chrep = *temp_buf++;

		query_len = sizeof(struct beacon_metrics_query) +
			num_chrep * sizeof(struct ap_chn_rpt);
		bcn_query = (struct beacon_metrics_query *)malloc(query_len);
		if (!bcn_query) {
			debug(DEBUG_ERROR, "alloc struct beacon_metrics_query fail\n");
			return -1;
		}

		memset(bcn_query, 0, query_len);
		memcpy(bcn_query->sta_mac, sta_mac, ETH_ALEN);
		bcn_query->oper_class = opclass;
		bcn_query->ch = ch;
		memcpy(bcn_query->bssid, bssid, ETH_ALEN);
		bcn_query->rpt_detail_val = rpt_detail;
		bcn_query->ssid_len = ssid_len;
		memcpy(bcn_query->ssid, ssid, ssid_len);
		bcn_query->ap_ch_rpt_num = num_chrep;
		chn_rpt = bcn_query->rpt;
		for (i = 0; i < num_chrep; i++) {
			/*ap channel report info*/
			chn_rpt->ch_rpt_len = *temp_buf++;
			chn_rpt->oper_class = *temp_buf++;
			memcpy(chn_rpt->ch_list, temp_buf, chn_rpt->ch_rpt_len - 1);
			temp_buf += chn_rpt->ch_rpt_len - 1;
			chn_rpt++;
		}
		/*element IDs*/
		bcn_query->elemnt_num = *temp_buf++;
		if (bcn_query->elemnt_num > MAX_ELEMNT_NUM)
			bcn_query->elemnt_num = MAX_ELEMNT_NUM;
		memcpy(bcn_query->elemnt_list, temp_buf, bcn_query->elemnt_num);

		*beacon = bcn_query;
	}
	return (length+3);
}

int parse_beacon_metrics_query_message(
	struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == BEACON_METRICS_QUERY_TYPE) {
			integrity |= 0x1;
			length = parse_beacon_metrics_query_tlv(ctx, temp_buf, &ctx->metric_entry.bcn_query);
			if(length < 0) {
				debug(DEBUG_ERROR, "error beacon metrics query tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no beacon metrics query tlv\n");
		return -1;
	}

	return 0;
}

unsigned short append_backhaul_steer_response_tlv(
		unsigned char *pkt, struct backhaul_steer_rsp *steer_rsp)
{
	unsigned short total_length = 0;
	unsigned char *temp_buf;

	temp_buf = pkt;

	*temp_buf++ = BACKHAUL_STEERING_RESPONSE_TYPE;
	total_length += 1;

	*(unsigned short *)(temp_buf) = cpu2be16(BACKHAUL_STEERING_RESPONSE_LENGTH);
	temp_buf += 2;
	total_length += 2;

	memcpy(temp_buf, steer_rsp->backhaul_mac, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	memcpy(temp_buf, steer_rsp->target_bssid, ETH_ALEN);
	temp_buf += ETH_ALEN;
	total_length += ETH_ALEN;

	*temp_buf++ = steer_rsp->status;
	total_length += 1;

	return total_length;
}

unsigned short backhaul_steering_response_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	if (ctx->Certification == 1) {
		length = append_backhaul_steer_response_tlv(tlv_temp_buf, &ctx->bsteer_rsp);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	} else {
		length = append_send_tlv(tlv_temp_buf, ctx);
		total_tlvs_length += length;
		tlv_temp_buf += length;
	}
	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(BACKHAUL_STEERING_RESPONSE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

int parse_backhaul_steering_request_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == BACKHAUL_STEERING_REQUEST_TYPE) {
		temp_buf++;
	} else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
		temp_buf += 2;

		memcpy(ctx->bsteer_req.backhaul_mac, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;
		memcpy(ctx->bsteer_req.target_bssid, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;
		ctx->bsteer_req.oper_class = *temp_buf++;
		ctx->bsteer_req.channel = *temp_buf++;

		debug(DEBUG_TRACE, "backhaul_mac(%02x:%02x:%02x:%02x:%02x:%02x)"
			"target_bssid(%02x:%02x:%02x:%02x:%02x:%02x)"
			"operating class=%d, channel=%d\n",
			PRINT_MAC(ctx->bsteer_req.backhaul_mac),
			PRINT_MAC(ctx->bsteer_req.target_bssid),
			ctx->bsteer_req.oper_class,
			ctx->bsteer_req.channel);
	}
	return (length+3);
}

unsigned short backhaul_steering_request_message(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(BACKHAUL_STEERING_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

int parse_backhaul_steering_request_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;

	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == BACKHAUL_STEERING_REQUEST_TYPE) {
			integrity |= 0x1;
			length = parse_backhaul_steering_request_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error backhaul steering request tlv\n");
				return -1;
			}
			if (ctx->Certification == 0) {
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no backhaul steering request tlv\n");
		return -1;
	}

	return 0;
}

int parse_backhaul_steering_rsp_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf = NULL;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == BACKHAUL_STEERING_RESPONSE_TYPE) {
		temp_buf++;
	} else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	return (length+3);
}

int parse_backhaul_steering_rsp_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;

	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == BACKHAUL_STEERING_RESPONSE_TYPE) {
			integrity |= 0x1;
			length = parse_backhaul_steering_rsp_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error backhaul steering response tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == ERROR_CODE_TYPE) {
			length = parse_error_code_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error error code tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no backhaul steering response tlv\n");
		return -1;
	}

	return 0;
}

int dev_send_1905(struct p1905_managerd_ctx* ctx, struct _1905_cmdu_request* request)
{
	struct p1905_neighbor_info* dev_info = NULL;
	struct topology_response_db *tpgr_db = NULL;
	int i = 0, j = 0, ifidx = -1;
	int is_mal = 0;
	unsigned char al_multi_address[6] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x13 };
	unsigned char need_update_mid = 1;

	if(memcmp(al_multi_address, request->dest_al_mac, ETH_ALEN)) {
		for(i = 0; i < ctx->itf_number; i++) {
			if(!LIST_EMPTY(&(ctx->p1905_neighbor_dev[i].p1905nbr_head))) {
				LIST_FOREACH(dev_info, &(ctx->p1905_neighbor_dev[i].p1905nbr_head), p1905nbr_entry) {
					if(!memcmp(dev_info->al_mac_addr, request->dest_al_mac, ETH_ALEN)) {
						ifidx = i;
						break;
					}
				}
			}
		}

		if (ifidx == -1) {
			SLIST_FOREACH(tpgr_db, &(ctx->topology_entry.tprdb_head), tprdb_entry) {
				if(!memcmp(request->dest_al_mac, tpgr_db->al_mac_addr, ETH_ALEN)) {
				   ifidx = tpgr_db->recv_ifid;
				}
			}
		}
	} else {
		is_mal = 1;
		debug(DEBUG_TRACE, "send 1905 broadcast cmdu\n");
	}

	if(request->len > 0)
	{
		debug(DEBUG_TRACE, "DEV_SEND_1905 type=0x%04x, len=%d\n", request->type, request->len);
		ctx->mid++;
		if (is_mal == 1) {
			for(j = 0; j < ctx->itf_number; j++) {
	            insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
	            	e_dev_send_1905_request, ctx->mid, ctx->itf[j].if_name, need_update_mid);
#ifdef SUPPORT_CMDU_RELIABLE
				if (request->type == TOPOLOGY_NOTIFICATION ||
						request->type == AP_AUTOCONFIG_RENEW)
					cmdu_reliable_send(ctx, e_dev_send_1905_request, ctx->mid, j);
#endif
			}
		} else {
			debug(DEBUG_TRACE, "DEV_SEND_1905_REQUEST to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
				e_dev_send_1905_request, ctx->mid,
				(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
		}
		ctx->pcmdu_tx_buf = request->body;
		ctx->cmdu_tx_msg_type = request->type;
		ctx->cmdu_tx_buf_len = request->len;
		return 0;
	}

	switch(request->type)
	{
		/*this is for test=====start*/
		case TOPOLOGY_DISCOVERY:
			debug(DEBUG_TRACE, "send topology discovery data\n");
			ctx->mid++;
			for(j = 0; j < ctx->itf_number; j++) {
	            insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
	            	e_topology_discovery, ctx->mid, ctx->itf[j].if_name, need_update_mid);
			}
			break;
		case TOPOLOGY_NOTIFICATION:
			debug(DEBUG_TRACE, "send topology notification data\n");
			ctx->mid++;
			for(j = 0; j < ctx->itf_number; j++) {
	            insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
	            	e_topology_notification, ctx->mid, ctx->itf[j].if_name, need_update_mid);
			}
			break;
		/*this is for test=====end*/
		case TOPOLOGY_QUERY:
			 /* if a AP want to send unicast cmdu to STA, it needs to use interface mac
		   	   * instead of AL mac address
		  	   */
	    	debug(DEBUG_TRACE, "send topology query to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));
	        insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_topology_query, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case AP_CAPABILITY_QUERY:
			 /* if a AP want to send unicast cmdu to STA, it needs to use interface mac
			   * instead of AL mac address
			   */
			debug(DEBUG_TRACE, "send ap capability query to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_ap_capability_query, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CHANNLE_PREFERENCE_QUERY:
			debug(DEBUG_TRACE, "send channel preference query to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
				e_channel_preference_query, ++ctx->mid,
				(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CHANNLE_SELECTION_REQUEST:
			 /* if a AP want to send unicast cmdu to STA, it needs to use interface mac
		   	   * instead of AL mac address
		  	   */
	    	debug(DEBUG_TRACE, "send channel selection request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));
	        insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_channel_selection_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case COMBINED_INFRASTRUCTURE_METRICS:
			debug(DEBUG_TRACE, "send combined infrastructure metrics query event to mapd\n");
			_1905_notify_combined_infrastructure_metrics_query(ctx, ctx->p1905_al_mac_addr,
				request->dest_al_mac);

			break;
		default:
			debug(DEBUG_ERROR, "unknown type of message (0x%04x)\n", request->type);
			break;
	}
	return 0;
}


/*process event from 1905.1 library*/
int process_1905_request(struct p1905_managerd_ctx* ctx, struct _1905_cmdu_request* request)
{
	struct p1905_neighbor_info* dev_info = NULL;
	struct topology_response_db *tpgr_db = NULL;
	int i = 0, j = 0, ifidx = -1;
	unsigned char al_multi_address[6] = { 0x01, 0x80, 0xc2, 0x00, 0x00, 0x13 };
#ifdef MAP_R2
	unsigned char all_agent_mac[ETH_ALEN] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
#endif
	unsigned char need_update_mid = 0;

	if(memcmp(al_multi_address, request->dest_al_mac, ETH_ALEN)) {
		for(i = 0; i < ctx->itf_number; i++) {
			if(!LIST_EMPTY(&(ctx->p1905_neighbor_dev[i].p1905nbr_head))) {
				LIST_FOREACH(dev_info, &(ctx->p1905_neighbor_dev[i].p1905nbr_head), p1905nbr_entry) {
					if(!memcmp(dev_info->al_mac_addr, request->dest_al_mac, ETH_ALEN)) {
						ifidx = i;
						break;
					}
				}
			}
		}

		if (ifidx == -1) {
			SLIST_FOREACH(tpgr_db, &(ctx->topology_entry.tprdb_head), tprdb_entry) {
				if(!memcmp(request->dest_al_mac, tpgr_db->al_mac_addr, ETH_ALEN)) {
				   ifidx = tpgr_db->recv_ifid;
				}
			}
		}
	} else {
		debug(DEBUG_TRACE, "send 1905 broadcast cmdu\n");
	}


	/*below code for DEV_SEND_1905 length==0, so 1905 must build the content of 1905 cmdu*/
	debug(DEBUG_TRACE, "receive cmd(0x%04x) from library\n", request->type);
	switch(request->type)
	{
		case TOPOLOGY_DISCOVERY:
			ctx->mid++;
			for(j = 0; j < ctx->itf_number; j++) {
	            insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,\
	            	e_topology_discovery, ctx->mid, ctx->itf[j].if_name, need_update_mid);
			}
			break;
		case TOPOLOGY_NOTIFICATION:
			ctx->mid++;
			for(j = 0; j < ctx->itf_number; j++) {
	            insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
	            	e_topology_notification, ctx->mid, ctx->itf[j].if_name, need_update_mid);
#ifdef SUPPORT_CMDU_RELIABLE
				cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, j);
#endif
			}
			break;
		case TOPOLOGY_QUERY:
			 /* if a AP want to send unicast cmdu to STA, it needs to use interface mac
		   	   * instead of AL mac address
		  	   */
	    	debug(DEBUG_TRACE, "send topology query to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));
	        insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_topology_query, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case AP_CAPABILITY_QUERY:
			 /* if a AP want to send unicast cmdu to STA, it needs to use interface mac
			   * instead of AL mac address
			   */
			debug(DEBUG_TRACE, "send ap capability query to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_ap_capability_query, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CHANNLE_PREFERENCE_QUERY:
			debug(DEBUG_TRACE, "send channel preference query to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
				e_channel_preference_query, ++ctx->mid,
				(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case COMBINED_INFRASTRUCTURE_METRICS:
			debug(DEBUG_TRACE, "send combined infrastructure metrics message to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));
			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			need_update_mid = 1;
	        insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_combined_infrastructure_metrics, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);

			break;
		case LINK_METRICS_QUERY:
			debug(DEBUG_TRACE, "send link metrics query to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));
			memcpy(ctx->link_metric_query_para.target, request->body, ETH_ALEN);
			memcpy(&(ctx->link_metric_query_para.type), request->body + ETH_ALEN, sizeof(unsigned char));
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_link_metric_query, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case AP_AUTOCONFIG_SEARCH:
			debug(DEBUG_TRACE, "send ap autoconfig search to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(ctx->controller_search_state != controller_search_idle ||
				ctx->enrolle_state != no_ap_autoconfig)
			{
				debug(DEBUG_TRACE, "now current autoconfig is ongoing, drop this autoconfig search\n");
				break;
			}
			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			ctx->autoconfig_search_mid = ++ctx->mid;
			for(j = 0; j < ctx->itf_number; j++) {
				insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
					e_ap_autoconfiguration_search, ctx->autoconfig_search_mid,
					ctx->itf[j].if_name, need_update_mid);
#ifdef SUPPORT_CMDU_RELIABLE
				cmdu_reliable_send(ctx, e_ap_autoconfiguration_search,
					ctx->autoconfig_search_mid, j);
#endif
			}
			break;
		case AP_AUTOCONFIG_RENEW:
			debug(DEBUG_TRACE, "send ap autoconfig renew to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			ctx->mid++;
			for(j = 0; j < ctx->itf_number; j++) {
				insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
					e_ap_autoconfiguration_renew, ctx->mid, ctx->itf[j].if_name,
					need_update_mid);
#ifdef SUPPORT_CMDU_RELIABLE
				cmdu_reliable_send(ctx, e_ap_autoconfiguration_renew, ctx->mid, j);
#endif
			}
			break;
		case CLIENT_CAPABILITY_QUERY:
			{
				struct client_info* cinfo = NULL;
				debug(DEBUG_TRACE, "send client capability query to %02x:%02x:%02x:%02x:%02x:%02x\n",
				 	PRINT_MAC(request->dest_al_mac));
				if(ctx->revd_tlv_len != 0)
				{
					debug(DEBUG_TRACE, "a 1905 cmdu request exist, drop current one\n");
					break;
				}
				cinfo = (struct client_info*)ctx->send_tlv;
				memcpy(cinfo->bssid, request->body, ETH_ALEN);
				memcpy(cinfo->sta_mac, request->body + ETH_ALEN, ETH_ALEN);
				ctx->send_tlv_len = sizeof(struct client_info);
				insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
		                e_cli_capability_query, ++ctx->mid,
		                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			}
			break;
		case AP_LINK_METRICS_QUERY:
			{
				debug(DEBUG_TRACE, "send ap link metrics query to %02x:%02x:%02x:%02x:%02x:%02x\n",
				 	PRINT_MAC(request->dest_al_mac));

				if(fill_send_tlv(ctx, request->body, request->len) < 0)
				{
					break;
				}
				insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
		                e_ap_metrics_query, ++ctx->mid,
		                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			}
			break;
		case ASSOC_STA_LINK_METRICS_QUERY:
			debug(DEBUG_TRACE, "send assoc sta link metrics query to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_associated_sta_link_metrics_query, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case UNASSOC_STA_LINK_METRICS_QUERY:
			debug(DEBUG_TRACE, "send unassoc sta link metrics query to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_unassociated_sta_link_metrics_query, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case BEACON_METRICS_QUERY:
			debug(DEBUG_TRACE, "send beacon metrics query to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_beacon_metrics_query, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CLIENT_STEERING_REQUEST:
			debug(DEBUG_TRACE, "send client steering request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_client_steering_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CLIENT_ASSOC_CONTROL_REQUEST:
			debug(DEBUG_TRACE, "send client assoc control request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_client_association_control_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case HIGHER_LAYER_DATA_MESSAGE:
			debug(DEBUG_TRACE, "send higher layer data message to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				debug(DEBUG_TRACE, "a 1905 cmdu request exist, drop current one\n");
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_higher_layer_data, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case BACKHAUL_STEERING_REQUEST:
			debug(DEBUG_TRACE, "send backhaul steering request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_backhaul_steering_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CHANNLE_SELECTION_REQUEST:
			debug(DEBUG_TRACE, "send channel select request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_channel_selection_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case MAP_POLICY_CONFIG_REQUEST:
			debug(DEBUG_TRACE, "send map policy config request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_multi_ap_policy_config_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case VENDOR_SPECIFIC:
			debug(DEBUG_TRACE, "send VENDOR_SPECIFIC request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_vendor_specific, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;

#ifdef MAP_R2
		/*channel scan feature*/
		case CHANNEL_SCAN_REQUEST:
			debug(DEBUG_TRACE, "send channel scan request to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
	                e_channel_scan_request, ++ctx->mid,
	                (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
		case CHANNEL_SCAN_REPORT:
			debug(DEBUG_TRACE, "send channel scan report to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_channel_scan_report, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;

		case TUNNELED_MESSAGE:
			debug(DEBUG_TRACE, "send tunneled message to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_tunneled_message, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;

		case ASSOCIATION_STATUS_NOTIFICATION:
			if (0 == memcmp(all_agent_mac, request->dest_al_mac, ETH_ALEN)) {
				debug(DEBUG_OFF, "send assoc status notification to all map device\n");
			} else {
				debug(DEBUG_TRACE, "send assoc status notification to %02x:%02x:%02x:%02x:%02x:%02x\n",
					    PRINT_MAC(request->dest_al_mac));
			}
			if(fill_send_tlv(ctx, request->body, request->len) < 0)
				break;
			ctx->mid++;
			if (0 == memcmp(all_agent_mac, request->dest_al_mac, ETH_ALEN)) {
				for(j = 0; j < ctx->itf_number; j++) {
					insert_cmdu_txq(al_multi_address, ctx->p1905_al_mac_addr,
						e_association_status_notification, ctx->mid, ctx->itf[j].if_name, need_update_mid);
#ifdef SUPPORT_CMDU_RELIABLE
					cmdu_reliable_send(ctx, e_association_status_notification, ctx->mid, j);
#endif
				}
			} else {
				insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_association_status_notification, ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			}
			break;

	    case CAC_REQUEST:
		    debug(DEBUG_TRACE, "send CAC_REQUEST message to %02x:%02x:%02x:%02x:%02x:%02x\n",
			    PRINT_MAC(request->dest_al_mac));

		    if(fill_send_tlv(ctx, request->body, request->len) < 0)
		    {
			    break;
		    }
		    insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
				    e_cac_request, ++ctx->mid,
				    (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
		    break;

		case CAC_TERMINATION:
			debug(DEBUG_TRACE, "send CAC_TERMINATION message to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_cac_termination, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;

		case CLIENT_DISASSOCIATION_STATS:
			debug(DEBUG_TRACE, "send CLIENT_DISASSOCIATION_STATS message to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_client_disassociation_stats, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;

		case FAILED_ASSOCIATION_MESSAGE:
			debug(DEBUG_TRACE, "send FAILED_ASSOCIATION_MESSAGE message to %02x:%02x:%02x:%02x:%02x:%02x\n",
				PRINT_MAC(request->dest_al_mac));

			if(fill_send_tlv(ctx, request->body, request->len) < 0)
			{
				break;
			}
			insert_cmdu_txq(request->dest_al_mac, ctx->p1905_al_mac_addr,
					e_failed_association_message, ++ctx->mid,
					(ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, need_update_mid);
			break;
#endif // #ifdef MAP_R2
		default:
			debug(DEBUG_OFF, "unknown type of message\n");
			break;
	}
	return 0;
}

void update_radio_info(struct p1905_managerd_ctx* ctx, struct wps_get_config* info)
{
	int i = 0, j = 0, k = 0;
	unsigned char add_num = 0;

	for(i = 0; i < ctx->radio_number; i++)
	{
		if(!memcmp(ctx->rinfo[i].identifier, info->identifier, ETH_ALEN))
		{
			ctx->rinfo[i].band = info->band;
			ctx->rinfo[i].dev_type = info->dev_type;

			debug(DEBUG_OFF, "add new radio(%02x:%02x:%02x:%02x:%02x:%02x) band=%d, bssnum=%d\n",
				PRINT_MAC(info->identifier), info->band, info->num_of_inf);
			for(k = 0; k < info->num_of_inf; k++)
			{
				for(j = 0; j < ctx->rinfo[i].bss_number; j++)
				{

					if(!memcmp(ctx->rinfo[i].bss[j].ifname, info->inf_data[k].ifname,
							strlen((char*)ctx->rinfo[i].bss[j].ifname)))
					{
						break;
					}
				}
				if(j >= ctx->rinfo[i].bss_number)
				{
					/*new bss, should add new one*/
					memcpy(ctx->rinfo[i].bss[ctx->rinfo[i].bss_number + add_num].ifname, info->inf_data[k].ifname, IFNAMSIZ);
					memcpy(ctx->rinfo[i].bss[ctx->rinfo[i].bss_number + add_num].mac, info->inf_data[k].mac_addr, ETH_ALEN);
					ctx->rinfo[i].bss[ctx->rinfo[i].bss_number + add_num].config_status = 0;
					ctx->rinfo[i].bss[ctx->rinfo[i].bss_number + add_num].priority= 0;
					add_num++;
					debug(DEBUG_OFF, "readd new bss(%s) to existed radio(%02x:%02x:%02x:%02x:%02x:%02x)\n",
						info->inf_data[k].ifname, PRINT_MAC(info->inf_data[k].mac_addr));
				}
			}
			ctx->rinfo[i].bss_number += add_num;
			debug(DEBUG_OFF, "rbss_number(%d)\n", ctx->rinfo[i].bss_number);
			break;
		}
	}
	/*new radio, add new one*/
	if(i >= ctx->radio_number)
	{
		memcpy(ctx->rinfo[ctx->radio_number].identifier, info->identifier, ETH_ALEN);
		ctx->rinfo[ctx->radio_number].band = info->band;
		ctx->rinfo[ctx->radio_number].dev_type = info->dev_type;
		ctx->rinfo[ctx->radio_number].bss_number = info->num_of_inf;
		ctx->rinfo[ctx->radio_number].teared_down = 0;
		debug(DEBUG_OFF, "add new radio(%02x:%02x:%02x:%02x:%02x:%02x) band=%d, bssnum=%d\n",
			PRINT_MAC(info->identifier), info->band, info->num_of_inf);
		for(i = 0; i < info->num_of_inf; i++)
		{
			memcpy(ctx->rinfo[ctx->radio_number].bss[i].ifname, info->inf_data[i].ifname, IFNAMSIZ);
			memcpy(ctx->rinfo[ctx->radio_number].bss[i].mac, info->inf_data[i].mac_addr, ETH_ALEN);
			ctx->rinfo[ctx->radio_number].bss[i].config_status = 0;
			ctx->rinfo[i].bss[ctx->rinfo[i].bss_number + add_num].priority= 0;
			debug(DEBUG_OFF, "add new bss(%s) to above new radio\n", ctx->rinfo[ctx->radio_number].bss[i].ifname);
		}
		ctx->radio_number++;
	}

	/*set the priority to the bss that will be configured*/
	for (i = 0; i < ctx->radio_number; i++) {
		if (!os_memcmp(ctx->rinfo[i].identifier, info->identifier, ETH_ALEN)) {
			for (j = 0; j < ctx->rinfo[i].bss_number; j++) {
				for (k = 0; k < ctx->itf_number; k++) {
					/*if (os_strcmp((char*)ctx->rinfo[i].bss[j].ifname, (char*)ctx->itf[k].if_name) == 0) {*/
					if (os_memcmp((char*)ctx->rinfo[i].bss[j].mac, (char*)ctx->itf[k].mac_addr, ETH_ALEN) == 0) {
						ctx->rinfo[i].bss[j].priority = ctx->itf[k].config_priority;
					}
				}
			}
			break;
		}
	}
}

int set_radio_autoconf_prepare(struct p1905_managerd_ctx* ctx, unsigned int band, unsigned char set_value)
{
	int i = 0;

	for (i = 0; i < ctx->radio_number; i++) {
		if (ctx->rinfo[i].band == band) {
			ctx->rinfo[i].trrigerd_autoconf = set_value;
			debug(DEBUG_OFF, "radio(%02x:%02x:%02x:%02x:%02x:%02x) need %sto do autoconfig\n",
				PRINT_MAC(ctx->rinfo[i].identifier), set_value ? "" : "not ");
		}
	}

	return 0;
}

int set_radio_autoconf_trriger(struct p1905_managerd_ctx* ctx, unsigned char* radio_id, unsigned char set_value)
{
	int i = 0;
	unsigned char radio_index = 0;

	for(i = 0; i < ctx->radio_number; i++)
	{
		if(!memcmp(ctx->rinfo[i].identifier, radio_id, ETH_ALEN))
		{
			if(set_value== 1)
			{
				if(ctx->rinfo[i].teared_down == 1)
				{
					debug(DEBUG_OFF, "this radio has teared down, wait for WSC renew message from controller\n");
					return -2;
				}
				if(ctx->current_autoconfig_info.radio_index == i)
				{
					debug(DEBUG_OFF, "this radio autoconfig is ongoing, wait it done\n");
					return -1;
				}
				else if(ctx->current_autoconfig_info.radio_index != -1)
				{
					radio_index = ctx->current_autoconfig_info.radio_index;
					debug(DEBUG_OFF, "cancel the autoconfig of current radio(%02x:%02x:%02x:%02x:%02x:%02x)\n",
						PRINT_MAC(ctx->rinfo[radio_index].identifier));
					ctx->enrolle_state = no_ap_autoconfig;
					auto_configuration_done(ctx);
				}
			}
			ctx->rinfo[i].trrigerd_autoconf = set_value;
			debug(DEBUG_OFF, "radio(%02x:%02x:%02x:%02x:%02x:%02x) autoconfig %strrigered\n",
				PRINT_MAC(ctx->rinfo[i].identifier), set_value ? "" : "un");
			break;
		}
	}
	return 0;
}
int insert_agent_info(struct p1905_managerd_ctx *ctx, unsigned char *almac)
{
	unsigned char is_found = 0;
	struct agent_list_db *agent_info = NULL;

	SLIST_FOREACH(agent_info, &(ctx->agent_head), agent_entry)
	{
		if (!memcmp(agent_info->almac, almac, ETH_ALEN)) {
			is_found = 1;
			break;
		}
	}

	if (is_found) {
		debug(DEBUG_ERROR, "this agent is alraedy insert agent list\n");
		return 0;
	}

	agent_info = (struct agent_list_db *)malloc(sizeof(struct agent_list_db));
	if (!agent_info) {
		debug(DEBUG_ERROR, "alloc struct agent_list_db fail\n");
		return -1;
	}

	debug(DEBUG_TRACE, "insert new agent(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(almac));
	memset(agent_info, 0, sizeof(struct agent_list_db));
	memcpy(agent_info->almac, almac, ETH_ALEN);
	SLIST_INIT(&(agent_info->ch_prefer_head));
	SLIST_INIT(&(agent_info->oper_restrict_head));
	SLIST_INIT(&(agent_info->metrics_rsp_head));
	SLIST_INIT(&(agent_info->tx_metrics_head));
	SLIST_INIT(&(agent_info->rx_metrics_head));
	SLIST_INSERT_HEAD(&(ctx->agent_head), agent_info, agent_entry);

	return 0;

}

void find_agent_info(struct p1905_managerd_ctx *ctx,
	unsigned char *almac, struct agent_list_db **agent)
{
	struct agent_list_db *agent_info = NULL;

	SLIST_FOREACH(agent_info, &(ctx->agent_head), agent_entry)
	{
		if (!memcmp(agent_info->almac, almac, ETH_ALEN)) {
			*agent = agent_info;
			break;
		}
	}

}
int delete_agent_ch_prefer_info(struct p1905_managerd_ctx *ctx,
	struct agent_list_db *agent)
{
	struct ch_prefer_db *chcap = NULL, *chcap_tmp = NULL;
	struct prefer_info_db *prefer = NULL, *prefer_tmp = NULL;
	struct oper_restrict_db *restriction = NULL, *restriction_tmp = NULL;;
	struct restrict_db *resdb = NULL, *resdb_tmp = NULL;
	int i = 0;

	debug(DEBUG_TRACE, "agent(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(agent->almac));
	debug(DEBUG_TRACE, "########delete channel preference########\n");
	chcap = SLIST_FIRST(&(agent->ch_prefer_head));
	while (chcap != NULL) {
		chcap_tmp = SLIST_NEXT(chcap, ch_prefer_entry);
  	  	prefer = SLIST_FIRST(&(chcap->prefer_info_head));
    	while(prefer) {
    		debug(DEBUG_TRACE, "delete_exist struct prefer_info_db\n");
			debug(DEBUG_TRACE, "opclass=%d, ch_num=%d perference=%d, reason=%d\n",
				prefer->op_class, prefer->ch_num, prefer->perference, prefer->reason);
			debug(DEBUG_TRACE, "ch_list: ");
			for (i = 0; i < prefer->ch_num; i++) {
				debugbyte(DEBUG_TRACE, "%d ", prefer->ch_list[i]);
			}
			debugbyte(DEBUG_TRACE, "\n");
        	prefer_tmp = SLIST_NEXT(prefer, prefer_info_entry);
       	 	free(prefer);
        	prefer = prefer_tmp;
    	}
		debug(DEBUG_TRACE, "delete_exist struct ch_prefer_db\n");
		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(chcap->identifier));
		debug(DEBUG_TRACE, "tx_power_limit=%d, oper_bss_num:%d\n",
			chcap->tx_power_limit, chcap->op_class_num);
		free(chcap);
		chcap = chcap_tmp;
	}
	SLIST_INIT(&(agent->ch_prefer_head));

	debug(DEBUG_TRACE, "########delete radio operating restriction########\n");
	restriction = SLIST_FIRST(&(agent->oper_restrict_head));
	while (restriction != NULL) {
		restriction_tmp = SLIST_NEXT(restriction, oper_restrict_entry);
  	  	resdb = SLIST_FIRST(&(restriction->restrict_head));
    	while(resdb) {
    		debug(DEBUG_TRACE, "delete_exist struct restrict_db\n");
			debug(DEBUG_TRACE, "opclass=%d ch_num=%d\n",
				resdb->op_class, resdb->ch_num);
			debug(DEBUG_TRACE, "ch_list: ");
			for (i = 0; i < resdb->ch_num; i++) {
				debugbyte(DEBUG_TRACE, "%d ", resdb->ch_list[i]);
			}
			debugbyte(DEBUG_TRACE, "\n");
			debug(DEBUG_TRACE, "min_fre_sep_list: ");
			for (i = 0; i < resdb->ch_num; i++) {
				debugbyte(DEBUG_TRACE, "%d ", resdb->min_fre_sep[i]);
			}
			debugbyte(DEBUG_TRACE, "\n");
        	resdb_tmp = SLIST_NEXT(resdb, restrict_entry);
       	 	free(resdb);
        	resdb = resdb_tmp;
    	}
		debug(DEBUG_TRACE, "delete_exist struct oper_restrict_db\n");
		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(restriction->identifier));
		debug(DEBUG_TRACE, "oper_bss_num:%d\n", restriction->op_class_num);
		free(restriction);
		restriction = restriction_tmp;
	}
	SLIST_INIT(&(agent->oper_restrict_head));

	return 0;
}

int delete_all_radio_oper_restrict_info(struct list_head_oper_restrict *oper_restrict_head)
{
	struct oper_restrict_db *restriction = NULL, *restriction_tmp = NULL;
	struct restrict_db *resdb = NULL, *resdb_tmp = NULL;
	int i = 0;

	restriction = SLIST_FIRST(oper_restrict_head);
	while (restriction != NULL) {
		restriction_tmp = SLIST_NEXT(restriction, oper_restrict_entry);
		resdb = SLIST_FIRST(&(restriction->restrict_head));
		while(resdb)
		{
			debug(DEBUG_TRACE, "delete_exist struct restrict_db\n");
			debug(DEBUG_TRACE, "opclass=%d ch_num=%d\n", resdb->op_class, resdb->ch_num);
			debug(DEBUG_TRACE, "ch_list: ");
			for (i = 0; i < resdb->ch_num; i++) {
				debugbyte(DEBUG_TRACE, "%d ", resdb->ch_list[i]);
			}
			debugbyte(DEBUG_TRACE, "\n");
			debug(DEBUG_TRACE, "min_fre_sep_list: ");
			for (i = 0; i < resdb->ch_num; i++) {
				debugbyte(DEBUG_TRACE, "%d ", resdb->min_fre_sep[i]);
			}
			debugbyte(DEBUG_TRACE, "\n");
			resdb_tmp = SLIST_NEXT(resdb, restrict_entry);
			free(resdb);
			resdb = resdb_tmp;
		}
		debug(DEBUG_TRACE, "delete_exist struct oper_restrict_db\n");
		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(restriction->identifier));
		debug(DEBUG_TRACE, "oper_bss_num:%d\n", restriction->op_class_num);
		free(restriction);
		restriction = restriction_tmp;
	}
	SLIST_INIT(oper_restrict_head);

	return wapp_utils_success;
}


int insert_new_radio_oper_restrict(
	struct list_head_oper_restrict *oper_restrict_head, unsigned char *buf, unsigned short len)
{
	struct oper_restrict_db *restriction = NULL;
	struct restrict_db *resdb = NULL;
	 unsigned char *pos = buf;
	 unsigned short check_len = 0, prefer_len = 0;
	 unsigned char i = 0, j = 0, op_class = 0, ch_num = 0;

	if (len < 7) {
		debug(DEBUG_ERROR, "length error less than 7\n");
		return wapp_utils_error;
	}

	restriction = (struct oper_restrict_db *)malloc(sizeof(struct oper_restrict_db));
	if (!restriction) {
		debug(DEBUG_ERROR, "alloc struct oper_restrict_db fail\n");
		return wapp_utils_error;
	}
	memset(restriction, 0, sizeof(struct oper_restrict_db));
	memcpy(restriction->identifier, pos, ETH_ALEN);
	pos += ETH_ALEN;
	check_len += ETH_ALEN;
	restriction->op_class_num = *pos++;
	check_len += 1;
	SLIST_INIT(&(restriction->restrict_head));
	SLIST_INSERT_HEAD(oper_restrict_head, restriction, oper_restrict_entry);

	debug(DEBUG_TRACE, "insert struct oper_restrict_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x) op_class_num=%d\n",
		PRINT_MAC(restriction->identifier), restriction->op_class_num);

	for(i = 0; i < restriction->op_class_num; i++) {
		op_class = *pos++;
		check_len += 1;
		ch_num = *pos++;
		check_len += 1;
		resdb = (struct restrict_db *)malloc(sizeof(struct restrict_db));
		if (!resdb) {
			debug(DEBUG_ERROR, "alloc struct restrict_db fail\n");
			return wapp_utils_error;
		}
		memset(resdb, 0 , prefer_len);
		resdb->op_class = op_class;
		resdb->ch_num = ch_num;
		for (j = 0; j < resdb->ch_num; j++) {
			resdb->ch_list[j] = *pos++;
			resdb->min_fre_sep[j] = *pos++;
			check_len += 2;
		}
		SLIST_INSERT_HEAD(&restriction->restrict_head, resdb, restrict_entry);
		debug(DEBUG_TRACE, "insert struct restrict_db\n");
		debug(DEBUG_TRACE, "opclass=%d, ch_num=%d\n", resdb->op_class, resdb->ch_num);
		debug(DEBUG_TRACE, "ch_list: ");
		for (j = 0; j < resdb->ch_num; j++) {
			debugbyte(DEBUG_TRACE, "%d ", resdb->ch_list[j]);
		}
		debugbyte(DEBUG_TRACE, "\n");
		debug(DEBUG_TRACE, "min_fre_sep_list: ");
		for (j = 0; j < resdb->ch_num; j++) {
			debugbyte(DEBUG_TRACE, "%d ", resdb->min_fre_sep[j]);
		}
		debugbyte(DEBUG_TRACE, "\n");
	}

	if (len != check_len) {
		debug(DEBUG_ERROR, "length mismatch len(%d) != check_len(%d)\n",
			len, check_len);
		delete_all_radio_oper_restrict_info(oper_restrict_head);
		return wapp_utils_error;
	}

	return wapp_utils_success;

}

int parse_radio_operation_restriction_tlv(unsigned char *buf,
	struct list_head_oper_restrict *oper_restrict_head)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == RADIO_OPERATION_RESTRICTION_TYPE) {
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf+=2;

	/*insert new channel preference info*/
	if (0 > insert_new_radio_oper_restrict(oper_restrict_head, temp_buf, length)) {
		debug(DEBUG_ERROR, "insert_new_radio_oper_restrict fail\n");
		return -1;
	}

	return (length+3);
}


#ifdef MAP_R2

int parse_cac_completion_report_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == CAC_COMPLETION_REPORT_TYPE) {
	    debug(DEBUG_WARN, "cac completion report type:%02x \n", *temp_buf);
	    temp_buf++;
	}
	else {
	    return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	temp_buf += 2;

	debug(DEBUG_WARN, "cac completion report type len:%d \n", length);

	return (length+3);
}



int parse_cac_status_report_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == CAC_STATUS_REPORT_TYPE) {
	    debug(DEBUG_WARN, "cac status report type:%02x \n", *temp_buf);
	    temp_buf++;
	}
	else {
	    return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	temp_buf += 2;

	debug(DEBUG_WARN, "cac status report type len:%d \n", length);

	return (length+3);
}

#endif // #ifdef MAP_R2

int parse_channel_preference_report_message(struct p1905_managerd_ctx *ctx, struct agent_list_db *agent,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;

	temp_buf = buf;
	reset_stored_tlv(ctx);
	while(1) {
		if (*temp_buf == CH_PREFERENCE_TYPE) {
			if (ctx->Certification == 1)
			{
				length = parse_channel_preference_tlv(ctx, temp_buf,
					(struct list_head_ch_prefer *)&agent->ch_prefer_head);
				if(length < 0)
				{
					debug(DEBUG_ERROR, "error channel preference tlv\n");
					return -1;
				}
			}
			else if(ctx->Certification == 0)
			{
				length = get_cmdu_tlv_length(temp_buf);
				if(length < 0)
				{
					debug(DEBUG_ERROR, "error channel preference tlv\n");
					return -1;
				}
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == RADIO_OPERATION_RESTRICTION_TYPE) {
			if (ctx->Certification == 1)
			{
				length = parse_radio_operation_restriction_tlv(temp_buf,
					(struct list_head_oper_restrict *)&agent->oper_restrict_head);
				if(length < 0)
				{
					debug(DEBUG_ERROR, "error operation restriction tlv\n");
					return -1;
				}
			}
			else if(ctx->Certification == 0)
			{
				length = get_cmdu_tlv_length(temp_buf);
				if(length < 0)
				{
					debug(DEBUG_ERROR, "error operation restriction tlv\n");
					return -1;
				}
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
#ifdef MAP_R2
		else if (*temp_buf == CAC_COMPLETION_REPORT_TYPE) {
			length = parse_cac_completion_report_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error csc completion report type tlv\n");
				return -1;
			}

			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == CAC_STATUS_REPORT_TYPE) {
			length = parse_cac_status_report_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error csc status report type tlv\n");
				return -1;
			}

			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
#endif
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}

	return 0;
}

int delete_agent_ap_metrics_info(struct list_head_metrics_rsp_agent *metrics_head)
{
	 struct mrsp_db *mrsp = NULL, *mrsp_tmp = NULL;
	 struct esp_db *esp = NULL, *esp_tmp = NULL;

	mrsp = SLIST_FIRST(metrics_head);
	while (mrsp != NULL) {
		mrsp_tmp = SLIST_NEXT(mrsp, mrsp_entry);
        esp = SLIST_FIRST(&mrsp->esp_head);
        while(esp) {
        	debug(DEBUG_TRACE, "delete_exist struct esp_db\n");
			debug(DEBUG_TRACE, "ac=%d, format=%d ba_win_size=%d, e_air_time_fraction=%d, ppdu_dur_target=%d\n",
				esp->ac, esp->format, esp->ba_win_size, esp->e_air_time_fraction, esp->ppdu_dur_target);
            esp_tmp = SLIST_NEXT(esp, esp_entry);
            free(esp);
            esp = esp_tmp;
        }
		debug(DEBUG_TRACE, "delete_exist struct mrsp_db\n");
		debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(mrsp->bssid));
		debug(DEBUG_TRACE, "ch_util=%d, assoc_sta_cnt:%d\n",
			mrsp->ch_util, mrsp->assoc_sta_cnt);
		free(mrsp);
		mrsp = mrsp_tmp;
    }
	SLIST_INIT(metrics_head);

	return wapp_utils_success;
}


int insert_new_ap_metrics_info(
	struct list_head_metrics_rsp_agent *metrics_head, unsigned char *buf, unsigned short len)
{
	 unsigned char *pos = buf;
	 unsigned short check_len = 0;
	 struct mrsp_db *mrsp = NULL;
	 struct esp_db *esp = NULL;

	if (len != 13 && len != 22 && len != 16 && len != 19) {
		debug(DEBUG_ERROR, "length error(%d)\n", len);
		return wapp_utils_error;
	}

    SLIST_FOREACH(mrsp, metrics_head, mrsp_entry)
    {
        if(!memcmp(mrsp->bssid, buf, ETH_ALEN)) {
			debug(DEBUG_TRACE, "bss(%02x:%02x:%02x:%02x:%02x:%02x) found\n", PRINT_MAC(mrsp->bssid));
            break;
        }
    }

	if (!mrsp) {
		mrsp = (struct mrsp_db *)malloc(sizeof(*mrsp));
		if (!mrsp) {
			debug(DEBUG_ERROR, "alloc struct mrsp_db fail\n");
			return wapp_utils_error;
		}
		memset(mrsp, 0, sizeof(*mrsp));
		memcpy(mrsp->bssid, pos, ETH_ALEN);
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		mrsp->ch_util = *pos++;
		check_len += 1;
		mrsp->assoc_sta_cnt = *(unsigned short *)pos;
		mrsp->assoc_sta_cnt = be2cpu16(mrsp->assoc_sta_cnt);
		pos += 2;
		check_len += 2;
		/*skip esp indicator*/
		pos += 1;
		check_len += 1;
		SLIST_INIT(&mrsp->esp_head);
		SLIST_INSERT_HEAD(metrics_head, mrsp, mrsp_entry);

		debug(DEBUG_TRACE, "insert struct mrsp_db\n");
		debug(DEBUG_TRACE, "bssid(%02x:%02x:%02x:%02x:%02x:%02x) ch_util=%d assoc_sta_cnt=%d\n",
			PRINT_MAC(mrsp->bssid), mrsp->ch_util, mrsp->assoc_sta_cnt);
		while(check_len < len) {
			esp = (struct esp_db *)malloc(sizeof(struct esp_db));
			if (!esp) {
				debug(DEBUG_ERROR, "alloc struct esp_db fail\n");
				return wapp_utils_error;
			}
			memset(esp, 0 , sizeof(struct esp_db));
			esp->ac = (*pos) & 0x03;
			esp->format = (*pos) & 0x18;
			esp->ba_win_size = (*pos) & 0xe0;
			esp->e_air_time_fraction = *(pos + 1);
			esp->ppdu_dur_target = *(pos + 2);
			SLIST_INSERT_HEAD(&mrsp->esp_head, esp, esp_entry);
			pos += 3;
			check_len += 3;
			debug(DEBUG_TRACE, "insert struct esp_db\n");
			debug(DEBUG_TRACE, "ac=%d, format=%d ba_win_size=%d, e_air_time_fraction=%d, ",
				esp->ac, esp->format, esp->ba_win_size, esp->e_air_time_fraction);
			debug(DEBUG_TRACE, "ppdu_dur_target=%d\n", esp->ppdu_dur_target);
		}
	}else {
		debug(DEBUG_ERROR, "bssid exist\n");
		return wapp_utils_error;
	}

	return wapp_utils_success;

}

int parse_ap_metrics_tlv(struct p1905_managerd_ctx* ctx, unsigned char *buf,
	struct list_head_metrics_rsp_agent *metrics_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == AP_METRICS_TYPE) {
        temp_buf++;
    } else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if(ctx->Certification == 1)
    {
	    //shift to tlv value field
	    temp_buf += 2;
		/*insert new channel preference info*/
		if (0 > insert_new_ap_metrics_info(metrics_head, temp_buf, length)) {
			debug(DEBUG_ERROR, "insert_new_ap_metrics_info fail\n");
			return -1;
		}
	}

    return (length+3);
}

int parse_assoc_sta_traffic_stats_tlv(struct p1905_managerd_ctx* ctx, unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == ASSOC_STA_TRAFFIC_STATS_TYPE) {
        temp_buf++;
    } else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_assoc_sta_link_metrics_tlv(struct p1905_managerd_ctx* ctx, unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == ASSOC_STA_LINK_METRICS_TYPE) {
        temp_buf++;
    } else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_ap_metrics_response_message(struct p1905_managerd_ctx *ctx, struct agent_list_db *agent,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned int all_integrity = 0xff;

	temp_buf = buf;
	reset_stored_tlv(ctx);
	while(1) {
		/* One or more AP mMetrics TLVs */
		if (*temp_buf == AP_METRICS_TYPE) {
			integrity |= (1 << 0);
			length = parse_ap_metrics_tlv(ctx, temp_buf, &agent->metrics_rsp_head);
			if(length < 0) {
				debug(DEBUG_ERROR, "error ap metrics tlv\n");
				return -1;
			}
			if (ctx->Certification == 0) {
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		/* Zero or more Associated STA Traffic Stats TLVs */
		else if (*temp_buf == ASSOC_STA_TRAFFIC_STATS_TYPE) {
			integrity |= (1 << 1);
			length = parse_assoc_sta_traffic_stats_tlv(ctx, temp_buf);
			if(length < 0) {
				debug(DEBUG_ERROR, "error assoc sta traffic stats tlv\n");
				return -1;
			}
			if (ctx->Certification == 0) {
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		/* Zero or more Associated STA Link Metrics TLVs */
		else if (*temp_buf == ASSOC_STA_LINK_METRICS_TYPE) {
			integrity |= (1 << 2);
			length = parse_assoc_sta_link_metrics_tlv(ctx, temp_buf);
			if(length < 0) {
				debug(DEBUG_ERROR, "error assoc sta link metrics tlv\n");
				return -1;
			}
			if (ctx->Certification == 0) {
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
#ifdef MAP_R2
		/* Zero or more Radio Metrics TLVs */
		else if (*temp_buf == RADIO_METRIC_TYPE) {
			integrity |= (1 << 3);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* One or more AP Extended Metrics TLVs */
		else if (*temp_buf == AP_EXTENDED_METRIC_TYPE) {
			integrity |= (1 << 4);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* Zero or more Associated STA Extended Link Metrics TLVs */
		else if (*temp_buf == ASSOCIATED_STA_EXTENDED_LINK_METRIC_TYPE) {
			integrity |= (1 << 5);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
#endif
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if ((integrity & all_integrity) == 0) {
		debug(DEBUG_ERROR, "incomplete ap metrics response message 0x%x 0x%x\n",
			integrity, all_integrity);
		return -1;
	}

	return 0;
}

int delete_agent_tx_link_metrics_info(struct list_head_tx_metrics_agent *tx_metrics_head)
{
	 struct tx_link_metric_db *tx_link_metric = NULL, *tx_link_metric_tmp = NULL;
	 struct tx_metric_db *tx_metric = NULL, *tx_metric_tmp = NULL;

	tx_link_metric = SLIST_FIRST(tx_metrics_head);
	while (tx_link_metric != NULL) {
		tx_link_metric_tmp = SLIST_NEXT(tx_link_metric, tx_link_metric_entry);
        tx_metric = SLIST_FIRST(&tx_link_metric->tx_metric_head);
        while(tx_metric) {
        	debug(DEBUG_TRACE, "delete_exist struct tx_metric_db\n");
			debug(DEBUG_TRACE, "own inf(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(tx_metric->mac));
			debug(DEBUG_TRACE, "neighbor connect inf(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(tx_metric->nmac));
			debug(DEBUG_TRACE, "intf_type=%d, bridge_flag=%d, error_packet=%d, tx_packets=%d, ",
				tx_metric->intf_type, tx_metric->bridge_flag, tx_metric->error_packet,
				tx_metric->tx_packets);
			debug(DEBUG_TRACE, "mac_tpcap=%d, linkavl=%d phyrate=%d\n",
				tx_metric->mac_tpcap, tx_metric->linkavl, tx_metric->phyrate);
            tx_metric_tmp = SLIST_NEXT(tx_metric, tx_metric_entry);
            free(tx_metric);
            tx_metric = tx_metric_tmp;
        }
		debug(DEBUG_TRACE, "delete_exist struct tx_link_metric_db\n");
		debug(DEBUG_TRACE, "own almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(tx_link_metric->almac));
		debug(DEBUG_TRACE, "neighbor almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(tx_link_metric->nalmac));
		tx_link_metric = tx_link_metric_tmp;
    }
	SLIST_INIT(tx_metrics_head);

	return wapp_utils_success;
}

int delete_agent_rx_link_metrics_info(struct list_head_rx_metrics_agent *rx_metrics_head)
{
	 struct rx_link_metric_db *rx_link_metric = NULL, *rx_link_metric_tmp = NULL;
	 struct rx_metric_db *rx_metric = NULL, *rx_metric_tmp = NULL;

	rx_link_metric = SLIST_FIRST(rx_metrics_head);
	while (rx_link_metric != NULL) {
		rx_link_metric_tmp = SLIST_NEXT(rx_link_metric, rx_link_metric_entry);
        rx_metric = SLIST_FIRST(&rx_link_metric->rx_metric_head);
        while(rx_metric) {
        	debug(DEBUG_TRACE, "delete_exist struct rx_metric_db\n");
			debug(DEBUG_TRACE, "own inf(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(rx_metric->mac));
			debug(DEBUG_TRACE, "neighbor connect inf(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(rx_metric->nmac));
			debug(DEBUG_TRACE, "intf_type=%d, error_packet=%d, rx_packets=%d, rssi=%d\n",
				rx_metric->intf_type, rx_metric->error_packet, rx_metric->rx_packets,
				rx_metric->rssi);
            rx_metric_tmp = SLIST_NEXT(rx_metric, rx_metric_entry);
            free(rx_metric);
            rx_metric = rx_metric_tmp;
        }
		debug(DEBUG_TRACE, "delete_exist struct rx_link_metric_db\n");
		debug(DEBUG_TRACE, "own almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(rx_link_metric->almac));
		debug(DEBUG_TRACE, "neighbor almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(rx_link_metric->nalmac));
		rx_link_metric = rx_link_metric_tmp;
    }
	SLIST_INIT(rx_metrics_head);

	return wapp_utils_success;
}
int insert_new_tx_link_metrics_info(
	struct list_head_tx_metrics_agent *tx_metrics_head, unsigned char *buf, unsigned short len)
{
	 unsigned char *pos = buf;
	 unsigned short check_len = 0;
	 struct tx_link_metric_db *tx_link_metric = NULL;
	 struct tx_metric_db *tx_metric = NULL;
	 unsigned char *neighbor_dev = buf + 6;

    SLIST_FOREACH(tx_link_metric, tx_metrics_head, tx_link_metric_entry)
    {
        if(!memcmp(tx_link_metric->nalmac, neighbor_dev, ETH_ALEN)) {
			debug(DEBUG_TRACE, "neighbor dev(%02x:%02x:%02x:%02x:%02x:%02x) found\n",
				PRINT_MAC(neighbor_dev));
            break;
        }
    }

	if (!tx_link_metric) {
		tx_link_metric = (struct tx_link_metric_db *)malloc(sizeof(*tx_link_metric));
		if (!tx_link_metric) {
			debug(DEBUG_ERROR, "alloc struct tx_link_metric_db fail\n");
			return wapp_utils_error;
		}
		memset(tx_link_metric, 0, sizeof(*tx_link_metric));
		memcpy(tx_link_metric->almac, pos, ETH_ALEN);
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		memcpy(tx_link_metric->nalmac, pos, ETH_ALEN);
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		SLIST_INIT(&tx_link_metric->tx_metric_head);
		SLIST_INSERT_HEAD(tx_metrics_head, tx_link_metric, tx_link_metric_entry);

		debug(DEBUG_TRACE, "insert struct tx_link_metric_db\n");
		debug(DEBUG_TRACE, "almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(tx_link_metric->almac));
		debug(DEBUG_TRACE, "nalmac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(tx_link_metric->nalmac));
		while(check_len < len) {
			tx_metric = (struct tx_metric_db *)malloc(sizeof(struct tx_metric_db));
			if (!tx_metric) {
				debug(DEBUG_ERROR, "alloc struct tx_metric_db fail\n");
				return wapp_utils_error;
			}
			memcpy(tx_metric->mac, pos, ETH_ALEN);
			pos += ETH_ALEN;
			check_len += ETH_ALEN;
			memcpy(tx_metric->nmac, pos, ETH_ALEN);
			pos += ETH_ALEN;
			check_len += ETH_ALEN;
			tx_metric->intf_type = (*(unsigned short *)pos);
			tx_metric->intf_type = be2cpu16(tx_metric->intf_type);
			pos += 2;
			check_len += 2;
			tx_metric->bridge_flag = *pos;
			pos += 1;
			check_len += 1;
			tx_metric->error_packet = (*(unsigned int *)pos);
			tx_metric->error_packet = be2cpu32(tx_metric->error_packet);
			pos += 4;
			check_len += 4;
			tx_metric->tx_packets = (*(unsigned int *)pos);
			tx_metric->tx_packets = be2cpu32(tx_metric->tx_packets);
			pos += 4;
			check_len += 4;
			tx_metric->mac_tpcap = (*(unsigned short *)pos);
			tx_metric->mac_tpcap = be2cpu16(tx_metric->mac_tpcap);
			pos += 2;
			check_len += 2;
			tx_metric->linkavl = (*(unsigned short *)pos);
			tx_metric->linkavl = be2cpu16(tx_metric->linkavl);
			pos += 2;
			check_len += 2;
			tx_metric->phyrate = (*(unsigned short *)pos);
			tx_metric->phyrate = be2cpu16(tx_metric->phyrate);
			pos += 2;
			check_len += 2;
			SLIST_INSERT_HEAD(&tx_link_metric->tx_metric_head, tx_metric, tx_metric_entry);

			debug(DEBUG_TRACE, "insert struct tx_metric_db\n");
			debug(DEBUG_TRACE, "mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(tx_metric->mac));
			debug(DEBUG_TRACE, "connect mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(tx_metric->nmac));
			debug(DEBUG_TRACE, "intf_type=%d, bridge_flag=%d error_packet=%d, ",
				tx_metric->intf_type, tx_metric->bridge_flag, tx_metric->error_packet);
			debug(DEBUG_TRACE, "tx_packets=%d, mac_tpcap=%d linkavl=%d, phyrate=%d\n",
				tx_metric->tx_packets, tx_metric->mac_tpcap, tx_metric->linkavl, tx_metric->phyrate);
		}
	}else {
		debug(DEBUG_ERROR, "tx_link_metric exist\n");
		return wapp_utils_error;
	}

	return wapp_utils_success;

}


int parse_tx_link_metrics_tlv(struct p1905_managerd_ctx* ctx, unsigned char *buf,
	struct list_head_tx_metrics_agent *tx_metrics_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == TRANSMITTER_LINK_METRIC_TYPE) {
        temp_buf++;
    } else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if(ctx->Certification == 1)
   	{
	   	//shift to tlv value field
	    temp_buf += 2;

		/*insert new channel preference info*/
		if (0 > insert_new_tx_link_metrics_info(tx_metrics_head, temp_buf, (unsigned short)length)) {
			debug(DEBUG_ERROR, "insert_new_tx_link_metrics_info fail\n");
			return -1;
		}
	}
    return (length+3);
}

int insert_new_rx_link_metrics_info(
	struct list_head_rx_metrics_agent *rx_metrics_head, unsigned char *buf, unsigned short len)
{
	 unsigned char *pos = buf;
	 unsigned short check_len = 0;
	 struct rx_link_metric_db *rx_link_metric = NULL;
	 struct rx_metric_db *rx_metric = NULL;
	 unsigned char *neighbor_dev = buf + 6;

    SLIST_FOREACH(rx_link_metric, rx_metrics_head, rx_link_metric_entry)
    {
        if(!memcmp(rx_link_metric->nalmac, neighbor_dev, ETH_ALEN)) {
			debug(DEBUG_TRACE, "neighbor dev(%02x:%02x:%02x:%02x:%02x:%02x) found\n",
				PRINT_MAC(neighbor_dev));
            break;
        }
    }

	if (!rx_link_metric) {
		rx_link_metric = (struct rx_link_metric_db *)malloc(sizeof(*rx_link_metric));
		if (!rx_link_metric) {
			debug(DEBUG_ERROR, "alloc struct rx_link_metric_db fail\n");
			return wapp_utils_error;
		}
		memset(rx_link_metric, 0, sizeof(*rx_link_metric));
		memcpy(rx_link_metric->almac, pos, ETH_ALEN);
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		memcpy(rx_link_metric->nalmac, pos, ETH_ALEN);
		pos += ETH_ALEN;
		check_len += ETH_ALEN;
		SLIST_INIT(&rx_link_metric->rx_metric_head);
		SLIST_INSERT_HEAD(rx_metrics_head, rx_link_metric, rx_link_metric_entry);

		debug(DEBUG_TRACE, "insert struct rx_link_metric_db\n");
		debug(DEBUG_TRACE, "almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(rx_link_metric->almac));
		debug(DEBUG_TRACE, "nalmac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(rx_link_metric->nalmac));
		while(check_len < len) {
			rx_metric = (struct rx_metric_db *)malloc(sizeof(struct rx_metric_db));
			if (!rx_metric) {
				debug(DEBUG_ERROR, "alloc struct rx_metric_db fail\n");
				return wapp_utils_error;
			}
			memcpy(rx_metric->mac, pos, ETH_ALEN);
			pos += ETH_ALEN;
			check_len += ETH_ALEN;
			memcpy(rx_metric->nmac, pos, ETH_ALEN);
			pos += ETH_ALEN;
			check_len += ETH_ALEN;
			rx_metric->intf_type = (*(unsigned short *)pos);
			rx_metric->intf_type = be2cpu16(rx_metric->intf_type);
			pos += 2;
			check_len += 2;
			rx_metric->error_packet = (*(unsigned int *)pos);
			rx_metric->error_packet = be2cpu32(rx_metric->error_packet);
			pos += 4;
			check_len += 4;
			rx_metric->rx_packets = (*(unsigned int *)pos);
			rx_metric->rx_packets = be2cpu32(rx_metric->rx_packets);
			pos += 4;
			check_len += 4;
			rx_metric->rssi= *pos;
			pos += 1;
			check_len += 1;
			SLIST_INSERT_HEAD(&rx_link_metric->rx_metric_head, rx_metric, rx_metric_entry);
			debug(DEBUG_TRACE, "insert struct rx_metric_db\n");
			debug(DEBUG_TRACE, "mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(rx_metric->mac));
			debug(DEBUG_TRACE, "connect mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(rx_metric->nmac));
			debug(DEBUG_TRACE, "intf_type=%d, error_packet=%d rx_packets=%d rssi=%d\n",
				rx_metric->intf_type, rx_metric->error_packet, rx_metric->rx_packets,
				rx_metric->rssi);
		}
	}else {
		debug(DEBUG_ERROR, "rx_link_metric exist\n");
		return wapp_utils_error;
	}

	return wapp_utils_success;

}

int parse_rx_link_metrics_tlv(struct p1905_managerd_ctx* ctx, unsigned char *buf,
	struct list_head_rx_metrics_agent *rx_metrics_head)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == RECEIVER_LINK_METRIC_TYPE) {
        temp_buf++;
    } else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if(ctx->Certification == 1)
	{
		//shift to tlv value field
	    temp_buf += 2;

		/*insert new channel preference info*/
		if (0 > insert_new_rx_link_metrics_info(rx_metrics_head, temp_buf, (unsigned short)length)) {
			debug(DEBUG_ERROR, "insert_new_rx_link_metrics_info fail\n");
			return -1;
		}
	}
    return (length+3);
}

int parse_link_metrics_response_message(struct p1905_managerd_ctx *ctx,
	struct agent_list_db *agent, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;

	temp_buf = buf;
	reset_stored_tlv(ctx);
	while(1) {
		if (*temp_buf == TRANSMITTER_LINK_METRIC_TYPE) {
			length = parse_tx_link_metrics_tlv(ctx, temp_buf, &agent->tx_metrics_head);
			if(length < 0) {
				debug(DEBUG_ERROR, "error tx link metrics tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == RECEIVER_LINK_METRIC_TYPE) {
			length = parse_rx_link_metrics_tlv(ctx, temp_buf, &agent->rx_metrics_head);
			if(length < 0) {
				debug(DEBUG_ERROR, "error rx link metrics tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}

	return 0;
}

unsigned short append_tx_link_metric_tlv(
        unsigned char *pkt, struct tx_link_metric_db *tx_link_metric)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;
	struct tx_metric_db *tx_metric = NULL;

    temp_buf = pkt;

    *temp_buf = TRANSMITTER_LINK_METRIC_TYPE;
    temp_buf +=1;
    total_length += 1;

    /* The length of tx link metric tlv is a variable.
     * It depends on connetced interface number, shift to payload first
     */
    temp_buf +=2;
    total_length += 2;

    /*fill into local abstration layer mac addr*/
    memcpy(temp_buf, tx_link_metric->almac, ETH_ALEN);
    temp_buf += ETH_ALEN;
    total_length += ETH_ALEN;

    /*fill into neighbor abstration layer mac addr*/
    memcpy(temp_buf, tx_link_metric->nalmac, ETH_ALEN);
    temp_buf += ETH_ALEN;
    total_length += ETH_ALEN;

    SLIST_FOREACH(tx_metric, &tx_link_metric->tx_metric_head, tx_metric_entry)
    {
		/*fill into local interface mac addr*/
		memcpy(temp_buf, tx_metric->mac, ETH_ALEN);
		temp_buf += ETH_ALEN;
		total_length += ETH_ALEN;
		/*fill into neighbor interface mac addr*/
		memcpy(temp_buf, tx_metric->nmac, ETH_ALEN);
		temp_buf += ETH_ALEN;
		total_length += ETH_ALEN;
		/*fill into interface media type*/
		*(unsigned short *)(temp_buf) = cpu2be16(tx_metric->intf_type);
		temp_buf += 2;
		total_length += 2;
		/*fill into bridge flag*/
		*temp_buf = tx_metric->bridge_flag;
		temp_buf += 1;
		total_length += 1;
		/*fill into tx packets error*/
		*(unsigned int *)(temp_buf) = cpu2be32(tx_metric->error_packet);
		temp_buf += 4;
		total_length += 4;
		/*fill into total transmitted packets*/
		*(unsigned int *)(temp_buf) = cpu2be32(tx_metric->tx_packets);
		temp_buf += 4;
		total_length += 4;
		/*fill into max throughput capability*/
		*(unsigned short *)(temp_buf) = cpu2be16(tx_metric->mac_tpcap);
		temp_buf += 2;
		total_length += 2;
		/*fill into link availability field */
		*(unsigned short *)(temp_buf) = cpu2be16(tx_metric->linkavl);
		temp_buf += 2;
		total_length += 2;
		/* fill into phy rate*/
		*(unsigned short *)(temp_buf) = cpu2be16(tx_metric->phyrate);
		temp_buf += 2;
		total_length += 2;
    }

    /*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

unsigned short append_rx_link_metric_tlv(
        unsigned char *pkt, struct rx_link_metric_db *rx_link_metric)
{
    unsigned short total_length = 0;
    unsigned char *temp_buf;
	struct rx_metric_db *rx_metric = NULL;

    temp_buf = pkt;

    *temp_buf = RECEIVER_LINK_METRIC_TYPE;
    temp_buf +=1;
    total_length += 1;

    /* The length of rx link metric tlv is a variable.
     * It depends on connetced interface number, shift to payload first
     */
    temp_buf +=2;
    total_length += 2;

    /*fill into local abstration layer mac addr*/
    memcpy(temp_buf, rx_link_metric->almac, ETH_ALEN);
    temp_buf += ETH_ALEN;
    total_length += ETH_ALEN;

    /*fill into neighbor abstration layer mac addr*/
    memcpy(temp_buf, rx_link_metric->nalmac, ETH_ALEN);
    temp_buf += ETH_ALEN;
    total_length += ETH_ALEN;

    SLIST_FOREACH(rx_metric, &rx_link_metric->rx_metric_head, rx_metric_entry)
    {
		/*fill into local interface mac addr*/
		memcpy(temp_buf, rx_metric->mac, ETH_ALEN);
		temp_buf += ETH_ALEN;
		total_length += ETH_ALEN;
		/*fill into neighbor interface mac addr*/
		memcpy(temp_buf, rx_metric->nmac, ETH_ALEN);
		temp_buf += ETH_ALEN;
		total_length += ETH_ALEN;
		/*fill into interface media type*/
		*(unsigned short *)(temp_buf) = cpu2be16(rx_metric->intf_type);
		temp_buf += 2;
		total_length += 2;
		/*fill into rx packets error*/
		*(unsigned int *)(temp_buf) = cpu2be32(rx_metric->error_packet);
		temp_buf += 4;
		total_length += 4;
		/*fill into total received packets*/
		*(unsigned int *)(temp_buf) = cpu2be32(rx_metric->rx_packets);
		temp_buf += 4;
		total_length += 4;
		/* fill rssi*/
		*temp_buf = rx_metric->rssi;
		temp_buf += 1;
		total_length += 1;
    }

    /*calculate totoal length & fill into the length field*/
	*(unsigned short *)(pkt+1) = cpu2be16(total_length - 3);

    return total_length;
}

int free_all_the_agents_info(struct list_head_agent *agent_head)
{
	struct agent_list_db *agent = SLIST_FIRST(agent_head);
	struct agent_list_db *agent_tmp = NULL;

	while(agent != NULL) {
		agent_tmp = SLIST_NEXT(agent, agent_entry);
		debug(DEBUG_TRACE, "free agent_list_db(almac=%02x:%02x:%02x:%02x:%02x:%02x))\n",
			PRINT_MAC(agent->almac));

		delete_all_ch_prefer_info((struct list_head_ch_prefer *)&agent->ch_prefer_head);
		delete_all_radio_oper_restrict_info((struct list_head_oper_restrict *)&agent->oper_restrict_head);
		delete_agent_ap_metrics_info(&agent->metrics_rsp_head);
		delete_agent_tx_link_metrics_info(&agent->tx_metrics_head);
		delete_agent_rx_link_metrics_info(&agent->rx_metrics_head);
		agent = agent_tmp;
	}

	return 0;
}
extern unsigned char p1905_multicast_address[6];
extern unsigned char dev_send_1905_buf[3072];

void send_1905_raw_data(struct p1905_managerd_ctx* ctx, char *file)
{
	struct _1905_cmdu_request *request = (struct _1905_cmdu_request*)dev_send_1905_buf;

	if(_1905_read_dev_send_1905(ctx, file, request->dest_al_mac, &request->type,
			&request->len, request->body) < 0) {
		debug(DEBUG_ERROR, "parse raw data from %s fail\n", file);
		return;
	}

	dev_send_1905(ctx, request);
}

void read_1905_bss_config(struct p1905_managerd_ctx* ctx, char *file)
{
	if (_1905_read_set_config(ctx, file, ctx->bss_config, MAX_SET_BSS_INFO_NUM, &ctx->bss_config_num) < 0)
		debug(DEBUG_ERROR, "load bss config fail from %s\n", file);
}

#ifdef SUPPORT_CMDU_RELIABLE
void cmdu_reliable_send(struct p1905_managerd_ctx *ctx, unsigned short msg_type,
		unsigned short mid, int ifidx)
{
	struct topology_response_db *tpr_db = NULL;

	SLIST_FOREACH(tpr_db, &ctx->topology_entry.tprdb_head, tprdb_entry)
	{
		/*relay this unicast message to all devices in the same topology*/
		if (tpr_db->recv_ifid == ifidx) {
			insert_cmdu_txq(tpr_db->al_mac_addr, ctx->p1905_al_mac_addr,
					msg_type, mid, (ifidx != -1) ? ctx->itf[ifidx].if_name : ctx->br_name, 0);
		}
	}
}
#endif

int map_event_handler(struct p1905_managerd_ctx *ctx, char *buf, int len, unsigned char type)
{
	struct msg *wapp_event = NULL;
	int if_num = 0;

	wapp_event = (struct msg *)buf;

	if ((type != 0x00) && (wapp_event->type == type)) {
		if (wapp_event->length == 0) {
			debug(DEBUG_ERROR, "waittype(0x%04x) == wapp_event->type=(0x%04x); data is invalid\n",
				 	type, wapp_event->type);
			return -2;
		}
	}

	if (wapp_event->length == 0) {
		if (wapp_event->type != LIB_1905_CMDU_REQUEST &&
			wapp_event->type != LIB_STEERING_COMPLETED &&
			wapp_event->type != WAPP_GET_OWN_TOPOLOGY_RSP &&
			wapp_event->type != LIB_CLEAR_SWITCH_TABLE
			) {
			 debug(DEBUG_ERROR, "invalid data from wapp! type=0x%04x\n", wapp_event->type);
			return -1;
		}
	}

	if((type != 0x00) && wapp_event->type != type)
	{
		debug(DEBUG_TRACE, "waittype(0x%04x) != wapp_event->type=(0x%04x)\n", type, wapp_event->type);
		/*avoid to drop unsolicited message, insert to a waitq*/
		insert_message_wait_queue(wapp_event->type, (unsigned char *)buf, len);
		return -1;
	}

	debug(DEBUG_TRACE, "receive event(0x%04x)\n", wapp_event->type);

	switch (wapp_event->type) {
	case LIB_WIRELESS_INTERFACE_INFO:
	{
		struct interface_info_list_hdr *info = NULL;

		info = (struct interface_info_list_hdr *)wapp_event->buffer;
		init_wireless_interface(ctx, info);
		common_info_init(ctx);
	}
		break;
	case LIB_CLEAR_SWITCH_TABLE:
	{
		eth_layer_clear_switch_table();
	}
		break;
	case LIB_MAP_BH_READY:
	{
		struct bh_link_info *info = NULL;
		int i = 0;
		unsigned char notify = 0;
		int ret = 0, ret1 = 0;
		unsigned char almac[ETH_ALEN] = {0};

		info = (struct bh_link_info *)wapp_event->buffer;

		if (ctx->role == CONTROLLER)
			break;

		if (info->type == MAP_BH_ETH) {
			debug(DEBUG_ERROR, "eth link up event\n");
			ctx->controller_search_state = bk_link_ready;
		    ctx->controller_search_cnt = 0;
			eloop_register_timeout(0, 0, ap_controller_search_step, (void *)ctx, NULL);
			break;
		}

		for (i = 0; i < ctx->itf_number; i++) {
			if (!os_memcmp(info->mac_addr, ctx->itf[i].mac_addr, ETH_ALEN)) {
				ret = 1;
				if (ctx->itf[i].is_wifi_sta != 1) {
					debug(DEBUG_ERROR, "not apcli interface\n");
					ret = -1;
				}
				break;
			}
		}
		if (ret != 1) {
			debug(DEBUG_ERROR, "wrong bh_link_info!!! ret(%d)\n", ret);
			break;
		}

		debug(DEBUG_OFF, "LIB_MAP_BH_READY wifi connect\n");
		debug(DEBUG_OFF, "local interface(%s-%02x:%02x:%02x:%02x:%02x:%02x) ",
			info->ifname, PRINT_MAC(info->mac_addr));
		debug(DEBUG_OFF, "connect to bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(info->bssid));
		debug(DEBUG_OFF, "send discovery with vs info for[%s]\n", ctx->itf[i].if_name);
		eloop_register_timeout(0, 0, discovery_at_apcli_link_up,
			(void *)ctx, (void *)ctx->itf[i].if_name);
		memcpy(ctx->itf[i].vs_info, info->bssid, ETH_ALEN);
		ctx->itf[i].trx_config = TX_MUL | RX_MUL | TX_UNI | RX_UNI;

		/*check if bssid in this event is operating on one of neighbor MAP device*/
		ret = find_almac_by_connect_mac(ctx, info->bssid, almac);
		if (ret == 1) {
			debug(DEBUG_OFF, "already receive discovery from "
					"almac(%02x:%02x:%02x:%02x:%02x:%02x) by "
					"itf(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(almac), PRINT_MAC(info->bssid));
		} else if (ret == 2) {
			debug(DEBUG_OFF, "not receive discovery from "
					"almac(%02x:%02x:%02x:%02x:%02x:%02x) by "
					"itf(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(almac), PRINT_MAC(info->bssid));
			debug(DEBUG_OFF, "already receive response from "
					"almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(almac));
			debug(DEBUG_OFF, "need add neighbor info and discovery for"
					"almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(almac));
			ret1 = insert_topology_discovery_database(ctx, almac,
					info->bssid, info->mac_addr);
			if (ret1 >= 0) {
				update_p1905_neighbor_dev_info(ctx, almac, info->bssid,
					info->mac_addr, &notify);
				/*report updated own topology response to mapd*/
				report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
					ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
					ctx->service, &ctx->ap_cap_entry.oper_bss_head,
					&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
				update_topology_tree(ctx);
				/*send notification to notify other device my topology is changed*/
				ctx->mid++;
				for(if_num = 0; if_num < ctx->itf_number; if_num++) {
					insert_cmdu_txq((unsigned char*)p1905_multicast_address,
						(unsigned char*)ctx->p1905_al_mac_addr,
						e_topology_notification, ctx->mid,
						ctx->itf[if_num].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
					cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, if_num);
#endif
				}
			}
		} else {
			debug(DEBUG_OFF, "not receive discovery or response from "
					"dev with interface(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(info->bssid));
		}

		/*trigger_autconf will be set in first wifi link in wifi case or in eth case*/
		if (info->trigger_autconf) {
			ctx->controller_search_state = bk_link_ready;
		    ctx->controller_search_cnt = 0;
			eloop_register_timeout(0, 0, ap_controller_search_step, (void *)ctx, NULL);
		}
	}
		break;
	case WAPP_NOTIFY_WIRELESS_BH_LINK_DOWN:
	{
		struct bh_link_info *info = NULL;
		struct topology_discovery_db *db = NULL;
		int i = 0;
		unsigned char delete_almac[ETH_ALEN] = {0};
		int ret = 0;
		struct p1905_neighbor_info *dev_info, *dev_info_temp = NULL;

		info = (struct bh_link_info *)wapp_event->buffer;

		debug(DEBUG_OFF, "backhaul link down(%s-%02x:%02x:%02x:%02x:%02x:%02x)\n",
				info->ifname, PRINT_MAC(info->mac_addr));

		if (info->type != MAP_BH_WIFI) {
			debug(DEBUG_ERROR, "not wifi link down event\n");
			break;
		}

		for (i = 0; i < ctx->itf_number; i++) {
			if (!os_memcmp(info->mac_addr, ctx->itf[i].mac_addr, ETH_ALEN)) {
				ret = 1;
				if (ctx->itf[i].is_wifi_sta != 1) {
					debug(DEBUG_ERROR, "not apcli interface\n");
					ret = -1;
				}
				break;
			}
		}
		if (ret != 1) {
			debug(DEBUG_ERROR, "wrong bh_link_info!!! ret(%d)\n", ret);
			break;
		}

		reset_controller_connect_ifname(ctx, ctx->itf[i].if_name);

		db = get_tpd_db_by_mac(ctx, ctx->itf[i].vs_info);
		os_memset(ctx->itf[i].vs_info, 0, ETH_ALEN);
		/*temporarily set sta interface not to send multicast cmdu which is not connected*/
		ctx->itf[i].trx_config = 0;
		if (!db) {
			debug(DEBUG_ERROR, "cannot find discovery for mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(ctx->itf[i].vs_info));
			/*clear neighbor info for apcli for error handle*/
			dev_info = LIST_FIRST(&ctx->p1905_neighbor_dev[i].p1905nbr_head);
			while (dev_info) {
				dev_info_temp = LIST_NEXT(dev_info, p1905nbr_entry);
				LIST_REMOVE(dev_info, p1905nbr_entry);
				debug(DEBUG_OFF, GRN("delete almac(%02x:%02x:%02x:%02x:%02x:%02x)\n"),
					PRINT_MAC(dev_info->al_mac_addr));
				os_free(dev_info);
				dev_info = dev_info_temp;
			}
			break;
		}

		debug(DEBUG_OFF, "delete topology discovery db"
			"almac(%02x:%02x:%02x:%02x:%02x:%02x) "
			"itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(db->al_mac), PRINT_MAC(db->itf_mac));
		os_memcpy(delete_almac, db->al_mac, ETH_ALEN);
		delete_exist_topology_discovery_database(ctx, db->al_mac, db->itf_mac);
		/*cannot refer db, the memory is free*/
		report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
			ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
			ctx->service, &ctx->ap_cap_entry.oper_bss_head,
			&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
		if (find_discovery_by_almac(ctx, delete_almac) == NULL) {
			debug(DEBUG_OFF, "WAPP_NOTIFY_WIRELESS_BH_LINK_DOWN "
				"delete topology rsp(%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(delete_almac));
			delete_exist_topology_response_database(ctx, delete_almac);
			mark_valid_topo_rsp_node(ctx);
		}
		update_topology_tree(ctx);

		ctx->mid++;
		for(if_num = 0; if_num < ctx->itf_number; if_num++) {
			insert_cmdu_txq((unsigned char*)p1905_multicast_address, (unsigned char*)ctx->p1905_al_mac_addr,\
            	e_topology_notification, ctx->mid, ctx->itf[if_num].if_name, 0);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, if_num);
#endif
		}
	}
		break;
	case WAPP_NOTIFY_CH_BW_INFO:
	{
		struct channel_bw_info *info = NULL;
		info = (struct channel_bw_info *)wapp_event->buffer;
		for(if_num = 0; if_num < ctx->itf_number; if_num++) {
			if (!memcmp(ctx->itf[if_num].mac_addr, info->iface_addr,
				ETH_ALEN)) {
				ctx->itf[if_num].channel_bw = info->channel_bw;
				ctx->itf[if_num].channel_freq = info->channel_num;
			}
		}
		break;
	}
	case WAPP_GET_OWN_TOPOLOGY_RSP:
	{
		debug(DEBUG_OFF, YLW("get my own topology rsp") "\n");
		report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
			ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
	       	ctx->service, &ctx->ap_cap_entry.oper_bss_head,
	        &ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
	}
		break;
	case GET_SWITCH_STATUS:
	{
		int i = 0, status_temp = 0;
		unsigned char status = 0;

		for (i = 0; i < MAX_LAN_PORT_NUM; i++) {
			if ((status_temp = eth_layer_get_eth_port_status(i)) < 0) {
				status = 0xff;
				break;
			} else {
				if (status_temp)
					status |= (1 << i);
			}
		}

		_1905_notify_switch_status(ctx, status);
		debug(DEBUG_OFF, "notify switch status %02x\n", status);
	}
		break;
	case LIB_GET_WSC_CONF:
	{
		int i = 0;
		struct wps_get_config* autoconf = NULL;

		if (ctx->role == CONTROLLER)
			break;

		autoconf = (struct wps_get_config *)wapp_event->buffer;
		ctx->authenticated = 1;

		update_radio_info(ctx, autoconf);

		if (set_radio_autoconf_trriger(ctx, autoconf->identifier, 1) < 0)
			break;

		/*trigger auto configuration*/
		if (triger_autoconfiguration(ctx) < 0)
			break;

		if (ctx->Certification == 1)
		{
			i = ctx->current_autoconfig_info.radio_index;
			wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_RADIO_BASIC_CAP, LIB_RADIO_BASIC_CAP,
				ctx->rinfo[i].identifier, NULL, NULL, 0);
		}
	}
		break;
	case LIB_RADIO_BASIC_CAP:
	{
		struct ap_radio_basic_cap *bcap = NULL;

		bcap = (struct ap_radio_basic_cap *)wapp_event->buffer;
		delete_exist_radio_basic_capability(ctx, bcap->identifier);
		insert_new_radio_basic_capability(ctx, bcap);
	}
		break;
	case LIB_AP_CAPABILITY:
	{
		struct ap_capability *cap = NULL;

		cap = (struct ap_capability *)wapp_event->buffer;
		memcpy(&ctx->ap_cap_entry.ap_cap, cap, sizeof(*cap));
		debug(DEBUG_TRACE, "sta_report_on_cop=%d, sta_report_not_cop%d, rssi_steer=%d\n",
			cap->sta_report_on_cop, cap->sta_report_not_cop, cap->rssi_steer);
	}
		break;
	case LIB_AP_HT_CAPABILITY:
	{
		struct ap_ht_capability *cap = NULL;

		cap = (struct ap_ht_capability *)wapp_event->buffer;
		update_ap_ht_cap(ctx, cap);
		debug(DEBUG_TRACE, "tx_stream=%d, rx_stream=%d, sgi_20=%d, sgi_40=%d, ht_40=%d, band=%d\n",
			cap->tx_stream, cap->rx_stream, cap->sgi_20, cap->sgi_40, cap->ht_40, cap->band);
	}
		break;
	case LIB_AP_VHT_CAPABILITY:
	{
		struct ap_vht_capability *cap = NULL;

		cap = (struct ap_vht_capability *)wapp_event->buffer;
		update_ap_vht_cap(ctx, cap);
		debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(cap->identifier));
		debug(DEBUG_TRACE, "vht_tx_mcs=%x, vht_rx_mcs=%x, tx_stream=%d, rx_stream=%d\n",
			cap->vht_tx_mcs, cap->vht_rx_mcs, cap->tx_stream, cap->rx_stream);
		debug(DEBUG_TRACE, "sgi_80=%d, sgi_160=%d, vht_8080=%d, vht_160=%d\n",
			cap->sgi_80, cap->sgi_160, cap->vht_8080, cap->vht_160);
		debug(DEBUG_TRACE, "su_beamformer=%d, mu_beamformer=%d, band=%d\n",
			cap->su_beamformer, cap->mu_beamformer, cap->band);
	}
		break;
	case LIB_AP_HE_CAPABILITY:
	{
		struct ap_he_capability *cap = NULL;

		cap = (struct ap_he_capability *)wapp_event->buffer;
		update_ap_he_cap(ctx, cap);
		if (cap->he_mcs_len >= 2 && cap->he_mcs_len <=12)
			hex_dump("he_mcs", cap->he_mcs, cap->he_mcs_len);
		debug(DEBUG_TRACE, "tx_stream=%d, rx_stream=%d, he_8080=%d, he_160=%d\n",
			cap->tx_stream, cap->rx_stream, cap->he_8080, cap->he_160);
		debug(DEBUG_TRACE, "su_bf_cap=%d, mu_bf_cap=%d, ul_mu_mimo_cap=%d\n",
			cap->su_bf_cap, cap->mu_bf_cap, cap->ul_mu_mimo_cap);
		debug(DEBUG_TRACE, "ul_mu_mimo_ofdma_cap=%d, dl_mu_mimo_ofdma_cap=%d\n",
			cap->ul_mu_mimo_ofdma_cap, cap->dl_mu_mimo_ofdma_cap);
		debug(DEBUG_TRACE, "ul_ofdma_cap=%d, dl_ofdma_cap=%d\n",
			cap->ul_ofdma_cap, cap->dl_ofdma_cap);
	}
		break;
#ifdef MAP_R2
	/*channel scan feature*/
	//case WAPP_CHANNEL_SCAN_CAP:
	case LIB_CHANNEL_SCAN_CAPAB:
	{
		struct scan_capability_lib *scap = NULL;

		scap = (struct scan_capability_lib *)wapp_event->buffer;
		delete_exist_channel_scan_capability(ctx, scap->identifier);
		insert_new_channel_scan_capability(ctx, scap);
	}
		break;
#endif
	case LIB_CLI_CAPABILITY_REPORT:
	{
		struct client_capa_rep *cap = NULL;
		struct client_capa_rep *ctx_cap = NULL;

		cap = (struct client_capa_rep *)wapp_event->buffer;
		free(ctx->sinfo.pcli_rep);
		ctx->sinfo.pcli_rep = (struct client_capa_rep *)malloc(cap->length + 1);
		if(NULL == ctx->sinfo.pcli_rep) {
			debug(DEBUG_ERROR, "LIB_CLI_CAPABILITY_REPORT allocate memory fail (%s)\n",
				strerror(errno));
       		break;
    	}
		ctx_cap = ctx->sinfo.pcli_rep;
		ctx_cap->result = cap->result;
		ctx_cap->length = 0;
		if (ctx_cap->result == 0) {
			ctx_cap->length = cap->length;
			memcpy(ctx_cap->body, cap->body, ctx_cap->length);
			hex_dump("assoc req", ctx_cap->body, ctx_cap->length);
		}
	}
		break;
	case LIB_OPERBSS_REPORT:
	{
		struct oper_bss_cap *opcap = NULL;

		opcap = (struct oper_bss_cap *)wapp_event->buffer;
		delete_exist_operational_bss(ctx, opcap->identifier);
		insert_new_operational_bss(ctx, opcap);
		delete_legacy_assoc_clients_db(ctx);
	}
		break;
	case LIB_CHANNLE_PREFERENCE:
	{
		struct ch_prefer *prefer = NULL;

		prefer = (struct ch_prefer *)wapp_event->buffer;
		delete_exist_ch_prefer_info(ctx, prefer->identifier);
		insert_new_channel_prefer_info(ctx, prefer);
	}
		break;
	case WAPP_CHANNEL_PREFERENCE_REPORT_INFO:
	{
		if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
		{
			break;
		}
		if (!ctx->rcv_chprefer_query) {
			debug(DEBUG_TRACE, "receive unsolicited channel preference report event\n");
	    	debug(DEBUG_TRACE, "send WAPP_CHANNEL_PREFERENCE_REPORT_INFO to"
					"controller(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(ctx->cinfo.almac));
	        insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
	        	e_channel_preference_report, ++ctx->mid, ctx->cinfo.local_ifname, 0);
		}
	}
		break;
	case WAPP_CHANNEL_SELECTION_RSP_INFO:
	{
		if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
		{
			break;
		}
	}
		break;
	case LIB_RADIO_OPERATION_RESTRICTION:
	{
		struct restriction *resdb = NULL;

		resdb = (struct restriction *)wapp_event->buffer;
		delete_exist_restriction_info(ctx, resdb->identifier);
		insert_new_restriction_info(ctx, resdb);
	}
		break;
	case LIB_TX_LINK_STATISTICS:
	{
		struct tx_link_stat_rsp *rsp = NULL;

		rsp = (struct tx_link_stat_rsp *)wapp_event->buffer;
		memcpy(&ctx->link_stat.tx_link_stat, rsp,
			sizeof(struct tx_link_stat_rsp));
	}
		break;
	case LIB_RX_LINK_STATISTICS:
	{
		struct rx_link_stat_rsp *rsp = NULL;

		rsp = (struct rx_link_stat_rsp *)wapp_event->buffer;
		memcpy(&ctx->link_stat.rx_link_stat, rsp,
			sizeof(struct rx_link_stat_rsp));
	}
		break;
	case WAPP_AP_METRICS_RSP_INFO:
	{
		if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
		{
			break;
		}
		if (!ctx->rcv_apmetrics_query) {
			debug(DEBUG_TRACE, "receive unsolicited ap metrics rsp event\n");
	    	debug(DEBUG_TRACE, "send WAPP_AP_METRICS_RSP_INFO to"
					"controller(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(ctx->cinfo.almac));
	        insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
	        	e_ap_metrics_response, ++ctx->mid, ctx->cinfo.local_ifname, 0);
		}
	}
		break;
	case LIB_AP_METRICS_INFO:
	{
		struct ap_metrics_info *minfo = NULL;

		minfo = (struct ap_metrics_info *)wapp_event->buffer;
		delete_exist_metrics_info(ctx, minfo->bssid);
		insert_new_metrics_info(ctx, minfo);
	}
		break;
	case LIB_ALL_ASSOC_STA_TRAFFIC_STATS:
	{
		struct sta_traffic_stats *stats = NULL;

		stats = (struct sta_traffic_stats *)wapp_event->buffer;
		delete_exist_traffic_stats_info(ctx, stats->identifier);
		insert_new_traffic_stats_info(ctx, stats);
	}
		break;
	case LIB_ALL_ASSOC_STA_LINK_METRICS:
	{
		struct sta_link_metrics *metrics = NULL;

		metrics = (struct sta_link_metrics *)wapp_event->buffer;
		delete_exist_link_metrics_info(ctx, metrics->identifier);
		insert_new_link_metrics_info(ctx, metrics);
	}
		break;
	case LIB_ONE_ASSOC_STA_LINK_METRICS:
	{
		if(ctx->Certification == 1)
		{
			struct link_metrics *metrics = NULL;

			metrics = (struct link_metrics *)wapp_event->buffer;
			update_one_sta_link_metrics_info(ctx, metrics);
		}
		else
		{
			if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
			{
				break;
			}

		}
	}
		break;
	case LIB_UNASSOC_STA_LINK_METRICS:
	{
		if(ctx->Certification == 1)
		{
			struct unlink_metrics_rsp *unlink_metrics = NULL;

			unlink_metrics = (struct unlink_metrics_rsp *)wapp_event->buffer;
			delete_exist_unlink_metrics_rsp(&ctx->metric_entry.unlink_info);
			update_unlink_metrics_rsp(&ctx->metric_entry.unlink_info, unlink_metrics);
		}
		else
		{
			if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
			{
				break;
			}
		}
	}
		break;
	case LIB_OPERATING_CHANNEL_INFO:
		update_channel_report_info(ctx, (struct channel_report *)wapp_event->buffer,
			wapp_event->length);
		break;
	case LIB_CLIENT_NOTIFICATION:
	{
		struct client_association_event *evt = NULL;
		struct map_client_association_event *pevt = NULL;
		struct topology_discovery_db *db = NULL;
		unsigned char delete_almac[ETH_ALEN] = {0};

		evt = (struct client_association_event *)wapp_event->buffer;
		pevt = (struct map_client_association_event *)&evt->map_assoc_evt;
		memcpy(&ctx->sinfo.sassoc_evt.sta_mac, pevt->sta_mac, ETH_ALEN);
		memcpy(&ctx->sinfo.sassoc_evt.bssid, pevt->bssid, ETH_ALEN);
		ctx->sinfo.sassoc_evt.assoc_evt = pevt->assoc_evt;
		ctx->sta_notifier = 1;

		if (pevt->assoc_evt == 0) {
			debug(DEBUG_OFF, "receive disconnect event\n");
			db = get_tpd_db_by_mac(ctx, pevt->sta_mac);
			if (db) {
				debug(DEBUG_OFF, "sta(%02x:%02x:%02x:%02x:%02x:%02x) disconnect to"
					"bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(pevt->sta_mac), PRINT_MAC(pevt->bssid));
				if (!os_memcmp(db->receive_itf_mac, pevt->bssid, ETH_ALEN)) {
					os_memcpy(delete_almac, db->al_mac, ETH_ALEN);
					debug(DEBUG_OFF, "delete topology discovery db\n");
					delete_exist_topology_discovery_database(ctx, db->al_mac, db->itf_mac);
					/*cannot refer db, the memory is free*/
					report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
						ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
				       	ctx->service, &ctx->ap_cap_entry.oper_bss_head,
				        &ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
					if (find_discovery_by_almac(ctx, delete_almac) == NULL) {
						debug(DEBUG_OFF, "LIB_CLIENT_NOTIFICATION "
							"delete topology rsp(%02x:%02x:%02x:%02x:%02x:%02x)\n",
							PRINT_MAC(delete_almac));
						delete_exist_topology_response_database(ctx, delete_almac);
						mark_valid_topo_rsp_node(ctx);
					}
					update_topology_tree(ctx);
				} else {
					debug(DEBUG_ERROR, "bssid not match "
						"bssid(%02x:%02x:%02x:%02x:%02x:%02x)"
						"receive_itf_mac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
						PRINT_MAC(pevt->bssid), PRINT_MAC(db->receive_itf_mac));
				}
			}
		} else {
			detect_neighbor_existence(ctx, NULL, pevt->sta_mac);
		}

		if(update_assoc_sta_info(ctx, pevt) < 0) {
			debug(DEBUG_ERROR, "update station info fail\n");
			break;
		}

		/*send notification with assoc sta info to notify other device my own topology is changed*/
		ctx->mid++;
		for(if_num = 0; if_num < ctx->itf_number; if_num++) {
			insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
	        	e_topology_notification, ctx->mid, ctx->itf[if_num].if_name, 0);
			debug(DEBUG_TRACE, "insert multi notification for %s\n",ctx->itf[if_num].if_name);
#ifdef SUPPORT_CMDU_RELIABLE
			cmdu_reliable_send(ctx, e_topology_notification, ctx->mid, if_num);
#endif
		}

	}
		break;
	case LIB_OPERATING_CHANNEL_REPORT:
		if (0 > update_channel_report_info(ctx,
					(struct channel_report *)wapp_event->buffer, wapp_event->length)) {
			debug(DEBUG_ERROR, "no need to respond this invalid msg\n");
			break;
		}

		debug(DEBUG_TRACE, "send e_operating_channel_report to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->cinfo.almac));
		insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
			e_operating_channel_report, ++ctx->mid, ctx->cinfo.local_ifname, 0);
		break;
	case LIB_BEACON_METRICS_REPORT:
	{
		struct beacon_metrics_rsp *evt = NULL;

		if(ctx->Certification == 1)
		{
			evt = (struct beacon_metrics_rsp *)wapp_event->buffer;

			ctx->metric_entry.bcn_rsp = (struct beacon_metrics_rsp *)malloc(wapp_event->length);
			if (!ctx->metric_entry.bcn_rsp) {
				debug(DEBUG_ERROR, "alloc struct beacon_metrics_rsp fail\n");
				return wapp_utils_error;
			}
			memcpy(ctx->metric_entry.bcn_rsp, evt, wapp_event->length);
		}
		else
		{
			if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
			{
				break;
			}
		}
    	debug(DEBUG_TRACE, "send BEACON_METRICS_REPORT to %02x:%02x:%02x:%02x:%02x:%02x\n",
		 	PRINT_MAC(ctx->cinfo.almac));

        insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
        	e_beacon_metrics_response, ++ctx->mid, ctx->cinfo.local_ifname, 0);
	}
		break;
	case LIB_CLI_STEER_BTM_REPORT:
	{
		struct cli_steer_btm_event *evt = NULL;

		evt = (struct cli_steer_btm_event *)wapp_event->buffer;
		memcpy(ctx->sinfo.sbtm_evt.bssid, evt->bssid, ETH_ALEN);
		memcpy(ctx->sinfo.sbtm_evt.sta_mac, evt->sta_mac, ETH_ALEN);
		ctx->sinfo.sbtm_evt.status = evt->status;
		memcpy(ctx->sinfo.sbtm_evt.tbssid, evt->tbssid, ETH_ALEN);

    	debug(DEBUG_TRACE, "send CLI_STEER_BTM_REPORT to %02x:%02x:%02x:%02x:%02x:%02x\n",
		 	PRINT_MAC(ctx->cinfo.almac));
        insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
        	e_client_steering_btm_report, ++ctx->mid, ctx->cinfo.local_ifname, 0);
	}
		break;
	case LIB_STEERING_COMPLETED:
	{
		debug(DEBUG_TRACE, "send STEERING_COMPLETED to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->cinfo.almac));
		insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
			e_steering_completed, ++ctx->mid, ctx->cinfo.local_ifname, 0);
	}
		break;
	case LIB_BACKHAUL_STEER_RSP:
	{
		if (ctx->Certification == 1) {
			struct backhaul_steer_rsp *steer_rsp = NULL;

			steer_rsp = (struct backhaul_steer_rsp *)wapp_event->buffer;

			memcpy(ctx->bsteer_rsp.backhaul_mac, steer_rsp->backhaul_mac, ETH_ALEN);
			memcpy(ctx->bsteer_rsp.target_bssid, steer_rsp->target_bssid, ETH_ALEN);
			ctx->bsteer_rsp.status = steer_rsp->status;

	    	debug(DEBUG_TRACE, "send BACKHAUL_STEER_RSP to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(ctx->cinfo.almac));
	        insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
	        	e_backhaul_steering_response, ++ctx->mid, ctx->cinfo.local_ifname, 0);
		} else {
			if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
			{
				break;
			}
	    	debug(DEBUG_TRACE, "send BACKHAUL_STEER_RSP to %02x:%02x:%02x:%02x:%02x:%02x\n",
			 	PRINT_MAC(ctx->cinfo.almac));
	        insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
	        	e_backhaul_steering_response, ++ctx->mid, ctx->cinfo.local_ifname, 0);
		}
	}
		break;
	/*below event is mainly from 1905.1 library*/
	case LIB_1905_CMDU_REQUEST:
		{
			struct _1905_cmdu_request* request = (struct _1905_cmdu_request*)wapp_event->buffer;
			debug(DEBUG_TRACE, "[LIB CMDU REQUEST] dest al mac=%02x:%02x:%02x:%02x:%02x:%02x, type=%04x\n",
				PRINT_MAC(request->dest_al_mac), request->type);
			process_1905_request(ctx, request);
		}
		break;
	case LIB_1905_READ_BSS_CONF_REQUEST:
		{
			if (ctx->role != CONTROLLER)
				break;

			if(_1905_read_set_config(ctx, ctx->wts_bss_cfg_file, ctx->bss_config, MAX_SET_BSS_INFO_NUM, &ctx->bss_config_num) < 0)
			{
				debug(DEBUG_ERROR, "load bss config fail from %s\n", ctx->wts_bss_cfg_file);
			}
		}
		break;
	case LIB_1905_READ_BSS_CONF_AND_RENEW:
		{
			unsigned char local_only = 0;
			unsigned char version = 0;

			if (ctx->role != CONTROLLER)
				break;
			local_only = *(wapp_event->buffer);
			version = *(wapp_event->buffer + 1);

			/*read bss info to 1905*/
			if(version == 1) {
				if (read_bss_conf_and_renew(ctx, local_only) < 0) {
					debug(DEBUG_ERROR, "load bss config fail\n");
					break;
				}
			} else {
				if (read_bss_conf_and_renew_v2(ctx, local_only) < 0) {
					debug(DEBUG_ERROR, "load bss config fail\n");
					break;
				}
			}
		}
		break;
	case LIB_1905_READ_1905_TLV_REQUEST:
		{
			char file[64];
			struct _1905_cmdu_request *request = (struct _1905_cmdu_request*)dev_send_1905_buf;

			memset(dev_send_1905_buf, 0, sizeof(dev_send_1905_buf));
			memset(file, 0, sizeof(file));
			memcpy(file, wapp_event->buffer, wapp_event->length);
			if(_1905_read_dev_send_1905(ctx, file, request->dest_al_mac, &(request->type), &(request->len), request->body) < 0)
			{
				debug(DEBUG_ERROR, "parse dev send from %s fail\n", file);
				break;
			}
			dev_send_1905(ctx, request);
		}
		break;
	case LIB_1905_REQ:
		break;
	case LIB_AP_LINK_METRIC_REQ:
	{
		unsigned char i = 0, j = 0, radio_num = 3;
		unsigned char identifier[ETH_ALEN] = {0};

		memcpy(identifier, wapp_event->buffer, ETH_ALEN);
		debug(DEBUG_TRACE, "receive LIB_AP_LINK_METRIC_REQ of radio id(%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(identifier));
		/*query ap metrics response info from wapp*/
		for (i = 0; i < radio_num; i++) {
			if (!memcmp(ctx->rinfo[i].identifier, identifier, ETH_ALEN)) {
				for (j = 0; j < ctx->rinfo[i].bss_number; j++) {
					if (ctx->rinfo[i].bss[j].config_status)
						wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_AP_METRICS_INFO, LIB_AP_METRICS_INFO,
							ctx->rinfo[i].bss[j].mac, NULL, NULL, 0);
				}
				break;
			}
		}
		wapp_get_assoc_sta_traffic_stats(ctx);
		wapp_get_all_assoc_sta_link_metrics(ctx);
		debug(DEBUG_TRACE, "send automatic ap metrics response(ch_report) to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->cinfo.almac));
		insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr, e_ap_metrics_response,
			++ctx->mid, ctx->cinfo.local_ifname, 0);
	}
		break;
	case WAPP_LINK_METRICS_RSP_INFO:
		{
			debug(DEBUG_TRACE, "receive link metrics rsp info\n");
			if(fill_send_tlv(ctx, wapp_event->buffer, wapp_event->length) < 0)
			{
				break;
			}
		}
		break;
	case MANAGEMENT_1905_COMMAND:
		{
			debug(DEBUG_TRACE, "receive MANAGEMENT_1905_COMMAND\n");
			manage_cmd_process(ctx, (struct manage_cmd*)(wapp_event->buffer));
		}
		break;

#ifdef MAP_R2

	case LIB_CAC_CAPAB:
		{
			debug(DEBUG_TRACE, "receive LIB_CAC_CAPAB\n");

			insert_new_cac_capability(ctx, (struct cac_capability_lib *)wapp_event->buffer);
		}
		break;

	case LIB_R2_AP_CAP:
		{
			debug(DEBUG_TRACE, "receive LIB_R2_AP_CAP\n");

			update_r2_ap_capability(ctx, (struct ap_r2_capability *)wapp_event->buffer);
		}
		break;

	case LIB_METRIC_REP_INTERVAL_CAP:
		{
			debug(DEBUG_TRACE, "receive LIB_METRIC_REP_INTERVAL_CAP\n");
			unsigned int *interval = (unsigned int *)wapp_event->buffer;
			ctx->metric_entry.metric_collection_interval = *interval;
			debug(DEBUG_TRACE, "metric_rep_interval is %d\n", *interval);
		}
		break;

	case LIB_AP_EXTENDED_METRICS_INFO:
		{
			debug(DEBUG_TRACE, "receive LIB_AP_EXTENDED_METRICS_INFO\n");

			insert_new_ap_extended_capability(ctx, (struct ap_extended_metrics_lib *)wapp_event->buffer);
		}
		break;

	case LIB_RADIO_METRICS_INFO:
		{
			debug(DEBUG_TRACE, "receive LIB_RADIO_METRICS_INFO\n");

			insert_new_radio_metric(ctx, (struct radio_metrics_lib *)wapp_event->buffer);
		}
		break;

	case LIB_ASSOC_STA_EXTENDED_LINK_METRICS:
		{
			debug(DEBUG_TRACE, "receive LIB_ASSOC_STA_EXTENDED_LINK_METRICS\n");

			insert_assoc_sta_extended_link_metric(ctx, (struct sta_extended_metrics_lib *)wapp_event->buffer);
		}
		break;
#endif // #ifdef MAP_R2

	default:
		debug(DEBUG_ERROR, "unknow command(0x%04x)\n", wapp_event->type);
		break;
	}
	return 0;
}
int parse_combined_infra_metrics_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;

	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == AP_METRICS_TYPE) {
			integrity |= 0x1;
			length = parse_ap_metrics_tlv(ctx, temp_buf, NULL);
			if(length < 0) {
				debug(DEBUG_ERROR, "error parse ap metrics tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == TRANSMITTER_LINK_METRIC_TYPE) {
			integrity |= 0x1;
			length = parse_tx_link_metrics_tlv(ctx, temp_buf, NULL);
			if(length < 0) {
				debug(DEBUG_ERROR, "error tx link metrics tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == RECEIVER_LINK_METRIC_TYPE) {
			integrity |= 0x1;
			length = parse_rx_link_metrics_tlv(ctx, temp_buf, NULL);
			if(length < 0) {
				debug(DEBUG_ERROR, "error tx link metrics tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == ERROR_CODE_TYPE) {
			length = parse_error_code_tlv(temp_buf, ctx);
			if(length < 0) {
				debug(DEBUG_ERROR, "error error code tlv\n");
				return -1;
			}
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no backhaul steering response tlv\n");
		return -1;
	}

	return 0;
}

int change_role_dynamic(struct p1905_managerd_ctx* ctx, unsigned char role)
{
	if (ctx->current_autoconfig_info.radio_index != -1) {
		debug(DEBUG_ERROR, "auto configuration is ongoing; reject this operation\n");
		return -1;
	}


	if (ctx->role == CONTROLLER) {
		debug(DEBUG_ERROR, "cannot set controller to other role\n");
		return -1;
	}

	if (ctx->role == role) {
		debug(DEBUG_ERROR, "already set the role\n");
		return -1;
	}

	ctx->role = role;
	ctx->service = ctx->role;
	debug(DEBUG_OFF, "change role to %s\n", role? "agent":"controller");

	return 0;
}

int get_local_device_info(struct p1905_managerd_ctx* ctx, unsigned char *buf, unsigned short *len)
{
	unsigned char *p = NULL, *old_p = NULL;
	int i = 0;

	p = buf;
	*(unsigned short*)p = MANAGE_GET_LOCAL_DEVINFO;
	p += sizeof(unsigned short);
	old_p = p;

	p += sizeof(unsigned short);
	/*device info*/
	memcpy(p, ctx->p1905_al_mac_addr, ETH_ALEN);
	p += ETH_ALEN;

	*p++ = (unsigned char)(ctx->itf_number - 1);

	for (i = FIRST_VITUAL_ITF + 1; i < ctx->itf_number; i++) {
		memcpy(p, ctx->itf[i].mac_addr, ETH_ALEN);
		p += ETH_ALEN;
		memcpy(p, &(ctx->itf[i].media_type), sizeof(unsigned short));
		p += sizeof(unsigned short);
	}

	/*bridge capa*/
	*p++ = BRIDGE_NUM;
	for (i = 0; i < BRIDGE_NUM; i++) {
		*p++ = os_strlen((char*)ctx->br_name);
		os_strncpy((char*)p, (char*)ctx->br_name, os_strlen((char*)ctx->br_name));
		p += os_strlen((char*)ctx->br_name);
		*p++ = ctx->br_cap[i].interface_num;
		memcpy(p, ctx->br_cap[i].itf_mac_list, ctx->br_cap[i].interface_num * ETH_ALEN);
		p += ctx->br_cap[i].interface_num * ETH_ALEN;
	}

	/*supported service*/
	*p++ = 1;
	*p++ = ctx->service;

	*(unsigned short*)old_p = (unsigned short)(p - old_p - 2);
	*len = (unsigned short)(p - buf);

	return 0;
}

int parse_ap_capability_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == AP_CAPABILITY_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if (length != 1) {
		return -1;
	}

    return (length+3);
}

int parse_ap_ht_capability_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == AP_HT_CAPABILITY_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if (length != 7) {
		return -1;
	}

    return (length+3);
}

int parse_ap_vht_capability_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == AP_VHT_CAPABILITY_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if (length != 12) {
		return -1;
	}

    return (length+3);
}

int parse_ap_he_capability_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == AP_HE_CAPABILITY_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_ap_capability_report_message(struct p1905_managerd_ctx *ctx,
	unsigned char *almac, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0, all_integrity = 0xff;
	unsigned char temp_band = 0;
	unsigned char temp_identifier[ETH_ALEN] = {0};
	unsigned char bss_need_config = 0;
	unsigned char temp_radio_cap = 0, temp_band_cap = 0;
	struct agent_list_db *agent_info = NULL;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		/* One AP Capability TLV */
		if(*temp_buf == AP_CAPABILITY_TYPE)
		{
			length = parse_ap_capability_tlv(ctx, temp_buf);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error ap capability tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		/* One or more AP Radio Basic Capabilities TLV */
		else if(*temp_buf == AP_RADIO_BASIC_CAPABILITY_TYPE) {
			length = parse_ap_radio_basic_cap_tlv(ctx, temp_buf, temp_identifier,
				&bss_need_config, &temp_band);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error ap radio basic cap tlv\n");
				return -1;
			}
			integrity |= (1 << 1);

			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
			if (ctx->role == CONTROLLER) {
				switch (temp_band) {
				case BAND_2G_CAP:
					temp_radio_cap |= RADIO_2G_CAP;
				break;
				case BAND_5GL_CAP:
					temp_radio_cap |= RADIO_5GL_CAP;
				break;
				case BAND_5GH_CAP:
					temp_radio_cap |= RADIO_5GH_CAP;
				break;
				case BAND_5G_CAP:
					temp_radio_cap |= RADIO_5G_CAP;
				break;
				default:
				break;
				}
				temp_band_cap |= temp_band;
				debug(DEBUG_TRACE, "band(%02x) temp_band_cap(%02x) temp_radio_cap(%02x)\n",
					temp_band, temp_band_cap, temp_radio_cap);
			}
		}
		/* Zero or more AP HT Capabilities TLV */
		else if(*temp_buf == AP_HT_CAPABILITY_TYPE) {
			length = parse_ap_ht_capability_tlv(ctx, temp_buf);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error ap ht capability tlv\n");
				return -1;
			}
			integrity |= (1 << 2);

			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		/* Zero or more AP VHT Capabilities TLV */
		else if(*temp_buf == AP_VHT_CAPABILITY_TYPE) {
			length = parse_ap_vht_capability_tlv(ctx, temp_buf);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error ap ht capability tlv\n");
				return -1;
			}

			integrity |= (1 << 3);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
		/* Zero or more AP HE Capabilities TLV */
		else if(*temp_buf == AP_HE_CAPABILITY_TYPE) {
			length = parse_ap_he_capability_tlv(ctx, temp_buf);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error ap ht capability tlv\n");
				return -1;
			}
			integrity |= (1 << 4);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}
#ifdef MAP_R2
		/* One Channel Scan Capabilities TLV */
		else if (*temp_buf == CHANNEL_SCAN_CAPABILITY_TYPE) {
			integrity |= (1 << 5);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* One CAC Capabilities TLV */
		else if (*temp_buf == CAC_CAPABILITIES_TYPE) {
			integrity |= (1 << 6);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* One R2 AP Capability TLV */
		else if (*temp_buf == R2_AP_CAPABILITY_TYPE) {
			integrity |= (1 << 7);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* One Metric Collection Interval TLV */
		else if (*temp_buf == METRIC_COLLECTION_INTERVAL_TYPE) {
			integrity |= (1 << 8);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* skip: One Device 1905 Layer Security Capability TLV
		else if (*temp_buf == METRIC_COLLECTION_INTERVAL_TYPE) {
			integrity |= (1 << 2);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}*/
#endif
		else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if((integrity & all_integrity) == 0) {
		debug(DEBUG_ERROR, "error when check ap capability report tlvs\n");
		return -1;
	}

	/*refill agent radio capability*/
	if (ctx->role == CONTROLLER) {
		find_agent_info(ctx, almac, &agent_info);
		if (!agent_info) {
			debug(DEBUG_ERROR, "error! no agent info exist inset it\n");
			insert_agent_info(ctx, almac);
			find_agent_info(ctx, almac, &agent_info);
		}
		agent_info->radio_cap = temp_radio_cap;
		agent_info->band_cap = temp_band_cap;
		debug(DEBUG_TRACE, "agent(%02x:%02x:%02x:%02x:%02x:%02x) radio_cap(%02x) band_cap(%02x)\n",
			PRINT_MAC(almac), agent_info->radio_cap, agent_info->band_cap);
	}
	return 0;
}

int parse_ch_selection_rsp_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == CH_SELECTION_RESPONSE_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

	if (length != 7) {
		return -1;
	}

    return (length+3);
}

int parse_channel_selection_rsp_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
    unsigned int integrity = 0;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == CH_SELECTION_RESPONSE_TYPE)
		{
			length = parse_ch_selection_rsp_tlv(ctx, temp_buf);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error channel selection rsp tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when channel selection response tlvs\n");
		return -1;
	}
	return 0;
}

int parse_operating_channel_report_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == OPERATING_CHANNEL_REPORT_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_operating_channel_report_message(struct p1905_managerd_ctx *ctx,
		unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;

	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == OPERATING_CHANNEL_REPORT_TYPE)
		{
			length = parse_operating_channel_report_tlv(ctx, temp_buf);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error operating channel report tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check operating channel report tlvs\n");
		return -1;
	}
	return 0;
}

int parse_client_capability_report_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == CLIENT_CAPABILITY_REPORT_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_client_capability_report_message(struct p1905_managerd_ctx *ctx,
		unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	unsigned char temp_bssid[ETH_ALEN] = {0};
	unsigned char temp_sta[ETH_ALEN] = {0};
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == CLIENT_INFO_TYPE)
		{
			length = parse_client_info_tlv(temp_buf, ctx, temp_bssid, temp_sta);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error client info tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if (*temp_buf == CLIENT_CAPABILITY_REPORT_TYPE) {
			length = parse_client_capability_report_tlv(ctx, temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error client capability report tlv\n");
				return -1;
			}
			integrity |= (1 << 1);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
			break;
		} else if (*temp_buf == ERROR_CODE_TYPE) {
			length = parse_error_code_tlv(temp_buf, ctx);
			if(length < 0)
			{
				debug(DEBUG_ERROR, "error in error code tlv\n");
				return -1;
			}

			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
			break;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 3) {
		debug(DEBUG_ERROR, "error when check operating channel report tlvs\n");
		return -1;
	}
	return 0;
}

int parse_unassociated_sta_link_metrics_rsp_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == UNASSOC_STA_LINK_METRICS_RSP_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_unassoc_sta_link_metrics_response_message(struct p1905_managerd_ctx *ctx,
		unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == UNASSOC_STA_LINK_METRICS_RSP_TYPE)
		{
			length = parse_unassociated_sta_link_metrics_rsp_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error client info tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check operating channel report tlvs\n");
		return -1;
	}
	return 0;
}

int parse_beacon_metrics_rsp_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == BEACON_METRICS_RESPONSE_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_beacon_metrics_response_message(struct p1905_managerd_ctx *ctx,
		unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == BEACON_METRICS_RESPONSE_TYPE)
		{
			length = parse_beacon_metrics_rsp_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error beacon metrics response tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check beacon metrics response tlvs\n");
		return -1;
	}
	return 0;
}

int parse_client_steering_btm_report_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == STEERING_BTM_REPORT_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}

int parse_client_steering_btm_report_message(struct p1905_managerd_ctx *ctx,
		unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == STEERING_BTM_REPORT_TYPE)
		{
			length = parse_client_steering_btm_report_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error client steering btm report tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check client steering btm report report tlvs\n");
		return -1;
	}
	return 0;
}

int parse_higher_layer_data_tlv(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == HIGH_LAYER_DATA_TYPE) {
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    return (length+3);
}


int parse_higher_layer_data_message(struct p1905_managerd_ctx *ctx,
		unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == HIGH_LAYER_DATA_TYPE)
		{
			length = parse_higher_layer_data_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error higher layer data tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check higher layer data tlvs\n");
		return -1;
	}
	return 0;
}

int set_bss_config(struct p1905_managerd_ctx *ctx, enum BSS_CONFIG_OPERATION operation,
	struct bss_config_info* info)
{
	unsigned char i = 0, j = 0, k = 0;
	unsigned char band_mask = 0x01;
	struct config_info* conf = NULL;
	char opclass[3][4] = {"8x", "11x", "12x"};
	if (operation == BSS_ADD) {
		j = ctx->bss_config_num;
	}

	conf = info->info;
	for (i = 0; i < info->cnt; i++) {
		for (k = 0; k < 3; k ++) {
			if ((conf->band & (band_mask << k)) == 0x00) {
				continue;
			}
			memset(&(ctx->bss_config[j]), 0, sizeof(struct set_config_bss_info));
			memcpy(ctx->bss_config[j].mac, conf->almac, ETH_ALEN);
			ctx->bss_config[j].authmode = conf->auth_mode;
			ctx->bss_config[j].encryptype = conf->encry_type;
			ctx->bss_config[j].wfa_vendor_extension = conf->wfa_vendor_extension;
			ctx->bss_config[j].hidden_ssid = conf->hidden_ssid;
			memcpy(ctx->bss_config[j].ssid, conf->ssid, 33);
			memcpy(ctx->bss_config[j].key, conf->key, 65);
			memcpy(ctx->bss_config[j].oper_class, opclass[k], 4);
			debug(DEBUG_OFF, "add bss, mac=%02x:%02x:%02x:%02x:%02x:%02x, opclass=%s,"
				" ssid=%s, authmode=%04x, encrytype=%04x, key=%s, bh_bss=%s, fh_bss=%s"
				"hidden_ssid=%d\n",
				PRINT_MAC(ctx->bss_config[j].mac), ctx->bss_config[j].oper_class,
				ctx->bss_config[j].ssid, ctx->bss_config[j].authmode, ctx->bss_config[j].encryptype,
				ctx->bss_config[j].key,
				ctx->bss_config[j].wfa_vendor_extension & BIT_BH_BSS ? "1" : "0",
				ctx->bss_config[j].wfa_vendor_extension & BIT_FH_BSS ? "1" : "0",
				ctx->bss_config[j].hidden_ssid);
			j++;
		}
		conf++;
	}
	ctx->bss_config_num = j;
	debug(DEBUG_OFF, "total bss config number %d\n", ctx->bss_config_num);

	return 0;
}

void build_ap_metric_query_tlv(struct p1905_managerd_ctx *ctx)
{
	unsigned char buf[512] = {0}, bssid[512] = {0};
	unsigned char i = 0, j = 0, bssid_num = 0;
	unsigned short length = 0;
	unsigned char *pos = buf, *bss_pos = bssid;

#ifdef MAP_R2
	struct radio_identifier_db *p_tmp = NULL;

	unsigned char identifier[512] = {0};
	unsigned char id_total_len = 0;
	unsigned char *identifier_pos = identifier;
#endif
	reset_stored_tlv(ctx);

	for (i = 0; i < MAX_RADIO_NUM; i++) {
		for (j = 0; j < ctx->rinfo[i].bss_number; j++) {
			if (ctx->rinfo[i].bss[j].config_status) {
				memcpy(bss_pos, ctx->rinfo[i].bss[j].mac, ETH_ALEN);
				bss_pos += ETH_ALEN;
				bssid_num++;
			}
		}
	}

	/*tlvType*/
	*pos++ = AP_METRICS_QUERY_TYPE;
	/*tlvLength*/
	length = 1 + ETH_ALEN * bssid_num;
	*(unsigned short *)(pos) = cpu2be16(length);
	pos += 2;
	/*Number of BSSIDs included in this TLV*/
	*pos++ = bssid_num;
	/*BSSID of a BSS*/
	memcpy(pos, bssid, length-1);

	store_revd_tlvs(ctx, buf, length + 3);
#ifdef MAP_R2
	/* Zero or more AP Radio Identifier TLVs */
	if(!SLIST_EMPTY(&ctx->metric_entry.radio_identifier_head))
	{
		p_tmp = SLIST_FIRST(&ctx->metric_entry.radio_identifier_head);
		while (p_tmp)
		{
			debug(DEBUG_TRACE, "radio identifier(%02x:%02x:%02x:%02x:%02x:%02x) \n", PRINT_MAC(p_tmp->identifier));

			/*tlvType*/
			*identifier_pos++ = AP_RADIO_IDENTIFIER_TYPE;
			/*tlvLength*/
			length = ETH_ALEN;
			*(unsigned short *)(identifier_pos) = cpu2be16(length);
			identifier_pos += 2;
			/*radio identifier*/
			memcpy(identifier_pos, p_tmp->identifier, ETH_ALEN);

			id_total_len += (length + 3);
			p_tmp = SLIST_NEXT(p_tmp, radio_identifier_entry);
		}
	}
	store_revd_tlvs(ctx, identifier, id_total_len);
#endif // MAP_R2
}

void report_own_topology_rsp(struct p1905_managerd_ctx *ctx,
	unsigned char *al_mac, struct bridge_capabiltiy *br_cap_list,
	struct p1905_neighbor *p1905_dev, struct non_p1905_neighbor *non_p1905_dev,
	unsigned char service, struct list_head_oper_bss *opbss_head,
    struct list_head_assoc_clients *asscli_head, unsigned int cnt)
{
    unsigned char *tlv_temp_buf = tlv_temp;
    unsigned short length = 0;
    unsigned short total_tlvs_length = 0;
    int i = 0;
    unsigned char seperate = 0;

    length = append_device_info_type_tlv(tlv_temp_buf, al_mac, ctx);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    length = append_device_bridge_capability_type_tlv(tlv_temp_buf, br_cap_list);
    total_tlvs_length += length;
    tlv_temp_buf += length;

    for (i=0; i<ctx->itf_number; i++) {
       do {
	       	length = append_p1905_neighbor_device_type_tlv(tlv_temp_buf, p1905_dev, i, &seperate);
	        total_tlvs_length += length;
	        tlv_temp_buf += length;
	      } while (seperate);
    }

    seperate = 0;
    /*append non-p1905.1 device tlv*/
    for (i=0; i<ctx->itf_number; i++) {
        do {
            length = append_non_p1905_neighbor_device_type_tlv(tlv_temp_buf,
                     non_p1905_dev,i, &seperate);
            total_tlvs_length += length;
            tlv_temp_buf += length;
        } while (seperate);
    }

	length = append_supported_service_tlv(tlv_temp_buf, service);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_operational_bss_tlv(tlv_temp_buf, opbss_head);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_associated_clients_tlv(tlv_temp_buf, asscli_head, cnt);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	if (ctx->Certification == 0) {
		if (total_tlvs_length > 10240) {
			debug(DEBUG_ERROR, "send length(%d) exceeds max revd_tlv len\n",
				total_tlvs_length);
			return;
		}
		reset_stored_tlv(ctx);
		store_revd_tlvs(ctx, tlv_temp, total_tlvs_length);
		os_get_time(&ctx->own_topo_rsp_update_time);
		_1905_notify_topology_rsp_event(ctx, ctx->p1905_al_mac_addr, ctx->p1905_al_mac_addr);
	}
}

int read_bss_conf_and_renew_v2(struct p1905_managerd_ctx *ctx, unsigned char local_only) {
	/*update bss setting into ctx->bss_config*/
	debug(DEBUG_OFF, "read wts_bss_info_config to ctx->bss_config, radio_number=%d\n",
		ctx->radio_number);
	if (_1905_read_set_config(ctx, ctx->wts_bss_cfg_file, ctx->bss_config, MAX_SET_BSS_INFO_NUM, &ctx->bss_config_num) < 0 ||
		(ctx->bss_config_num == 0)) {
		debug(DEBUG_ERROR, "read wts_bss_info_config fail or no valid bss info\n");
		return -1;
	}

	ctx->trigger_renew = 1;
	ctx->renew_state = wait_4_dump_topo;
	eloop_register_timeout(0, 0, controller_renew_bss_step, (void *)ctx, NULL);

	return 0;
}


int read_bss_conf_and_renew(struct p1905_managerd_ctx *ctx, unsigned char local_only)
{
	int i = 0;
	unsigned char band = 0;
	unsigned char buffer[128] = {0};

	/*update bss setting into ctx->bss_config*/
	debug(DEBUG_OFF, "read wts_bss_info_config to ctx->bss_config, radio_number=%d\n",
		ctx->radio_number);
	if (_1905_read_set_config(ctx, ctx->wts_bss_cfg_file, ctx->bss_config, MAX_SET_BSS_INFO_NUM, &ctx->bss_config_num) < 0 ||
		(ctx->bss_config_num == 0)) {
		debug(DEBUG_ERROR, "read wts_bss_info_config fail or no valid bss info\n");
		return -1;
	}

	 for (i = 0; i < ctx->radio_number; i++) {
		_1905_update_bss_info_per_radio(ctx, &ctx->rinfo[i]);
	 }


	if (!local_only) {
			band = 0;
			ctx->mid++;
			if (fill_send_tlv(ctx, &band, 1) < 0) {
				return 0;
			}

			for(i = 0; i < ctx->itf_number; i++) {
				insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
					e_ap_autoconfiguration_renew, ctx->mid, ctx->itf[i].if_name, 0);
				debug(DEBUG_TRACE, "insert renew message for %s band(%s)\n",ctx->itf[i].if_name, "2g");
#ifdef SUPPORT_CMDU_RELIABLE
				cmdu_reliable_send(ctx, e_ap_autoconfiguration_renew, ctx->mid, i);
#endif
			}
			/*handle the txq before insert another band renew message*/
			process_cmdu_txq(ctx, buffer);

			band = 1;
			ctx->mid++;
			if (fill_send_tlv(ctx, &band, 1) < 0) {
				return 0;
			}

			for(i = 0; i < ctx->itf_number; i++) {
				insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
					e_ap_autoconfiguration_renew, ctx->mid, ctx->itf[i].if_name, 0);
				debug(DEBUG_TRACE, "insert renew message for %s band(%s)\n",ctx->itf[i].if_name, "5g");
#ifdef SUPPORT_CMDU_RELIABLE
				cmdu_reliable_send(ctx, e_ap_autoconfiguration_renew, ctx->mid, i);
#endif
			}
			process_cmdu_txq(ctx, buffer);
	}
	return 0;
}

unsigned char determin_band_config(struct p1905_managerd_ctx *ctx, unsigned char *almac,
	unsigned char band_cap)
{
	unsigned char i = 0;
	unsigned char bss_cap = 0;
	unsigned char wild_card_mac[ETH_ALEN] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

	for(i = 0; i < ctx->bss_config_num; i++) {
		if(!memcmp(almac, ctx->bss_config[i].mac, ETH_ALEN) ||
		   !memcmp(wild_card_mac, ctx->bss_config[i].mac, ETH_ALEN)) {
			if (ctx->bss_config[i].oper_class[0] == '8') {
				bss_cap |= BSS_2G_CAP;
			} else if (ctx->bss_config[i].oper_class[1] == '1') {
				bss_cap |= BSS_5GL_CAP;
			} else if (ctx->bss_config[i].oper_class[1] == '2') {
				bss_cap |= BSS_5GH_CAP;
			} else {
				debug(DEBUG_ERROR, "unknown oper_class\n");
			}
		}
	}
	debug(DEBUG_OFF, "mac(%02x:%02x:%02x:%02x:%02x:%02x) bsscap = %d\n",
		PRINT_MAC(almac), bss_cap);

	if (band_cap == BAND_2G_CAP && ((bss_cap & BSS_2G_CAP) == BSS_2G_CAP)) {
		return BAND_2G;
	} else if (band_cap == BAND_5G_CAP && ((bss_cap & BSS_5G_CAP) == BSS_5G_CAP)) {
		return BAND_5GL;
	} else if (band_cap == BAND_5G_CAP && ((bss_cap & BSS_5G_CAP) == BSS_5GL_CAP)) {
		return BAND_5GL;
	} else if (band_cap == BAND_5G_CAP &&((bss_cap & BSS_5G_CAP) == BSS_5GH_CAP)) {
		return BAND_5GH;
	} else if (band_cap == BAND_5GL_CAP &&((bss_cap & BSS_5GL_CAP) == BSS_5GL_CAP)) {
		return BAND_5GL;
	} else if (band_cap == BAND_5GH_CAP &&((bss_cap & BSS_5GH_CAP) == BSS_5GH_CAP)) {
		return BAND_5GH;
	} else {
		debug(DEBUG_ERROR, "bandcap=%d, bsscap=%d cannot decide which band to config\n",
			band_cap, bss_cap);
		return BAND_UNKNOWN;
	}
}

WIFI_UTILS_STATUS config_bss_by_band(struct p1905_managerd_ctx *ctx, unsigned char band,
	WSC_CONFIG *wsc, unsigned char *wfa_vendor_extension)
{
	unsigned char i = 0;
	unsigned char wild_card_mac[ETH_ALEN] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

	memset(wsc, 0, sizeof(WSC_CONFIG));
	for (i = ctx->current_bss_config_index; i < ctx->bss_config_num; i++) {
		debug(DEBUG_INFO, "current_bss_config_index=%d, bss_config_num=%d\n",
			ctx->current_bss_config_index, ctx->bss_config_num);
		debug(DEBUG_INFO, "enrolle_mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->ap_config.enrolle_mac));
		debug(DEBUG_INFO, "bss_mac=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->bss_config[i].mac));
		debug(DEBUG_INFO, "band=%d, oper_class=%s\n",band, ctx->bss_config[i].oper_class);
		/*2.4g/5g band*/
		if (!memcmp(ctx->ap_config.enrolle_mac, ctx->bss_config[i].mac, ETH_ALEN) &&
			((band == BAND_2G && ctx->bss_config[i].oper_class[0] == '8')
			|| (band == BAND_5GL && ctx->bss_config[i].oper_class[1] == '1')
			|| (band == BAND_5GH && ctx->bss_config[i].oper_class[1] == '2'))) {
			ctx->current_bss_config_index = i;
			ctx->current_bss_config_index++;
			debug(DEBUG_TRACE, "current_bss_config_index=%d, ssid=%s\n", i, ctx->bss_config[i].ssid);
			break;
		}

		if (!memcmp(ctx->bss_config[i].mac, wild_card_mac, ETH_ALEN) &&
			((band == BAND_2G && ctx->bss_config[i].oper_class[0] == '8')
			|| (band == BAND_5GL && ctx->bss_config[i].oper_class[1] == '1')
			|| (band == BAND_5GH && ctx->bss_config[i].oper_class[1] == '2'))) {
			ctx->current_bss_config_index = i;
			ctx->current_bss_config_index++;
			debug(DEBUG_TRACE, "wild card mac match, index=%d, ssid=%s", i, ctx->bss_config[i].ssid);
			break;
		}

	}

	if (i >= ctx->bss_config_num) {
		if (ctx->bss_success_config == 0) {
			debug(DEBUG_ERROR, "[%d]can not get any wsc config(almac=%02x:%02x:%02x:%02x:%02x:%02x), no bss config from file\n",
				i, PRINT_MAC(ctx->ap_config.enrolle_mac));
			debug(DEBUG_ERROR, "error!!! cannot happen!!! no bss info for band=%d send tear down bit in m2\n", band);
			*wfa_vendor_extension = 0x10;
			goto fail;
		} else {
			debug(DEBUG_ERROR, "no enough bss info for band=%d not add extra m2 now\n", band);
			goto fail;
		}
	}

	memcpy(wsc->Ssid, ctx->bss_config[i].ssid, 32);
	wsc->EncrypType = ctx->bss_config[i].encryptype;
	wsc->AuthMode = ctx->bss_config[i].authmode;
	memcpy(wsc->WPAKey, ctx->bss_config[i].key, 64);
	*wfa_vendor_extension = ctx->bss_config[i].wfa_vendor_extension;
	wsc->hidden_ssid = ctx->bss_config[i].hidden_ssid;
	ctx->bss_success_config++;
	return wifi_utils_success;
fail:
	return wifi_utils_error;
}

void init_radio_info_by_intf(struct p1905_managerd_ctx* ctx, struct p1905_interface *itf)
{
	int i = 0, j = 0, k = 0;

	for (i = 0; i < ctx->radio_number; i++) {
		if (!memcmp(ctx->rinfo[i].identifier, itf->identifier, ETH_ALEN)) {
			for (j = 0; j < ctx->rinfo[i].bss_number; j++) {
				if (!memcmp(ctx->rinfo[i].bss[j].ifname, itf->if_name, IFNAMSIZ)) {
					break;
				}
			}
			if (j >= ctx->rinfo[i].bss_number) {
				/*new bss, should add new one*/
				memcpy(ctx->rinfo[i].bss[ctx->rinfo[i].bss_number].ifname, itf->if_name, IFNAMSIZ);
				memcpy(ctx->rinfo[i].bss[ctx->rinfo[i].bss_number].mac, itf->mac_addr, ETH_ALEN);
				ctx->rinfo[i].bss[ctx->rinfo[i].bss_number].priority= 0;
				debug(DEBUG_TRACE, "add new bss(%s) to existed radio(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					itf->if_name, PRINT_MAC(itf->identifier));
				ctx->rinfo[i].bss_number++;
			}
			break;
		}
	}
	/*new radio, add new one*/
	if (i >= ctx->radio_number) {
		memcpy(ctx->rinfo[ctx->radio_number].identifier, itf->identifier, ETH_ALEN);
		ctx->rinfo[ctx->radio_number].bss_number = 1;
		ctx->rinfo[ctx->radio_number].band = itf->radio_band;
		debug(DEBUG_TRACE, "add new radio(%02x:%02x:%02x:%02x:%02x:%02x)\n", PRINT_MAC(itf->identifier));
		memcpy(ctx->rinfo[ctx->radio_number].bss[0].ifname, itf->if_name, IFNAMSIZ);
		memcpy(ctx->rinfo[ctx->radio_number].bss[0].mac, itf->mac_addr, ETH_ALEN);
		ctx->rinfo[ctx->radio_number].bss[0].priority= 0;

		debug(DEBUG_TRACE, "add new bss(%s) to above new radio\n", ctx->rinfo[ctx->radio_number].bss[0].ifname);
		ctx->radio_number++;
	}

	/*set the priority to the bss that will be configured*/
	for (i = 0; i < ctx->radio_number; i++) {
		if (!os_memcmp(ctx->rinfo[i].identifier, itf->identifier, ETH_ALEN)) {
			for (j = 0; j < ctx->rinfo[i].bss_number; j++) {
				for (k = 0; k < ctx->itf_number; k++) {
					/*if (os_strcmp((char*)ctx->rinfo[i].bss[j].ifname, (char*)ctx->itf[k].if_name) == 0) {*/
					if (os_memcmp((char*)ctx->rinfo[i].bss[j].mac, (char*)ctx->itf[k].mac_addr, ETH_ALEN) == 0) {
						ctx->rinfo[i].bss[j].priority = ctx->itf[k].config_priority;
					}
				}
			}
			break;
		}
	}
}

void metrics_report_timeout(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx *)eloop_data;

	if (ctx->Certification == 1) {
		unsigned char i = 0, j = 0, radio_num = 3;

		for (i = 0; i < radio_num; i++) {
			for (j = 0; j < ctx->rinfo[i].bss_number; j++) {
				if (ctx->rinfo[i].bss[j].config_status)
					wapp_get_info_by_msgtype(ctx, WAPP_USER_GET_AP_METRICS_INFO, LIB_AP_METRICS_INFO,
						ctx->rinfo[i].bss[j].mac, NULL, NULL, 0);
			}
		}
		wapp_get_assoc_sta_traffic_stats(ctx);
		wapp_get_all_assoc_sta_link_metrics(ctx);
		debug(DEBUG_TRACE, "send automatic ap metrics response(report_interval) to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->cinfo.almac));
		insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr, e_ap_metrics_response,
			++ctx->mid, ctx->cinfo.local_ifname, 0);
	} else {
		/*build ap metric query tlv*/
		ctx->rcv_apmetrics_query = 1;
		debug(DEBUG_TRACE, "send automatic ap metrics response(report_interval) to %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(ctx->cinfo.almac));
		build_ap_metric_query_tlv(ctx);
		if (_1905_notify_ap_metrics_query(ctx, ctx->cinfo.almac
#ifdef MAP_R2
		, 1
#endif
		) < 0) {
			debug(DEBUG_ERROR, "notify ap metrics query fail\n");
			ctx->rcv_apmetrics_query = 0;
		} else {
			insert_cmdu_txq(ctx->cinfo.almac, ctx->p1905_al_mac_addr,
				e_ap_metrics_response, ++ctx->mid, ctx->cinfo.local_ifname, 0);
			ctx->rcv_apmetrics_query = 0;
		}
	}

	common_process(ctx, ctx->rx_buf);

	eloop_register_timeout(ctx->map_policy.mpolicy.report_interval, 0,
		metrics_report_timeout, eloop_data, NULL);
}

unsigned short create_vs_info_for_specific_discovery(unsigned char *vs_info)
{
	/*1905 internal using vendor specific tlv format
	   type			1 byte			0x11
	   length			2 bytes			0x, 0x
	   oui			3 bytes			0x00, 0x0C, 0xE7
	   function type	1 byte			0xff
	   suboui			3 bytes			0x00, 0x0C, 0xE7
	   sub func type	1 byte			0x
	*/
	unsigned char *p = vs_info;
	unsigned short len = 0;
	unsigned char mtk_oui[3] = {0x00, 0x0C, 0xE7};

	/*vendor specific tlv type*/
	*p++ = 11;

	/*vendor specific tlv length*/
	*((unsigned short*)p) = cpu2be16(8);
	p += 2;

	/*oui*/
	os_memcpy(p, mtk_oui, sizeof(mtk_oui));
	p += 3;

	/*function type*/
	*p++ = 0xff;

	/*additional OUI*/
	os_memcpy(p, mtk_oui, sizeof(mtk_oui));
	p += 3;

	/*1905 internal used vendor specific subtype*/
	*p++ = internal_vendor_discovery_message_subtype;

	len = (unsigned short)(p - vs_info);

	return len;
}

void discovery_at_apcli_link_up(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_data;
	unsigned char *if_name = (unsigned char *)user_ctx;
	static int i = 3;

	/*send multicast cmdu to all the ethenet interface */
	debug(DEBUG_TRACE, "send topology discovery with vs by interface[%s]\n", if_name);
	insert_cmdu_txq(p1905_multicast_address, ctx->p1905_al_mac_addr,
		e_topology_discovery_with_vendor_ie, ++ctx->mid, if_name, 0);

	i--;

	if (i == 0) {
		i = 3;
		return;
	}

	eloop_register_timeout(3, 0, discovery_at_apcli_link_up, (void *)ctx, (void *)if_name);

	return;
}

void attach_action(void *eloop_data, void *user_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_data;
	struct topology_discovery_db *tpg_db = NULL;
	struct topology_response_db *tpgr_db = NULL;
	static int i = 2;

	report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
				ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
				ctx->service, &ctx->ap_cap_entry.oper_bss_head,
				&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);


    LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry) {
		insert_cmdu_txq(tpg_db->al_mac, ctx->p1905_al_mac_addr,
			e_topology_query, ++ctx->mid, ctx->br_name, 0);
    }

	SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry) {
		insert_cmdu_txq(tpgr_db->al_mac_addr, ctx->p1905_al_mac_addr,
			e_topology_query, ++ctx->mid, ctx->br_name, 0);
	}

	i--;

	if (i == 0) {
		i = 2;
		return;
	}

	eloop_register_timeout(1, 0, attach_action, (void *)ctx, NULL);
}


/*int check_send_rsp(struct p1905_managerd_ctx *ctx)
{
	int i = 0;
	unsigned char zero_mac[ETH_ALEN] = {0};
	struct topology_discovery_db *tpg_db = NULL;

	if (ctx->role == CONTROLLER)
		return 1;

	for (i = 1; i < ctx->itf_number; i++) {
		if ((ctx->itf[i].is_wifi_sta == 1) &&
			os_memcmp(ctx->itf[i].vs_info, zero_mac, ETH_ALEN)) {
			LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
			{
				if (!os_memcmp(tpg_db->itf_mac, ctx->itf[i].vs_info, ETH_ALEN))
					break;
			}
			if (!tpg_db) {
				debug(DEBUG_OFF, "apcli(%02x:%02x:%02x:%02x:%02x:%02x) "
					"connect to bssid(%02x:%02x:%02x:%02x:%02x:%02x)\n"
					"but not update neighbor! cannot tx rsp\n",
					PRINT_MAC(ctx->itf[i].mac_addr),
					PRINT_MAC(ctx->itf[i].vs_info));
				return 0;
			}
		}
	}

	return 1;
}


int check_valid_neighbor_rsp(struct p1905_managerd_ctx *ctx,
	struct topology_response_db *tpgr_db)
{
    struct device_info_db *dev = NULL;
	struct p1905_neighbor_device_db *neighbor = NULL;
	int rcv_id = ctx->recent_cmdu_rx_if_number;
	int i = 0, found_neighbor = 0, found_dev = 0;
	unsigned char zero_mac[ETH_ALEN] = {0};

	SLIST_FOREACH(dev, &tpgr_db->devinfo_head, devinfo_entry)
	{
		SLIST_FOREACH(neighbor, &dev->p1905_nbrdb_head, p1905_nbrdb_entry)
		{
			if (!os_memcmp(neighbor->p1905_neighbor_al_mac, ctx->p1905_al_mac_addr,
					ETH_ALEN)) {
				for (i = 1; i < ctx->itf_number; i++) {
					if ((ctx->itf[i].is_wifi_sta == 1) &&
						!os_memcmp(ctx->itf[i].vs_info, dev->mac_addr, ETH_ALEN))
						break;
				}
				if (i >= ctx->itf_number)
					return 0;
				break;
			}
		}
	}

	for (i = 1; i < ctx->itf_number; i++) {
		found_neighbor = 0;
		found_dev = 0;
		if ((ctx->itf[i].is_wifi_sta == 1) &&
			os_memcmp(ctx->itf[i].vs_info, zero_mac, ETH_ALEN)) {
			SLIST_FOREACH(dev, &tpgr_db->devinfo_head, devinfo_entry)
			{
				SLIST_FOREACH(neighbor, &dev->p1905_nbrdb_head, p1905_nbrdb_entry)
				{
					if (!os_memcmp(neighbor->p1905_neighbor_al_mac,
							ctx->p1905_al_mac_addr, ETH_ALEN)) {
							found_neighbor = 1;
							break;
					}
				}
				if (found_neighbor) {
					if (!os_memcmp(dev->mac_addr, ctx->itf[i].vs_info, ETH_ALEN)) {
						found_dev = 1;
						break;
					}
				}
			}
			if (!found_dev)
				return 0;
		}
	}

	return 1;
}

 struct topology_response_db * find_neighbor_topology_rsp(
 	struct p1905_managerd_ctx *ctx, unsigned char *al_mac)
{
	struct topology_discovery_db *tpg_db = NULL;
    struct topology_response_db *tpgr_db = NULL;

	LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
	{
		if (!os_memcmp(tpg_db->al_mac, al_mac ETH_ALEN))
			break;
    }
	if (!tpg_db) {
		return NULL;
	}

	SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry)
	{
		if (!os_memcmp(tpgr_db->al_mac_addr, al_mac, ETH_ALEN))
			return tpgr_db;
	}

	return NULL;
}

int update_own_neighbor_info_by_rsp(struct p1905_managerd_ctx *ctx,
	struct topology_response_db *tpgr_db)
{
	struct topology_discovery_db *tpg_db = NULL, *tpg_db_tmp;
    struct device_info_db *dev = NULL;
	struct p1905_neighbor_device_db *neighbor = NULL;
	unsigned char report = 0;

	LIST_FOREACH(tpg_db, &ctx->topology_entry.tpddb_head, tpddb_entry)
	{
		if (!os_memcmp(tpg_db->al_mac, tpgr_db->al_mac_addr, ETH_ALEN)) {
			SLIST_FOREACH(dev, &tpgr_db->devinfo_head, devinfo_entry)
			{
				SLIST_FOREACH(neighbor, &dev->p1905_nbrdb_head, p1905_nbrdb_entry)
				{
					if (!os_memcmp(neighbor->p1905_neighbor_al_mac,
							ctx->p1905_al_mac_addr, ETH_ALEN) &&
						!os_memcmp(dev->mac_addr, tpg_db->itf_mac, ETH_ALEN)) {
						break;
					}
				}
			}
			if (!dev) {
				tpg_db_tmp = LIST_NEXT(tpg_db, tpddb_entry);
				delete_p1905_neighbor_dev_info(ctx, tpg_db->al_mac,
					(unsigned char *)tpg_db->receive_itf_mac);
				LIST_REMOVE(tpg_db, tpddb_entry);
				free(tpg_db);
				tpg_db = tpg_db_tmp;
				report = 1;
			}
		}
	}

	if (report)
		report_own_topology_rsp(ctx, ctx->p1905_al_mac_addr, ctx->br_cap,
			ctx->p1905_neighbor_dev, ctx->non_p1905_neighbor_dev,
			ctx->service, &ctx->ap_cap_entry.oper_bss_head,
			&ctx->ap_cap_entry.assoc_clients_head, ctx->cnt);
}*/

unsigned short create_vs_info_for_specific_notification(
		struct p1905_managerd_ctx *ctx, unsigned char *vs_info)
{
	/*1905 internal using vendor specific tlv format
	   type 		1 byte			0x11
	   length			2 bytes 		0x, 0x
	   oui			3 bytes 		0x00, 0x0C, 0xE7
	   function type	1 byte			0xff
	   suboui			3 bytes 		0x00, 0x0C, 0xE7
	   sub func type	1 byte			0x01
	   sub len			2 byte
	   sub value		n byte
	*/
	unsigned char *p = vs_info;
	unsigned short len = 0;
	unsigned char mtk_oui[3] = {0x00, 0x0C, 0xE7};

	/*vendor specific tlv type*/
	*p++ = 11;

	/*vendor specific tlv length*/
	*((unsigned short*)p) = cpu2be16(16);
	p += 2;

	/*oui*/
	os_memcpy(p, mtk_oui, sizeof(mtk_oui));
	p += 3;

	/*function type*/
	*p++ = 0xff;

	/*additional OUI*/
	os_memcpy(p, mtk_oui, sizeof(mtk_oui));
	p += 3;

	/*1905 internal used vendor specific subtype*/
	*p++ = internal_vendor_notification_message_subtype;
	*((unsigned short*)p) = cpu2be16(ETH_ALEN);
	p += 2;
	os_memcpy(p, ctx->send_tlv, ETH_ALEN);
	p += ETH_ALEN;

	len = (unsigned short)(p - vs_info);

	return len;
}

void reset_controller_connect_ifname(struct p1905_managerd_ctx *ctx,
	unsigned char *link_down_ifname)
{
	int i = 0;

	if (ctx->role == CONTROLLER)
		return;

	if (os_memcmp(link_down_ifname, ctx->cinfo.local_ifname, IFNAMSIZ))
		return;

	for (i = 0; i < ctx->itf_number; i++) {
		if (ctx->itf[i].is_wifi_sta &&
			os_memcmp(ctx->itf[i].if_name, ctx->cinfo.local_ifname, IFNAMSIZ) &&
			ctx->itf[i].trx_config & (TX_MUL | TX_UNI)) {
			debug(DEBUG_OFF, "local_ifname[%s] link down use another one[%s]\n",
				ctx->cinfo.local_ifname, ctx->itf[i].if_name);
			os_memcpy(ctx->cinfo.local_ifname, ctx->itf[i].if_name, IFNAMSIZ);
			return;
		}
	}
}

#ifdef MAP_R2

/***channel scan***/
int delete_exist_channel_scan_capability(
	struct p1905_managerd_ctx *ctx, unsigned char* identifier)
{
	 struct channel_scan_cap_db *scan_cap = NULL;
	 struct scan_opcap_db *scap = NULL, *scap_tmp = NULL;
	 int i = 0;

	if(!SLIST_EMPTY(&(ctx->ap_cap_entry.ch_scan_cap_head)))
    {
        SLIST_FOREACH(scan_cap, &(ctx->ap_cap_entry.ch_scan_cap_head), ch_scan_cap_entry)
        {
            if(!memcmp(scan_cap->identifier, identifier, ETH_ALEN))
            {
                if(!SLIST_EMPTY(&(scan_cap->scan_opcap_head)))
                {
                    scap = SLIST_FIRST(&(scan_cap->scan_opcap_head));
                    while(scap)
                    {
                    	debug(DEBUG_TRACE, "delete_exist struct scan_opcap_db\n");
						debug(DEBUG_TRACE, "opclass:%d, ch_num=%d\n", scap->op_class, scap->ch_num);
						debug(DEBUG_TRACE, "ch_list: \n");
						for (i = 0; i < scap->ch_num; i++)
							debug(DEBUG_TRACE, "%d ", scap->ch_list[i]);
						debug(DEBUG_TRACE, "\n");
                        scap_tmp = SLIST_NEXT(scap, scan_opcap_entry);
                        SLIST_REMOVE(&scan_cap->scan_opcap_head, scap,
                                    scan_opcap_db, scan_opcap_entry);
                        free(scap);
                        scap = scap_tmp;
                    }
                }
				debug(DEBUG_TRACE, "delete_exist struct channel_scan_cap_db\n");
				debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(scan_cap->identifier));
				debug(DEBUG_TRACE, "boot_only:%d, scan_impact:%d, min_scan_interval:%d op_class_num:%d\n",
					scan_cap->boot_only, scan_cap->scan_impact, scan_cap->min_scan_interval,
					scan_cap->op_class_num);
                SLIST_REMOVE(&(ctx->ap_cap_entry.ch_scan_cap_head), scan_cap,
                            channel_scan_cap_db, ch_scan_cap_entry);
                free(scan_cap);
                break;
            }
        }
    }

	return wapp_utils_success;
}

int insert_new_channel_scan_capability(
	struct p1905_managerd_ctx *ctx, struct scan_capability_lib *scap)
{
	struct channel_scan_cap_db *scap_db = NULL;
	struct scan_opcap_db *opcap_db = NULL;
	struct scan_opcap *opcap = NULL;
	int op_class_num = 0;
	int i = 0, j=0;

	scap_db = (struct channel_scan_cap_db *)malloc(sizeof(*scap_db));
	memcpy(scap_db->identifier, scap->identifier, ETH_ALEN);
	scap_db->boot_only = scap->boot_only;
	scap_db->scan_impact = scap->scan_impact;
	scap_db->min_scan_interval = scap->min_scan_interval;
	scap_db->op_class_num = scap->op_class_num;
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ch_scan_cap_head), scap_db, ch_scan_cap_entry);

	debug(DEBUG_TRACE, "insert struct channel_scan_cap_db\n");
	debug(DEBUG_TRACE, "identifier(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(scap_db->identifier));
	debug(DEBUG_TRACE, "boot_only:%d, scan_impact:%d, min_scan_interval:%d op_class_num:%d\n",
		scap_db->boot_only, scap_db->scan_impact, scap_db->min_scan_interval,
		scap_db->op_class_num);

	SLIST_INIT(&scap_db->scan_opcap_head);
	op_class_num = scap_db->op_class_num;
	for(i = 0; i < op_class_num; i++) {
		opcap = &scap->opcap[i];
		opcap_db = (struct scan_opcap_db *)malloc(sizeof(*opcap_db));
		memcpy(opcap_db, opcap, sizeof(*opcap));
		SLIST_INSERT_HEAD(&scap_db->scan_opcap_head, opcap_db, scan_opcap_entry);

		debug(DEBUG_TRACE, "insert struct scan_opcap_db\n");
		debug(DEBUG_TRACE, "opclass:%d, ch_num=%d\n", opcap_db->op_class, opcap_db->ch_num);
		debug(DEBUG_TRACE, "ch_list: \n");
		for (j = 0; j < opcap_db->ch_num; j++)
			debug(DEBUG_TRACE, "%d ", opcap_db->ch_list[j]);
		debug(DEBUG_TRACE, "\n");
	}

	return wapp_utils_success;

}

unsigned short append_channel_scan_capability_tlv(
        unsigned char *pkt, struct list_head_ch_scan_cap *shead)
{
    unsigned short total_length = 0, tlv_len = 0;;
    unsigned char *temp_buf = NULL;
	struct channel_scan_cap_db *pdb_scan = NULL;
	struct scan_opcap_db *pdb_op = NULL;
	unsigned char rnum = 0;

    temp_buf = pkt;

    *temp_buf++ = CHANNEL_SCAN_CAPABILITY_TYPE;
	/*skip tlv length field*/
    temp_buf += 2;
	/*skip radio number field*/
	temp_buf += 1;

	SLIST_FOREACH(pdb_scan, shead, ch_scan_cap_entry)
	{
		debug(DEBUG_TRACE, "pdb_scan->identifier=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(pdb_scan->identifier));
		rnum++;
		/*radio identifier*/
		memcpy(temp_buf, pdb_scan->identifier, ETH_ALEN);
		temp_buf += ETH_ALEN;
		/*boot only flag & scan impact*/
		*temp_buf++ = (pdb_scan->boot_only << 7) | (pdb_scan->scan_impact << 5);
		/*minimum scan interval*/
		*((unsigned int *)temp_buf) = cpu2be32(pdb_scan->min_scan_interval);
		temp_buf += 4;
		/*number of operating classes*/
		*temp_buf++ = pdb_scan->op_class_num;
		SLIST_FOREACH(pdb_op, &pdb_scan->scan_opcap_head, scan_opcap_entry)
		{
			/*operating class*/
			*temp_buf++ = pdb_op->op_class;
			/*number of channels*/
			*temp_buf++ = pdb_op->ch_num;
			/*channel list*/
			if (pdb_op->ch_num)
				memcpy(temp_buf, pdb_op->ch_list, pdb_op->ch_num);
			temp_buf += pdb_op->ch_num;
		}
	}

	/*number of radios*/
	*(pkt + 3) = rnum;

	total_length = (unsigned short)(temp_buf - pkt);
	tlv_len = total_length - 3;
	*((unsigned short *)(pkt + 1)) = cpu2be16(tlv_len);

    return total_length;
}


/***dfs cac***/
int delete_exist_cac_capability(
	struct p1905_managerd_ctx *ctx, unsigned char* identifier)
{
	struct radio_cac_capability_db *p_radio_cac_cap_db = NULL;
	struct cac_capability_db *p_cac_capab_db = NULL, *p_cac_capab_db_tmp = NULL;
	struct cac_cap_db *p_cac_cap_db = NULL, *p_cac_cap_db_tmp = NULL;
	struct cac_opcap_db *p_cac_opcab_db = NULL, *p_cac_opcab_db_tmp = NULL;


	p_radio_cac_cap_db = &ctx->ap_cap_entry.radio_cac_cap;

	if(!SLIST_EMPTY(&p_radio_cac_cap_db->radio_cac_capab_head))
	{
		//SLIST_FOREACH(p_cac_capab_db, &p_radio_cac_cap_db->radio_cac_capab_head, cac_capab_entry)
		p_cac_capab_db = SLIST_FIRST(&(p_radio_cac_cap_db->radio_cac_capab_head));
		while (p_cac_capab_db)
		{
			p_cac_capab_db_tmp = SLIST_NEXT(p_cac_capab_db, cac_capab_entry);
			if(!memcmp(p_cac_capab_db->identifier, identifier, ETH_ALEN))
			{
				debug(DEBUG_TRACE, "delete existed cac_capability for %02x:%02x:%02x:%02x:%02x:%02x\n",
					PRINT_MAC(p_cac_capab_db->identifier));

				if(!SLIST_EMPTY(&(p_cac_capab_db->cac_capab_head)))
				{
					p_cac_cap_db = SLIST_FIRST(&(p_cac_capab_db->cac_capab_head));
					while(p_cac_cap_db)
					{
						p_cac_cap_db_tmp = SLIST_NEXT(p_cac_cap_db, cac_cap_entry);

						/*
						debug(DEBUG_TRACE, "delete cac_cap_db\n");
						debug(DEBUG_TRACE, "cac_mode:%d\n", p_cac_cap_db->cac_mode);
						*/

						if(!SLIST_EMPTY(&(p_cac_cap_db->cac_opcap_head)))
						{
							p_cac_opcab_db = SLIST_FIRST(&(p_cac_cap_db->cac_opcap_head));
							while(p_cac_opcab_db) {
								/*
								debug(DEBUG_TRACE, "delete cac_opcap_db\n");
								debug(DEBUG_TRACE, "opclass:%d, ch_num=%d\n", p_cac_opcab_db->op_class, p_cac_opcab_db->ch_num);
								debug(DEBUG_TRACE, "ch_list: \n");
								*/
								p_cac_opcab_db_tmp = SLIST_NEXT(p_cac_opcab_db, cac_opcap_entry);
								SLIST_REMOVE(&(p_cac_cap_db->cac_opcap_head), p_cac_opcab_db,
									cac_opcap_db, cac_opcap_entry);

								free(p_cac_opcab_db);
								p_cac_opcab_db = p_cac_opcab_db_tmp;
							}
						}

						SLIST_REMOVE(&(p_cac_capab_db->cac_capab_head), p_cac_cap_db,
							cac_cap_db, cac_cap_entry);
						free(p_cac_cap_db);
						p_cac_cap_db = p_cac_cap_db_tmp;
					}
				}

				SLIST_REMOVE(&(p_radio_cac_cap_db->radio_cac_capab_head), p_cac_capab_db,
					cac_capability_db, cac_capab_entry);
				free(p_cac_capab_db);
			}
			p_cac_capab_db = p_cac_capab_db_tmp;
		}
	}

	return wapp_utils_success;
}



int insert_new_cac_capability(
	struct p1905_managerd_ctx *ctx, struct cac_capability_lib *cac_capab_in)
{
	struct radio_cac_capability_db *p_radio_cac_cap_db = &ctx->ap_cap_entry.radio_cac_cap;
	struct cac_capability_db *p_cac_capab_db = NULL;
	struct cac_cap_db *p_cac_cap_db = NULL;
	struct cac_opcap_db *p_cac_opcap_db = NULL;
	struct cac_cap *p_cac_cap_in = NULL;
	struct cac_type *p_cac_type_in = NULL;
	struct cac_opcap *p_cac_opcap_in = NULL;
	int i = 0, j=0, k = 0, n = 0;
	int offset_type = 0, offset_opcap = 0;
	char ch_str[64] = {0}, offset_str = 0;

	if (!ctx || !cac_capab_in)
		return wapp_utils_error;

	memcpy(p_radio_cac_cap_db->country_code, cac_capab_in->country_code, 2);
	p_radio_cac_cap_db->radio_num = cac_capab_in->radio_num;
	debug(DEBUG_TRACE, "insert cac cap \n");

	debug(DEBUG_TRACE, "  country_code %s, radio_num %d\n",
		p_radio_cac_cap_db->country_code, p_radio_cac_cap_db->radio_num);

	p_cac_cap_in = cac_capab_in->cap;
	for (i = 0; i < p_radio_cac_cap_db->radio_num; i ++) {

		delete_exist_cac_capability(ctx, p_cac_cap_in->identifier);

		p_cac_capab_db = (struct cac_capability_db *)malloc(sizeof(struct cac_capability_db));
		if (!p_cac_capab_db)
			goto error;

		memcpy(p_cac_capab_db->identifier, p_cac_cap_in->identifier, ETH_ALEN);
		p_cac_capab_db->cac_type_num = p_cac_cap_in->cac_type_num;


		debug(DEBUG_TRACE, "    new identifier %02x:%02x:%02x:%02x:%02x:%02x, cac_type_num %d\n",
			PRINT_MAC(p_cac_capab_db->identifier), p_cac_capab_db->cac_type_num);

		SLIST_INIT(&p_cac_capab_db->cac_capab_head);

		SLIST_INSERT_HEAD(&(p_radio_cac_cap_db->radio_cac_capab_head), p_cac_capab_db, cac_capab_entry);

		p_cac_type_in = p_cac_cap_in->type;
		offset_type = 0;
		for (j = 0; j < p_cac_capab_db->cac_type_num; j ++) {
			p_cac_cap_db = (struct cac_cap_db *)malloc(sizeof(struct cac_cap_db));
			if (!p_cac_cap_db)
				goto error;

			p_cac_cap_db->cac_mode = p_cac_type_in->cac_mode;
			memcpy(p_cac_cap_db->cac_interval, p_cac_type_in->cac_interval, sizeof(p_cac_cap_db->cac_interval));
			p_cac_cap_db->op_class_num = p_cac_type_in->op_class_num;

			debug(DEBUG_TRACE, "      cac_mode %d, cac_interval 0x%02x,0x%02x,0x%02x, op_class_num %d\n",
				p_cac_cap_db->cac_mode, p_cac_cap_db->cac_interval[0], p_cac_cap_db->cac_interval[1],
				p_cac_cap_db->cac_interval[2], p_cac_cap_db->op_class_num);


			SLIST_INIT(&p_cac_cap_db->cac_opcap_head);

			SLIST_INSERT_HEAD(&p_cac_capab_db->cac_capab_head, p_cac_cap_db, cac_cap_entry);

			p_cac_opcap_in = p_cac_type_in->opcap;
			offset_opcap = 0;
			for (k = 0; k < p_cac_cap_db->op_class_num; k ++) {
				p_cac_opcap_db = (struct cac_opcap_db *)malloc(sizeof(struct cac_opcap_db));
				if (!p_cac_opcap_db)
					goto error;

				p_cac_opcap_db->op_class = p_cac_opcap_in->op_class;
				p_cac_opcap_db->ch_num = p_cac_opcap_in->ch_num;

				memcpy(p_cac_opcap_db->ch_list, p_cac_opcap_in->ch_list, p_cac_opcap_db->ch_num);

				debug(DEBUG_TRACE, "        op_class %d, ch_num %d\n",
					p_cac_opcap_db->op_class, p_cac_opcap_db->ch_num);

				offset_str = 0;
				memset(ch_str, 0, sizeof(ch_str));
				for (n = 0; n < p_cac_opcap_db->ch_num; n ++)
					offset_str += sprintf(ch_str + offset_str, "%02x,", p_cac_opcap_db->ch_list[n]);

				debug(DEBUG_TRACE, "          channel %s\n", ch_str);

				SLIST_INSERT_HEAD(&p_cac_cap_db->cac_opcap_head, p_cac_opcap_db, cac_opcap_entry);

				p_cac_opcap_in = (struct cac_opcap *)((char *)p_cac_opcap_in + sizeof(struct cac_opcap) + p_cac_opcap_db->ch_num);
				offset_opcap += sizeof(struct cac_opcap) + p_cac_opcap_db->ch_num;

			}

			p_cac_type_in = (struct cac_type *)((char *)p_cac_type_in + sizeof(struct cac_type) + offset_opcap);
			offset_type += sizeof(struct cac_type) + offset_opcap;
		}

		p_cac_cap_in = (struct cac_cap *)((char *)p_cac_cap_in + sizeof(struct cac_cap) + offset_type);
	}

	return wapp_utils_success;
error:
	return wapp_utils_error;

}

unsigned short append_cac_capability_tlv(
        unsigned char *pkt, struct radio_cac_capability_db *cachead)
{
	unsigned short total_length = 0, tlv_len = 0;;
	unsigned char *temp_buf = NULL;
	struct cac_capability_db *p_cac_capab = NULL;
	struct cac_cap_db *p_cac_cab = NULL;
	struct cac_opcap_db *p_cac_cab_opcap = NULL;

	temp_buf = pkt;

	*temp_buf++ = CAC_CAPABILITIES_TYPE;
	/*skip tlv length field*/
	temp_buf += 2;

	/* country code, 2 octets */
	memcpy(temp_buf, cachead->country_code, sizeof(cachead->country_code));
	temp_buf += 2;

	/* Number of radios, 1 octet */
	*temp_buf++ = cachead->radio_num;


	SLIST_FOREACH(p_cac_capab, &cachead->radio_cac_capab_head, cac_capab_entry)
	{
		/*radio identifier, 6 octets*/
		memcpy(temp_buf, p_cac_capab->identifier, ETH_ALEN);
		debug(DEBUG_TRACE, "identifier=%02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(p_cac_capab->identifier));
		temp_buf += ETH_ALEN;

		/*Number of types of CAC, 1 octet*/
		*temp_buf++ = p_cac_capab->cac_type_num;
		debug(DEBUG_TRACE, "cac_type_num %d\n", p_cac_capab->cac_type_num);

		SLIST_FOREACH(p_cac_cab, &p_cac_capab->cac_capab_head, cac_cap_entry)
		{
			/* CAC mode supported, 1 octet */
			*temp_buf++ = p_cac_cab->cac_mode;

			/* Number of seconds required to complete CAC, 3 octets */
			memcpy(temp_buf, p_cac_cab->cac_interval, 3);
			temp_buf += 3;

			/* Number of classes supported, 1 octet */
			*temp_buf++ = p_cac_cab->op_class_num;

			debug(DEBUG_TRACE, "  cac_mode %d\n", p_cac_cab->cac_mode);
			debug(DEBUG_TRACE, "  cac_interval 0x%02x,0x%02x,0x%02x\n",
				p_cac_cab->cac_interval[0], p_cac_cab->cac_interval[1], p_cac_cab->cac_interval[2]);
			debug(DEBUG_TRACE, "  op_class_num %d\n", p_cac_cab->op_class_num);
			SLIST_FOREACH(p_cac_cab_opcap, &p_cac_cab->cac_opcap_head, cac_opcap_entry)
			{
				/* perating class, 1 octet */
				*temp_buf++ = p_cac_cab_opcap->op_class;

				/* perating class, 1 octet */
				*temp_buf++ = p_cac_cab_opcap->ch_num;

				/*channel number, */
				memcpy(temp_buf, p_cac_cab_opcap->ch_list, p_cac_cab_opcap->ch_num);

				debug(DEBUG_TRACE, "    op_class %d\n", p_cac_cab_opcap->op_class);
				debug(DEBUG_TRACE, "    ch_num %d\n", p_cac_cab_opcap->ch_num);

				temp_buf += p_cac_cab_opcap->ch_num;
			}

		}

	}

	total_length = (unsigned short)(temp_buf - pkt);
	tlv_len = total_length - 3;
	*((unsigned short *)(pkt + 1)) = cpu2be16(tlv_len);

    return total_length;
}


int update_r2_ap_capability(
	struct p1905_managerd_ctx *ctx, struct ap_r2_capability *ap_r2_cap)
{
	debug(DEBUG_TRACE, "receive LIB_R2_AP_CAP\n");

	memcpy(&ctx->ap_cap_entry.ap_r2_cap, ap_r2_cap, sizeof(struct ap_r2_capability));

	debug(DEBUG_TRACE, "\tmax_total_num_sp_rules:%d\n\tbyte_counter_units:%d\n\tmax_total_num_vid %d\n",
		ap_r2_cap->max_total_num_sp_rules, ap_r2_cap->byte_counter_units, ap_r2_cap->max_total_num_vid);

	debug(DEBUG_TRACE, "\tbasic_sp_flag:%d\n\tenhanced_sp_flag:%d\n",
		ap_r2_cap->basic_sp_flag, ap_r2_cap->enhanced_sp_flag);

	return wapp_utils_success;

}



int insert_new_ap_extended_capability(
	struct p1905_managerd_ctx *ctx, struct ap_extended_metrics_lib *metric)
{
	struct ap_ext_cap_db *p_new_db = NULL;

	p_new_db = (struct ap_ext_cap_db *)malloc(sizeof(struct ap_ext_cap_db));
	if (p_new_db == NULL) {
		debug(DEBUG_ERROR, "malloc ap_ext_cap_db failed\n");
		return wapp_utils_error;
	}
	memcpy(p_new_db->bssid, metric->bssid, ETH_ALEN);
	p_new_db->uc_tx = metric->uc_tx;
	p_new_db->uc_rx = metric->uc_rx;
	p_new_db->mc_tx = metric->mc_tx;
	p_new_db->mc_rx = metric->mc_rx;
	p_new_db->bc_tx = metric->bc_tx;
	p_new_db->bc_rx = metric->bc_rx;
	SLIST_INSERT_HEAD(&(ctx->ap_cap_entry.ap_ext_cap_head), p_new_db, ap_ext_cap_entry);

	debug(DEBUG_TRACE, "insert new ap extended capability for %02x:%02x:%02x:%02x:%02x:%02x\n",
		PRINT_MAC(metric->bssid));
	debug(DEBUG_TRACE, "uc tx %d, uc rx %d, mc tx %d, mc rx %d, bc tx %d, bc rx %d\n",
		metric->uc_tx, metric->uc_rx, metric->mc_tx, metric->mc_rx, metric->mc_tx, metric->mc_rx);


	return wapp_utils_success;

}




int insert_new_radio_metric(
	struct p1905_managerd_ctx *ctx, struct radio_metrics_lib *metric)
{
	struct radio_metrics_db *p_new_db = NULL;

	p_new_db = (struct radio_metrics_db *)malloc(sizeof(struct radio_metrics_db));
	if (p_new_db == NULL) {
		debug(DEBUG_ERROR, "malloc radio_metrics_db failed\n");
		return wapp_utils_error;
	}
	memcpy(p_new_db->identifier, metric->identifier, ETH_ALEN);
	p_new_db->noise= metric->noise;
	p_new_db->transmit= metric->transmit;
	p_new_db->receive_self= metric->receive_self;
	p_new_db->receive_other= metric->receive_other;
	SLIST_INSERT_HEAD(&(ctx->metric_entry.radio_metrics_head), p_new_db, radio_metrics_entry);

	debug(DEBUG_TRACE, "insert new ap radio metric for %02x:%02x:%02x:%02x:%02x:%02x\n",
		PRINT_MAC(metric->identifier));
	debug(DEBUG_TRACE, "noise %d, transmit %d, receive_self %d, receive_other %d\n",
		metric->noise, metric->transmit, metric->receive_self, metric->receive_other);


	return wapp_utils_success;

}


int delete_assoc_sta_extended_link_metric(
	struct p1905_managerd_ctx *ctx, struct sta_extended_metrics_lib *metric)
{
	struct sta_extended_metrics_db *p_tmp = NULL, *p_next = NULL;
	struct extended_metrics_db *p_tmp_metric = NULL, * p_next_metric = NULL;

	/* rm exsited extended_metrics_db for the sta */
	if(!SLIST_EMPTY(&ctx->metric_entry.assoc_sta_extended_link_metrics_head))
	{
		p_tmp = SLIST_FIRST(&ctx->metric_entry.assoc_sta_extended_link_metrics_head);
		while (p_tmp)
		{
			p_next = SLIST_NEXT(p_tmp, sta_extended_metrics_entry);
			if(!memcmp(p_tmp->sta_mac, metric->sta_mac, ETH_ALEN)) {
				p_tmp_metric = SLIST_FIRST(&(p_tmp->extended_metrics_head));
				while(p_tmp_metric)
				{
					p_next_metric = SLIST_NEXT(p_tmp_metric, metrics_entry);

					SLIST_REMOVE(&(p_tmp->extended_metrics_head), p_tmp_metric,
						extended_metrics_db, metrics_entry);
					free(p_tmp_metric);
					p_tmp_metric = p_next_metric;
				}
			}

			SLIST_REMOVE(&ctx->metric_entry.assoc_sta_extended_link_metrics_head, p_tmp,
				sta_extended_metrics_db, sta_extended_metrics_entry);
			free(p_tmp);
			p_tmp = p_next;
		}

	}
	return wapp_utils_success;
}



int insert_assoc_sta_extended_link_metric(
	struct p1905_managerd_ctx *ctx, struct sta_extended_metrics_lib *metric)
{
	struct sta_extended_metrics_db *p_new_db = NULL;
	struct extended_metrics_db *p_new_metric_db = NULL;
	struct extended_metrics_info *p_metric_in = NULL;
	int i = 0;

	delete_assoc_sta_extended_link_metric(ctx, metric);

	/* insert  */
	p_new_db = (struct sta_extended_metrics_db *)malloc(sizeof(struct sta_extended_metrics_db));
	if (p_new_db == NULL) {
		debug(DEBUG_ERROR, "malloc sta_extended_metrics_db failed\n");
		return wapp_utils_error;
	}
	memcpy(p_new_db->sta_mac, metric->sta_mac, ETH_ALEN);
	p_new_db->extended_metric_cnt = metric->extended_metric_cnt;

	SLIST_INSERT_HEAD(&(ctx->metric_entry.assoc_sta_extended_link_metrics_head),
		p_new_db, sta_extended_metrics_entry);

	debug(DEBUG_TRACE, "insert new assoc sta link metric for %02x:%02x:%02x:%02x:%02x:%02x\n",
		PRINT_MAC(metric->sta_mac));

	p_metric_in = metric->metric_info;
	while (i < metric->extended_metric_cnt) {
		p_new_metric_db = (struct extended_metrics_db *)malloc(sizeof(struct extended_metrics_db));
		if (p_new_metric_db == NULL) {
			debug(DEBUG_ERROR, "malloc new_metric_db failed\n");
			return wapp_utils_error;
		}

		memcpy(p_new_metric_db->bssid, p_metric_in->bssid, ETH_ALEN);
		p_new_metric_db->last_data_ul_rate = p_metric_in->last_data_ul_rate;
		p_new_metric_db->last_data_dl_rate = p_metric_in->last_data_dl_rate;
		p_new_metric_db->utilization_rx = p_metric_in->utilization_rx;
		p_new_metric_db->utilization_tx= p_metric_in->utilization_tx;

		debug(DEBUG_TRACE, "insert new extended link metric for bssid %02x:%02x:%02x:%02x:%02x:%02x\n",
			PRINT_MAC(p_new_metric_db->bssid));

		SLIST_INSERT_HEAD(&(p_new_db->extended_metrics_head),
			p_new_metric_db, metrics_entry);

		p_metric_in += 1;
		i ++;
	}


	return wapp_utils_success;

}




int parse_channel_scan_request_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;

	reset_stored_tlv(ctx);

	while(1) {
		if (*temp_buf == CHANNEL_SCAN_REQUEST_TYPE) {
			integrity |= 0x1;
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x1) {
		debug(DEBUG_ERROR, "no channel scan request tlv\n");
		return -1;
	}

	return 0;
}

int parse_channel_scan_report_message(struct p1905_managerd_ctx *ctx,
	unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned char integrity = 0;

	temp_buf = buf;

	reset_stored_tlv(ctx);
	hex_dump_all("ScanRep", buf,100);
	while(1) {
		if (*temp_buf == TIMESTAMP_TYPE) {
			integrity |= 0x1;
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		} else if (*temp_buf == CHANNEL_SCAN_RESULT_TYPE) {
			integrity |= 0x2;
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		} else if (*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
	/*check integrity*/
	if(integrity != 0x3) {
		debug(DEBUG_ERROR, "incomplete channel scan report 0x%x 0x3\n",
			integrity);
		return -1;
	}

	return 0;
}

unsigned short channel_scan_request_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CHANNEL_SCAN_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}

unsigned short channel_scan_report_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CHANNEL_SCAN_REPORT);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}



unsigned short tunneled_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(TUNNELED_MESSAGE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;
}



unsigned short assoc_status_notification_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	/*for this kind of message, maybe we'll send more than one(for relayed multicast)
	*so we cannot call append_send_tlv which will set send_tlv_len to 0
	*/
	length = append_send_tlv_relayed(tlv_temp_buf, ctx);
		total_tlvs_length += length;
		tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(ASSOCIATION_STATUS_NOTIFICATION);
	msg_hdr->relay_indicator = 0x1;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;

}


unsigned short cac_request_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CAC_REQUEST);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;
}


unsigned short cac_terminate_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CAC_TERMINATION);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;
}


unsigned short client_disassciation_stats_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(CLIENT_DISASSOCIATION_STATS);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;
}


unsigned short failed_association_message(unsigned char *buf,
 	struct p1905_managerd_ctx *ctx)
{
	unsigned char *tlv_temp_buf = tlv_temp;
	cmdu_message_header *msg_hdr;
	unsigned short length = 0;
	unsigned short total_tlvs_length =0;

	msg_hdr = (cmdu_message_header*)buf;

	length = append_send_tlv(tlv_temp_buf, ctx);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	length = append_end_of_tlv(tlv_temp_buf);
	total_tlvs_length += length;
	tlv_temp_buf += length;

	/*tlvs size is less than (46(minimun ethernet frame payload size)
	 *-8(cmdu header size)) ==>padding
	 */
	if(total_tlvs_length < MIN_TLVS_LENGTH)
	{
		memset(tlv_temp_buf, 0, (MIN_TLVS_LENGTH - total_tlvs_length));
		total_tlvs_length = MIN_TLVS_LENGTH;
	}

	//0x00: for this version of the specification
	//0x01~0xFF: Reserved Values
	msg_hdr->message_version = 0x0;

	//set reserve field to 0
	msg_hdr->reserved_field_0 = 0x0;
	msg_hdr->message_type = cpu2be16(FAILED_ASSOCIATION_MESSAGE);
	msg_hdr->relay_indicator = 0x0;
	//set reserve field to 0
	msg_hdr->reserve_field_1 = 0x0;

	return total_tlvs_length;
}




int delete_exist_traffic_policy(struct p1905_managerd_ctx *ctx, struct traffics_policy *tpolicy)
{
	struct traffic_separation_db *policy = NULL, *policy_tmp = NULL;

	policy = SLIST_FIRST(&tpolicy->policy_head);
	while(policy != NULL) {
		debug(DEBUG_OFF, "SSID=%s, vlan_id=%d\n",policy->SSID, policy->vid);
		free(policy->SSID);
		policy_tmp = SLIST_NEXT(policy, policy_entry);
		free(policy);
		policy = policy_tmp;
	}
	SLIST_INIT(&tpolicy->policy_head);
	tpolicy->SSID_num= 0;

	return wapp_utils_success;
}

int parse_traffic_separation_policy_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0, i =0;
	struct traffic_separation_db *trffic_policy = NULL;
	struct traffics_policy *policy = &ctx->map_policy.tpolicy;
	int total_tlv_length = 0;
	unsigned char ssidbuf[64] = {0};

	temp_buf = buf;

	if((*temp_buf) == TRAFFIC_SEPARATION_POLICY_TYPE) {
		temp_buf++;
		total_tlv_length++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;
	total_tlv_length += (2+length);

	policy->SSID_num = *temp_buf++;
	debug(DEBUG_ERROR, "SSID_num(%d)\n", policy->SSID_num);
	for (i = 0; i < policy->SSID_num; i++) {
		trffic_policy = (struct traffic_separation_db *)malloc(sizeof(struct traffic_separation_db));
		if (!trffic_policy) {
			debug(DEBUG_ERROR, "alloc struct traffic_separation_db fail\n");
			return -1;
		}
		trffic_policy->SSID_len = *temp_buf++;
		debug(DEBUG_ERROR, "SSID_len(%d)\n", trffic_policy->SSID_len);

		trffic_policy->SSID = malloc(trffic_policy->SSID_len);
		if (!trffic_policy->SSID) {
			debug(DEBUG_ERROR, "kmalloc for trffic_policy->SSID fail\n");
			return -1;
		}
		memcpy(trffic_policy->SSID, temp_buf, trffic_policy->SSID_len);
		memset(ssidbuf, 0, 64);
		memcpy(ssidbuf, trffic_policy->SSID, trffic_policy->SSID_len);
		temp_buf += trffic_policy->SSID_len;

		trffic_policy->vid = *((unsigned short*)temp_buf);
		trffic_policy->vid = be2cpu16(trffic_policy->vid);
		temp_buf += 2;
		debug(DEBUG_ERROR, "parse:%s, %d\n", ssidbuf, trffic_policy->vid);
		SLIST_INSERT_HEAD(&policy->policy_head, trffic_policy, policy_entry);
	}

	return total_tlv_length;
}

int delete_exist_pfilter_policy(struct p1905_managerd_ctx *ctx, struct pfiltering_policy *fpolicy)
{
	struct pf_bssid_list *bssid = NULL, *bssid_tmp = NULL;
	struct dest_mac_list *dmac = NULL, *dmac_tmp = NULL;

	bssid = SLIST_FIRST(&fpolicy->bssid_head);
	while (bssid != NULL) {
		dmac = SLIST_FIRST(&bssid->dmac_head);
		while (dmac != NULL) {
			dmac_tmp = SLIST_NEXT(dmac, dmac_entry);
			free(dmac);
			dmac = dmac_tmp;
		}
		SLIST_INIT(&bssid->dmac_head);
		bssid->n = 0;

		bssid_tmp = SLIST_NEXT(bssid, bssid_entry);
		free(bssid);
		bssid = bssid_tmp;
	}
	SLIST_INIT(&fpolicy->bssid_head);
	fpolicy->k= 0;

	return wapp_utils_success;
}

int parse_packet_filtering_policy_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0, i = 0, j = 0;
	struct pf_bssid_list *pfBSSID = NULL;
	struct dest_mac_list *dmac = NULL;

	struct pfiltering_policy *fpolicy = &ctx->map_policy.fpolicy;
	int total_tlv_length = 0;

	temp_buf = buf;
	if ((*temp_buf) == PACKET_FILTERING_POLICY_TYPE) {
		temp_buf++;
		total_tlv_length++;
	} else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;
	total_tlv_length += (2+length);

	fpolicy->k = *temp_buf++;
	debug(DEBUG_ERROR, "pkt filtering policy, k=%d\n", fpolicy->k);
	for (i = 0; i < fpolicy->k; i++) {
		pfBSSID = (struct pf_bssid_list *)malloc(sizeof(struct pf_bssid_list));
		if (!pfBSSID) {
			debug(DEBUG_ERROR, "alloc struct pf_bssid_list fail\n");
			return -1;
		}
		memcpy(pfBSSID->bssid, temp_buf, ETH_ALEN);
		temp_buf += ETH_ALEN;
		pfBSSID->n = *temp_buf++;

		debug(DEBUG_ERROR, "BSSID:%02x:%02x:%02x:%02x:%02x:%02x, n=%d\n",
			pfBSSID->bssid[0], pfBSSID->bssid[1], pfBSSID->bssid[2],
			pfBSSID->bssid[3], pfBSSID->bssid[4], pfBSSID->bssid[5], pfBSSID->n);

		SLIST_INSERT_HEAD(&fpolicy->bssid_head, pfBSSID, bssid_entry);

		for (j = 0; j < pfBSSID->n; j ++) {
			dmac = (struct dest_mac_list *)malloc(sizeof(struct dest_mac_list));
			if (!dmac) {
				debug(DEBUG_ERROR, "alloc struct dest_mac_list fail\n");
				return -1;
			}
			memcpy(dmac->dest_addr, temp_buf, ETH_ALEN);
			temp_buf += ETH_ALEN;
			debug(DEBUG_ERROR, "\t\t dmac[%d]:%02x:%02x:%02x:%02x:%02x:%02x\n", j,
				dmac->dest_addr[0], dmac->dest_addr[1], dmac->dest_addr[2],
				dmac->dest_addr[3], dmac->dest_addr[4], dmac->dest_addr[5]);

			SLIST_INSERT_HEAD(&pfBSSID->dmac_head, dmac, dmac_entry);
		}
	}

	return total_tlv_length;
}

int parse_default_802_1q_setting_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0;
	struct def_8021q_setting *setting = &ctx->map_policy.setting;
	int total_tlv_length = 0;

	temp_buf = buf;

	if((*temp_buf) == DEFAULT_8021Q_SETTING_TYPE) {
		temp_buf++;
		total_tlv_length++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;
	total_tlv_length += (2+length);

	memcpy(setting, temp_buf, sizeof(*setting));
	setting->primary_vid = be2cpu16(setting->primary_vid);
	debug(DEBUG_ERROR, "primary_vid:%d pcp:%d, length=%d\n",
				setting->primary_vid,setting->dft.PCP, length);

	return total_tlv_length;
}

int delete_exist_eth_config_policy(struct p1905_managerd_ctx *ctx, struct ethernets_policy *epolicy)
{
	struct ethernet_config_db *policy = NULL, *policy_tmp = NULL;

	policy = SLIST_FIRST(&epolicy->policy_head);
	while(policy != NULL) {
		debug(DEBUG_TRACE, "intfid=%pM, type=%d\n",policy->intfid, policy->eth.type);
		policy_tmp = SLIST_NEXT(policy, policy_entry);
		free(policy);
		policy = policy_tmp;
	}
	SLIST_INIT(&epolicy->policy_head);
	epolicy->inf_num= 0;

	return wapp_utils_success;
}

int parse_eth_config_policy_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0, i =0;
	struct ethernet_config_db *eth_policy = NULL;
	struct ethernets_policy *policy = &ctx->map_policy.epolicy;
	int total_tlv_length = 0;

	temp_buf = buf;

	if((*temp_buf) == ETHERNET_CONFIGURATION_POLICY_TYPE) {
		temp_buf++;
		total_tlv_length++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;
	total_tlv_length += (2+length);

	policy->inf_num = *temp_buf++;
	debug(DEBUG_TRACE, "inf_num(%d)\n", policy->inf_num);
	for (i = 0; i < policy->inf_num; i++) {
		eth_policy = (struct ethernet_config_db *)malloc(sizeof(struct ethernet_config_db));
		if (!eth_policy) {
			debug(DEBUG_ERROR, "alloc struct ethernet_config_db fail\n");
			return -1;
		}
		memcpy(eth_policy->intfid, temp_buf, ETH_ALEN);
		debug(DEBUG_TRACE, "intfid(%pM)\n", eth_policy->intfid);

		temp_buf += ETH_ALEN;

		memcpy(&eth_policy->eth, temp_buf, sizeof(eth_policy->eth));
		temp_buf += 1;

		SLIST_INSERT_HEAD(&policy->policy_head, eth_policy, policy_entry);
	}

	return total_tlv_length;
}


int parse_unsuccess_assoc_policy_tlv(unsigned char *buf,
	struct p1905_managerd_ctx *ctx)
{
	unsigned char *temp_buf;
	unsigned short length = 0;
	struct unsuccess_assoc_policy *policy = &ctx->map_policy.unsuccess_assoc_policy;
	int total_tlv_length = 0;
	unsigned int report_rate = 0;

	temp_buf = buf;

	if((*temp_buf) == UNSUCCESSFUL_ASSOCIATION_POLICY_TYPE) {
		temp_buf++;
		total_tlv_length++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	//shift to tlv value field
	temp_buf += 2;
	total_tlv_length += (2+length);

	if ((*(temp_buf++)) & 0x80) {
		policy->report_switch = 1;
	}
	else
		policy->report_switch = 0;

	report_rate = (*((unsigned int *)temp_buf));

	policy->report_rate = be2cpu32(report_rate);


	return total_tlv_length;
}

#if 0
void mapfilter_set_pfilter_policy(struct p1905_managerd_ctx *ctx, struct pfiltering_policy *fpolicy)
{
	unsigned char buf[520] = {0};
	unsigned char *msgbuf;
	struct nlmsg_cmd *msg = NULL;
	struct pf_bssid_list *pfBSSID = NULL;
	struct dest_mac_list *dmac = NULL;

	msg = (struct nlmsg_cmd *)buf;
	msg->cmd = NLMSG_PERMMIT_MAC;
	msg->num = fpolicy->k;
	msg->bufsize = 0;
	msgbuf = msg->buf;

	SLIST_FOREACH(pfBSSID, &(fpolicy->bssid_head), bssid_entry) {
		memcpy(msgbuf, pfBSSID->bssid, ETH_ALEN);
		msgbuf += ETH_ALEN;
		msg->bufsize += ETH_ALEN;

		memcpy(msgbuf, &pfBSSID->n, 1);
		msgbuf += 1;
		msg->bufsize += 1;

		msg->bufsize += pfBSSID->n*ETH_ALEN;
		SLIST_FOREACH(dmac, &(pfBSSID->dmac_head), dmac_entry) {
			memcpy(msgbuf, dmac->dest_addr, ETH_ALEN);
			msgbuf += ETH_ALEN;
		}
	}
	send_msg_to_mapfilter(msg);

}

void mapfilter_set_ethernet_policy(struct p1905_managerd_ctx *ctx, struct ethernets_policy *epolicy)
{
	unsigned char buf[280] = {0};
	unsigned char *msgbuf;
	struct nlmsg_cmd *msg = NULL;
	struct ethernet_config_db *policy_db = NULL;
	struct intf_type_info inf_info = {0};

	if (!epolicy->inf_num)
		return ;

	msg = (struct nlmsg_cmd *)buf;
	msg->cmd = NLMSG_ETH_INTF;
	msg->num = epolicy->inf_num;
	msg->bufsize = msg->num * sizeof(struct intf_type_info);
	msgbuf = msg->buf;

	SLIST_FOREACH(policy_db, &(epolicy->policy_head), policy_entry)
	{
		memcpy(inf_info.mac, policy_db->intfid, ETH_ALEN);
		inf_info.type = policy_db->eth.type;
		memcpy(msgbuf, &inf_info, sizeof(struct intf_type_info));
		msgbuf += sizeof(struct intf_type_info);
	}
	send_msg_to_mapfilter(msg);

}
#endif

void map_r2_notify_ts_config(void *eloop_ctx, void *timeout_ctx)
{
	struct p1905_managerd_ctx *ctx = (struct p1905_managerd_ctx*)eloop_ctx;
	struct ssid_2_vid_mapping *mapping = NULL, *mapping_org = NULL;
	struct traffics_policy *pts_policy = &ctx->map_policy.tpolicy;
	struct traffic_separation_db *policy_db = NULL;

	/*ts policy does not update, do not notify it*/
	if (!ctx->map_policy.setting.updated && !pts_policy->updated)
		return;

	mapfilter_set_ts_default_8021q(ctx->map_policy.setting.primary_vid, ctx->map_policy.setting.dft.PCP);

	if (pts_policy->SSID_num != 0) {
		mapping = os_zalloc(pts_policy->SSID_num * sizeof(struct ssid_2_vid_mapping));
		if (mapping == NULL)
			return;
		mapping_org = mapping;
	}

	SLIST_FOREACH(policy_db, &(pts_policy->policy_head), policy_entry)
	{
		mapping->vlan_id = policy_db->vid;
		os_memcpy(mapping->ssid, policy_db->SSID, policy_db->SSID_len);
		debug(DEBUG_OFF, "set_traffic_policy:%s --> %d \n", mapping->ssid, mapping->vlan_id);
		mapping++;
	}

	mapfilter_set_ts_policy(pts_policy->SSID_num, mapping_org);

	/*TODO: notify the WiFi driver*/
	_1905_notify_ts_bh_setting(ctx, pts_policy->SSID_num, mapping_org);
	os_sleep(0, 50000);
	_1905_notify_ts_fh_setting(ctx, pts_policy->SSID_num, mapping_org);
	os_free(mapping_org);

	/*reset policy*/
	ctx->map_policy.setting.updated = 0;
	pts_policy->updated = 0;

}

void cmd_set_ts_pvlan(struct p1905_managerd_ctx *ctx, char* buf)
{
	struct def_8021q_setting *def_setting = &ctx->map_policy.setting;
	unsigned int pvid = 0;

	sscanf(buf, "%d", &pvid);
	def_setting->primary_vid = (unsigned short)pvid;
	debug(DEBUG_ERROR, "primary_vid(%d)\n", def_setting->primary_vid);
//	mapfilter_set_ts_default_8021q(ctx->map_policy.setting.primary_vid, ctx->map_policy.setting.dft.PCP);
}

void cmd_set_ts_policy(struct p1905_managerd_ctx *ctx, char* buf)
{
	unsigned int vid = 0;
	char ssid[64];
	struct traffic_separation_db *trffic_policy = NULL;
	struct traffics_policy *policy = &ctx->map_policy.tpolicy;
	memset(ssid, 0, sizeof(ssid));
	/* <ssid> <vid> */
	sscanf(buf, "%s %d", ssid, &vid);

	trffic_policy = (struct traffic_separation_db *)malloc(sizeof(struct traffic_separation_db));
	if (!trffic_policy) {
		debug(DEBUG_ERROR, "alloc struct traffic_separation_db fail\n");
		return;
	}
	trffic_policy->SSID_len = strlen(ssid);
	debug(DEBUG_ERROR, "SSID_len(%d)\n", trffic_policy->SSID_len);

	trffic_policy->SSID = malloc(trffic_policy->SSID_len);
	if (!trffic_policy->SSID) {
		debug(DEBUG_ERROR, "kmalloc for trffic_policy->SSID fail\n");
		return;
	}

	memcpy(trffic_policy->SSID, ssid, trffic_policy->SSID_len);
	trffic_policy->vid = (unsigned short)vid;
	debug(DEBUG_ERROR, "SSID(%s) VID(%d)\n", trffic_policy->SSID, trffic_policy->vid);

	policy->SSID_num ++;
	debug(DEBUG_ERROR, "current SSID_num(%d)\n", policy->SSID_num);

	SLIST_INSERT_HEAD(&policy->policy_head, trffic_policy, policy_entry);
}

void cmd_set_ts_policy_done(struct p1905_managerd_ctx *ctx)
{
	ctx->map_policy.setting.updated = 1;
	ctx->map_policy.tpolicy.updated = 1;
	map_r2_notify_ts_config((void*)ctx, NULL);
}

void cmd_set_ts_policy_clear(struct p1905_managerd_ctx *ctx)
{
	ctx->map_policy.setting.primary_vid = VLAN_N_VID;
	ctx->map_policy.setting.dft.PCP = 0;
	delete_exist_traffic_policy(ctx, &ctx->map_policy.tpolicy);
	ctx->map_policy.setting.updated = 1;
	ctx->map_policy.tpolicy.updated = 1;
	map_r2_notify_ts_config((void*)ctx, NULL);
}



int parse_source_info_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == SOURCE_INFO_TYPE) {
        debug(DEBUG_WARN, "source info type:%d \n", *temp_buf);
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf += 2;

    debug(DEBUG_WARN, "source info len:%d \n", length);

    debug(DEBUG_WARN, "mac  %02x:%02x:%02x:%02x:%02x:%02x \n", PRINT_MAC(temp_buf));

    return (length+3);
}

int parse_tunneled_message_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == TUNNELED_MESSAGE_TYPE) {
        debug(DEBUG_WARN, "tunneled message  type:%d \n", *temp_buf);
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf += 2;

    debug(DEBUG_WARN, "tunneled message type len:%d \n", length);


    debug(DEBUG_WARN, "tunneled msg type:%d\n", *temp_buf);

    return (length+3);
}




int parse_tunneled_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == TUNNELED_TYPE) {
        debug(DEBUG_WARN, "tunneled tlv type:%d \n", *temp_buf);
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);
    temp_buf += 2;

    debug(DEBUG_WARN, "tunneled tlv len:%d \n", length);

    return (length+3);
}



int parse_tunneled_message(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == SOURCE_INFO_TYPE)
		{
			length = parse_source_info_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error source info tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == TUNNELED_MESSAGE_TYPE) {
			length = parse_tunneled_message_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error tunneled message type tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		}  else if(*temp_buf == TUNNELED_TYPE) {
			length = parse_tunneled_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error tunneled type tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check tunneled message tlvs\n");
		return -1;
	}
	return 0;
}


int parse_assoc_status_notification_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
    unsigned char *temp_buf;
    unsigned short length = 0;

    temp_buf = buf;

    if((*temp_buf) == ASSOCIATION_STATUS_NOTIFICATION_TYPE) {
        debug(DEBUG_WARN, "assoc status nofification type:%02x \n", *temp_buf);
        temp_buf++;
    }
    else {
        return -1;
    }

    //calculate tlv length
    length = *(unsigned short *)temp_buf;
    length = be2cpu16(length);

    temp_buf += 2;

    debug(DEBUG_WARN, "assoc status nofification len:%d \n", length);

    return (length+3);
}

int parse_assoc_status_notification_message(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == ASSOCIATION_STATUS_NOTIFICATION_TYPE)
		{
			length = parse_assoc_status_notification_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error assoc status notification type tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check tunneled message tlvs\n");
		return -1;
	}
	return 0;
}


int parse_cac_request_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == CAC_REQUEST_TYPE) {
		debug(DEBUG_WARN, "cac request:%02x \n", *temp_buf);
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	temp_buf += 2;

	debug(DEBUG_WARN, "cac request len:%d \n", length);

	return (length+3);
}




int parse_cac_request_message(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == CAC_REQUEST_TYPE)
		{
			length = parse_cac_request_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error cac request type tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check cac request tlvs\n");
		return -1;
	}
	return 0;
}



int parse_cac_terminate_type_tlv(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	unsigned char *temp_buf;
	unsigned short length = 0;

	temp_buf = buf;

	if((*temp_buf) == CAC_TERMINATION_TYPE) {
		debug(DEBUG_WARN, "cac terminate:%02x \n", *temp_buf);
		temp_buf++;
	}
	else {
		return -1;
	}

	//calculate tlv length
	length = *(unsigned short *)temp_buf;
	length = be2cpu16(length);

	temp_buf += 2;

	debug(DEBUG_WARN, "cac terminate len:%d \n", length);

	return (length+3);
}



int parse_cac_terminate_message(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		if(*temp_buf == CAC_TERMINATION_TYPE)
		{
			length = parse_cac_terminate_type_tlv(ctx,temp_buf);

			if(length < 0)
			{
				debug(DEBUG_ERROR, "error cac terminate type tlv\n");
				return -1;
			}
			integrity |= (1 << 0);
			if(ctx->Certification == 0)
			{
				store_revd_tlvs(ctx, temp_buf, length);
			}
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 1) {
		debug(DEBUG_ERROR, "error when check cac terminate tlvs\n");
		return -1;
	}
	return 0;
}


int parse_client_disassciation_stats_message(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		/* One STA MAC Address TLV */
		if(*temp_buf == STA_MAC_ADDRESS_TYPE)
		{
			integrity |= (1 << 0);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* One Disassociation Reason Code TLV */
		else if (*temp_buf == DISASSOCIATION_REASON_CODE_TYPE) {
			integrity |= (1 << 1);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* One Associated STA Traffic Stats TLV */
		else if (*temp_buf == ASSOC_STA_TRAFFIC_STATS_TYPE) {
			integrity |= (1 << 2);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 0x7) {
		debug(DEBUG_ERROR, "error when check client disassciation stats tlvs\n");
		return -1;
	}
	return 0;
}



int parse_failed_association_message(struct p1905_managerd_ctx *ctx, unsigned char *buf)
{
	int length =0;
	unsigned char *temp_buf;
	unsigned int integrity = 0;
	temp_buf = buf;
	reset_stored_tlv(ctx);

	while(1)
	{
		/* a Failed Association message including a STA MAC Address TLV */
		if(*temp_buf == STA_MAC_ADDRESS_TYPE)
		{
			integrity |= (1 << 0);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		}
		/* an Association Status Code TLV */
		else if (*temp_buf == ASSOCIATED_STATUS_CODE_TYPE) {
			integrity |= (1 << 1);
			length = get_tlv_len_by_tlvtype(temp_buf);
			store_revd_tlvs(ctx, temp_buf, length);
			temp_buf += length;
		} else if(*temp_buf == END_OF_TLV_TYPE) {
			break;
		} else {
			length = get_cmdu_tlv_length(temp_buf);
			temp_buf += length;
		}
	}
		/*check integrity*/
	if(integrity != 0x3) {
		debug(DEBUG_ERROR, "error when check client disassciation stats tlvs\n");
		return -1;
	}
	return 0;
}

#endif // #ifdef MAP_R2

int init_wireless_interface(struct p1905_managerd_ctx *ctx,
	struct interface_info_list_hdr *info)
{
	int i = 0, j = 0;

	for (i = 0; i < info->interface_count; i++) {
		for (j = 0; j < ctx->itf_number; j++) {
			if (!os_memcmp(info->if_info[i].if_name, ctx->itf[j].if_name, IFNAMSIZ)) {
				/*mac address*/
				if (os_memcmp(ctx->itf[j].mac_addr, info->if_info[i].if_mac_addr, ETH_ALEN)) {
					debug(DEBUG_ERROR, "mac[%s] mismatch\n", info->if_info[i].if_name);
					return -1;
				}

				debug(DEBUG_TRACE, "mac[%s] = %02x:%02x:%02x:%02x:%02x:%02x\n",
					info->if_info[i].if_name, PRINT_MAC(info->if_info[i].if_mac_addr));
				/*role*/
				if (os_strcmp((char *)info->if_info[i].if_role, "wiap") == 0) {
					ctx->itf[j].is_wifi_ap = 1;
				} else if (os_strcmp((char *)info->if_info[i].if_role, "wista") == 0) {
					ctx->itf[j].is_wifi_sta = 1;
				} else {
					debug(DEBUG_ERROR, "unknow role[%s] = %s\n",
						info->if_info[i].if_name, info->if_info[i].if_role);
					return -1;
				}
				debug(DEBUG_TRACE, "role[%s] = %s\n", info->if_info[i].if_name,
					info->if_info[i].if_role);
				/*band*/
				if (info->if_info[i].if_ch <= 14)
					ctx->itf[j].radio_band = 0x00;
				else
					ctx->itf[j].radio_band = 0x01;
				debug(DEBUG_TRACE, "band[%s] = %d\n", info->if_info[i].if_name,
					ctx->itf[j].radio_band);
				/*media type*/
				if (info->if_info[i].if_ch <= 14 && os_strcmp((char *)info->if_info[i].if_phymode, "B") == 0) {
					ctx->itf[j].media_type = IEEE802_11B_24G_GROUP;
				} else if (info->if_info[i].if_ch <= 14 && os_strcmp((char *)info->if_info[i].if_phymode, "G") == 0) {
					ctx->itf[j].media_type = IEEE802_11G_24G_GROUP;
				} else if (info->if_info[i].if_ch <= 14 && os_strcmp((char *)info->if_info[i].if_phymode, "N") == 0) {
					ctx->itf[j].media_type = IEEE802_11N_24G_GROUP;
				} else if (info->if_info[i].if_ch <= 14 && os_strcmp((char *)info->if_info[i].if_phymode, "AX") == 0) {
					ctx->itf[j].media_type = IEEE802_11AX_GROUP;
				} else if (info->if_info[i].if_ch > 14 && os_strcmp((char *)info->if_info[i].if_phymode, "A") == 0) {
					ctx->itf[j].media_type = IEEE802_11A_5G_GROUP;
				} else if (info->if_info[i].if_ch > 14 && os_strcmp((char *)info->if_info[i].if_phymode, "N") == 0) {
					ctx->itf[j].media_type = IEEE802_11N_5G_GROUP;
				} else if (info->if_info[i].if_ch > 14 && os_strcmp((char *)info->if_info[i].if_phymode, "AC") == 0) {
					ctx->itf[j].media_type = IEEE802_11AC_5G_GROUP;
				} else if (info->if_info[i].if_ch > 14 && os_strcmp((char *)info->if_info[i].if_phymode, "AX") == 0) {
					ctx->itf[j].media_type = IEEE802_11AX_GROUP;
				}else {
					debug(DEBUG_ERROR, "%s band[%d] & phymode[%s] mismatch\n",
						info->if_info[i].if_name, info->if_info[i].if_ch,
						info->if_info[i].if_phymode);
					return -1;
				}
				debug(DEBUG_TRACE, "media_type[%s] = %d\n", info->if_info[i].if_name,
					ctx->itf[j].media_type);
				/*radio indentifer*/
				os_memcpy(ctx->itf[j].identifier, info->if_info[i].identifier, ETH_ALEN);
				debug(DEBUG_TRACE, "identifier[%s] = %02x:%02x:%02x:%02x:%02x:%02x\n",
					info->if_info[i].if_name, PRINT_MAC(info->if_info[i].identifier));
				/*init vs_info*/
				ctx->itf[j].vs_info_length = 10;
				ctx->itf[j].vs_info = os_zalloc(ctx->itf[j].vs_info_length);
				if (!ctx->itf[j].vs_info) {
					debug(DEBUG_ERROR, "cannot alloc vs_info for %s\n",
						info->if_info[i].if_name);
					return -1;
				}
				os_memcpy(ctx->itf[j].vs_info, ctx->itf[j].mac_addr, ETH_ALEN);
				if (ctx->itf[j].is_wifi_sta) {
					*(ctx->itf[j].vs_info + 6) = 0x40;	  //non-ap station
					os_memset(ctx->itf[j].vs_info, 0, 6);
				}
				break;
			}
		}
		if (j >= ctx->itf_number) {
			debug(DEBUG_ERROR, "cannot find if_name[%s]\n", info->if_info[i].if_name);
			return -1;
		}
	}

	return 0;
}


int common_info_init(struct p1905_managerd_ctx *ctx)
{
	int i = 0, j = 0;

	if (!ctx->MAP_Cer) {
		/*reset 1905almac to ap interface mac*/
		for (i = 0; i < ctx->itf_number; i++) {
			if ((ctx->itf[i].is_wifi_ap ||
				(ctx->itf[i].media_type == IEEE802_3_GROUP && !ctx->itf[i].is_veth))
				&& os_memcmp(ctx->itf[i].mac_addr, ctx->p1905_al_mac_addr, ETH_ALEN)) {
				os_memcpy(ctx->p1905_al_mac_addr, ctx->itf[i].mac_addr, ETH_ALEN);
				debug(DEBUG_OFF, "reset almac to %02x:%02x:%02x:%02x:%02x:%02x\n",
					PRINT_MAC(ctx->p1905_al_mac_addr));
				/*reset to 1905 config file*/
				if (write_almac_to_1905_cfg_file(ctx->map_cfg_file,
						ctx->p1905_al_mac_addr)) {
					debug(DEBUG_ERROR, "write almac to 1905 cfg file fail\n");
					return -1;
				}
				if (set_opt_not_forward_dest(ctx->p1905_al_mac_addr) < 0) {
					debug(DEBUG_ERROR, "set almac to forward module fail\n");
					return -1;
				}
				create_topology_tree(ctx);
				break;
			}
		}
		if (i == ctx->itf_number) {
			debug(DEBUG_ERROR, "cannot find suitable almac\n");
			return -1;
		}
	}else {
		create_topology_tree(ctx);
	}

	ctx->br_cap[0].interface_num = ctx->itf_number - 1; /*skip virtual interface*/
	ctx->br_cap[0].itf_mac_list= os_malloc(ETH_ALEN * ctx->br_cap[0].interface_num);
	if (!ctx->br_cap[0].itf_mac_list) {
		debug(DEBUG_ERROR, "cannot alloc br_cap itf_mac_list\n");
		return -1;
	}
	for(i = 0, j = FIRST_VITUAL_ITF + 1; i < ctx->br_cap[0].interface_num; i++, j++)
		os_memcpy(ctx->br_cap[0].itf_mac_list + i * ETH_ALEN, ctx->itf[j].mac_addr, ETH_ALEN);

	for (i = 0; i < ctx->itf_number; i++) {
		os_memcpy(ctx->non_p1905_neighbor_dev[i].local_mac_addr, ctx->itf[i].mac_addr, ETH_ALEN);
	    LIST_INIT(&(ctx->non_p1905_neighbor_dev[i].non_p1905nbr_head));
		os_memcpy(ctx->p1905_neighbor_dev[i].local_mac_addr, ctx->itf[i].mac_addr, ETH_ALEN);
		LIST_INIT(&(ctx->p1905_neighbor_dev[i].p1905nbr_head));
		if (ctx->itf[i].is_wifi_ap)
			init_radio_info_by_intf(ctx, &ctx->itf[i]);
		if (!ctx->itf[i].is_wifi_sta)
			ctx->itf[i].trx_config = TX_MUL | RX_MUL | TX_UNI | RX_UNI;
	}

	return 0;
}

int write_almac_to_1905_cfg_file(char *cfg_file_name, unsigned char *almac)
{
	FILE *file;
	char *cpos = NULL, *apos = NULL;
	char almac_str[32];
	int cpos_cnt = 0, apos_cnt = 0, i = 0;
	char file_name[MAX_FILE_PATH_LENGTH] = {0};
	char content[1024] = {0};
	signed char ch = 0;

	strcpy(file_name, cfg_file_name);

	file = fopen(file_name, "r");
	if (!file) {
		debug(DEBUG_ERROR, "open MAP cfg file (%s) fail\n", file_name);
		return -1;
	}
	do {
    	ch = fgetc(file);
		content[i++] = ch;
	}while (EOF != ch && i < (sizeof(content) - 1));
	fclose(file);

	cpos = strstr(content, "map_controller_alid");
	apos = strstr(content, "map_agent_alid");

	if (!cpos) {
		debug(DEBUG_ERROR, "cannot find map_controller_alid str\n");
		return -1;
	}
	if (!apos) {
		debug(DEBUG_ERROR, "cannot find map_agent_alid str\n");
		return -1;
	}

	cpos_cnt = cpos - content + os_strlen("map_controller_alid");
	apos_cnt = apos - content + os_strlen("map_agent_alid");

	file = fopen(file_name, "r+");
	if (!file) {
		debug(DEBUG_ERROR, "open MAP cfg file (%s) fail at second time\n", file_name);
		return -1;
	}
	fseek(file, 0, SEEK_SET);
	fseek(file, cpos_cnt, SEEK_SET);
	do {
    	ch = fgetc(file);
		if (ch != '=' && ch != ' ')
			break;
	} while (ch != '\n' && ch != '\r');
	os_memset(almac_str, 0, sizeof(almac_str));
	sprintf(almac_str, "%02x:%02x:%02x:%02x:%02x:%02x",
		PRINT_MAC(almac));
	fseek(file, -1, SEEK_CUR);
	fputs(almac_str, file);

	fseek(file, 0, SEEK_SET);
	fseek(file, apos_cnt, SEEK_SET);
	do {
    	ch = fgetc(file);
		if (ch != '=' && ch != ' ')
			break;
	} while (ch != '\n' && ch != '\r');
	os_memset(almac_str, 0, sizeof(almac_str));
	sprintf(almac_str, "%02x:%02x:%02x:%02x:%02x:%02x",
		PRINT_MAC(almac));
	fseek(file, -1, SEEK_CUR);
	fputs(almac_str, file);
	fclose(file);

	return 0;
}


void mark_valid_topo_rsp_node(struct p1905_managerd_ctx *ctx)
{
    struct p1905_neighbor_info *p1905_neigbor = NULL;
    struct topology_response_db *tpgr = NULL;
	int i = 0;

	SLIST_FOREACH(tpgr, &ctx->topology_entry.tprdb_head, tprdb_entry) {
		debug(DEBUG_OFF, "mark dev(%02x:%02x:%02x:%02x:%02x:%02x) invalid\n",
			PRINT_MAC(tpgr->al_mac_addr));
		tpgr->valid = 0;
	}

	for(i = 0; i < ctx->itf_number; i++) {
		LIST_FOREACH(p1905_neigbor, &(ctx->p1905_neighbor_dev[i].p1905nbr_head), p1905nbr_entry) {
			tpgr = find_topology_rsp_by_almac(ctx, p1905_neigbor->al_mac_addr);
			if (tpgr && !tpgr->valid) {
				tpgr->valid = 1;
				debug(DEBUG_OFF, "mark dev(%02x:%02x:%02x:%02x:%02x:%02x) valid\n",
					PRINT_MAC(tpgr->al_mac_addr));
				update_topology_info(ctx, tpgr);
			}
		}
	}
}

void update_topology_info(
	struct p1905_managerd_ctx *ctx, struct topology_response_db *tpgr)
{
	struct device_info_db *dev_info = NULL;
	struct p1905_neighbor_device_db *p1905_neighbor = NULL;
    struct topology_response_db *tpgr_child = NULL;

	SLIST_FOREACH(dev_info, &tpgr->devinfo_head, devinfo_entry) {
		SLIST_FOREACH(p1905_neighbor, &dev_info->p1905_nbrdb_head, p1905_nbrdb_entry) {
			tpgr_child = find_topology_rsp_by_almac(ctx,
							p1905_neighbor->p1905_neighbor_al_mac);
			if (tpgr_child && !tpgr_child->valid) {
				tpgr_child->valid = 1;
				debug(DEBUG_OFF, "mark dev(%02x:%02x:%02x:%02x:%02x:%02x) valid\n",
					PRINT_MAC(tpgr_child->al_mac_addr));
				update_topology_info(ctx, tpgr_child);
			}
		}
	}
}

struct topology_response_db *find_topology_rsp_by_almac(
	struct p1905_managerd_ctx *ctx, unsigned char *almac)
{
    struct topology_response_db *tpgr_db = NULL;

	SLIST_FOREACH(tpgr_db, &ctx->topology_entry.tprdb_head, tprdb_entry) {
		if (!os_memcmp(tpgr_db->al_mac_addr, almac, ETH_ALEN))
			return tpgr_db;
	}

	return tpgr_db;
}
