#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <assert.h>
#include <linux/if_ether.h>
#include <asm/byteorder.h>
#include <stdlib.h>
#include "p1905_managerd.h"
#include "cmdu_message.h"
#include "cmdu_tlv_parse.h"
#include "cmdu.h"
#include "os.h"
#include "byteorder.h"
#include "debug.h"
#include "_1905_lib_io.h"
#include "file_io.h"

#include "topology.h"
#include "multi_ap.h"

struct _find_leaf {
	unsigned char al_mac[ETH_ALEN];
	unsigned char find_result;
};

struct leaf *create_leaf(unsigned char *al_mac)
{
	struct leaf *node = NULL;

	node = (struct leaf*)os_malloc(sizeof(struct leaf));
	if (!node) {
		debug(DEBUG_ERROR, "allocate leaf memory fail for (%02x:%02x:%02x:%02x:%02x:%02x)\n",
			PRINT_MAC(al_mac));
		return NULL;
	}

	dl_list_init(&node->children);
	node->config_status = UNCONFIGURED;
	node->band_cap = 0;
	os_memcpy(node->al_mac, al_mac, ETH_ALEN);
	os_get_time(&node->create_time);
	node->renew_send_round = 0;
	return node;
}

void free_leaf(struct p1905_managerd_ctx *ctx, struct leaf *del_leaf)
{
	struct agent_list_db *agent_info = NULL;

	find_agent_info(ctx, del_leaf->al_mac, &agent_info);
	if (agent_info) {
		/*reset the band configured status of agent which leaves from current network*/
		agent_info->band_configured = 0;
		debug(DEBUG_WARN, "reset agent(%02x:%02x:%02x:%02x:%02x:%02x) band configured\n",
			PRINT_MAC(agent_info->almac));
	}
	dl_list_del(&del_leaf->entry);
	os_free(del_leaf);
}

struct topology_response_db *lookup_tprdb_by_almac(struct p1905_managerd_ctx *ctx, unsigned char *al_mac)
{
	struct topology_response_db *rpdb = NULL;

	SLIST_FOREACH(rpdb, &ctx->topology_entry.tprdb_head, tprdb_entry) {
		if (!os_memcmp(rpdb->al_mac_addr, al_mac, ETH_ALEN))
			break;
	}

	return rpdb;
}

unsigned char is_neighbor(struct topology_response_db *rpdb, unsigned char *al_mac)
{
	struct device_info_db *dev_info = NULL;
	struct p1905_neighbor_device_db *neighbor_db = NULL;

	SLIST_FOREACH(dev_info, &rpdb->devinfo_head, devinfo_entry) {
		SLIST_FOREACH(neighbor_db, &dev_info->p1905_nbrdb_head, p1905_nbrdb_entry) {
			if (!os_memcmp(neighbor_db->p1905_neighbor_al_mac, al_mac, ETH_ALEN))
				return 1;
		}
	}

	return 0;
}

struct leaf *lookup_childleaf_by_almac(struct leaf *current_leaf, unsigned char *almac)
{
	struct leaf *child_leaf = NULL;
	unsigned char found = 0;

	dl_list_for_each(child_leaf, &current_leaf->children, struct leaf, entry) {
		if (!os_memcmp(child_leaf->al_mac, almac, ETH_ALEN)) {
			found = 1;
			break;
		}
	}
	if (found == 0)
		child_leaf = NULL;

	return child_leaf;
}

void find_leaf_handler(void *context, void* parent_leaf, void *current_leaf, void *data)
{
	struct leaf *cur_leaf = (struct leaf *)current_leaf;
	struct _find_leaf *find_data = (struct _find_leaf *)data;

	if (find_data->find_result == 1)
		return;

	if (!os_memcmp(cur_leaf->al_mac, find_data->al_mac, ETH_ALEN))
		find_data->find_result = 1;
}

unsigned char is_leaf_exist(struct p1905_managerd_ctx * ctx, struct leaf *current_leaf, unsigned char *almac)
{
	struct _find_leaf find;

	os_memset(&find, 0, sizeof(struct _find_leaf));
	os_memcpy(find.al_mac, almac, ETH_ALEN);

	trace_topology_tree_cb(ctx, current_leaf, (topology_tree_cb_func)find_leaf_handler, (void*)&find);
	return find.find_result;
}

unsigned char build_tree_from_current_leaf(struct p1905_managerd_ctx *ctx, struct topology_response_db *current_rsp, struct leaf *current_leaf, struct leaf *parent_leaf)
{
	struct device_info_db *dev_info = NULL;
	struct p1905_neighbor_device_db *neighbor_db = NULL;
	struct topology_response_db *child_rpdb = NULL;
	struct leaf *child_leaf = NULL;

	SLIST_FOREACH(dev_info, &current_rsp->devinfo_head, devinfo_entry) {
		SLIST_FOREACH(neighbor_db, &dev_info->p1905_nbrdb_head, p1905_nbrdb_entry) {
			/*1. if neighbor exist in topology response db
			   2. if neighbor's 1905 neighbor tlv match the current devices's al mac and no leaf exist in current node
			       then create the child leaf
			   2.5 check whether this leaf exist, if yes, do not add this one until the exist one has deleted
			   3. and from this new/existed child leaf, buld all its child leaves*/
			child_rpdb = NULL;
			child_leaf = NULL;
			/*skip the parent leaf*/
			if (!os_memcmp(neighbor_db->p1905_neighbor_al_mac, parent_leaf->al_mac, ETH_ALEN))
				continue;
			/*1.*/
			child_rpdb = lookup_tprdb_by_almac(ctx, neighbor_db->p1905_neighbor_al_mac);
			if (!child_rpdb) {
				debug(DEBUG_TRACE, "almac(%02x:%02x:%02x:%02x:%02x:%02x) not exist in topology rsp db\n",
					PRINT_MAC(neighbor_db->p1905_neighbor_al_mac));
				continue;
			}

			/*2.*/
			if (!is_neighbor(child_rpdb, current_leaf->al_mac)) {
				debug(DEBUG_ERROR, "current device is not a neighbor of almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(child_rpdb->al_mac_addr));
				continue;
			}

			child_leaf = lookup_childleaf_by_almac(current_leaf, child_rpdb->al_mac_addr);
			if (!child_leaf) {
				/*2.5*/
				if (is_leaf_exist(ctx, ctx->root_leaf, child_rpdb->al_mac_addr)) {
					debug(DEBUG_ERROR, "current leaf(%02x:%02x:%02x:%02x:%02x:%02x) exist, do not create an new one\n",
						PRINT_MAC( child_rpdb->al_mac_addr));
					continue;
				}

				child_leaf = create_leaf(child_rpdb->al_mac_addr);

				if(!child_leaf)
					continue;
				debug(DEBUG_OFF, "\t\033[1m\033[45;33m leaf add!current(%p)(%02x:%02x:%02x:%02x:%02x:%02x)----->child(%p)(%02x:%02x:%02x:%02x:%02x:%02x)\033[0m\n",
					current_leaf, PRINT_MAC(current_leaf->al_mac), child_leaf, PRINT_MAC(child_leaf->al_mac));
				dl_list_add(&current_leaf->children, &child_leaf->entry);
			}

			/*3.*/
			build_tree_from_current_leaf(ctx, child_rpdb, child_leaf, current_leaf);
		}
	}

	return 0;
}

int delete_tree_from_current_leaf(struct p1905_managerd_ctx *ctx, struct leaf *current_leaf)
{
	struct leaf *child_leaf = NULL, *child_leaf_next = NULL;

	dl_list_for_each_safe(child_leaf, child_leaf_next, &current_leaf->children, struct leaf, entry) {
		debug(DEBUG_OFF, "\t\033[1m\033[40;31m leaf delete from current(%p)(%02x:%02x:%02x:%02x:%02x:%02x)\033[0m\n",
				child_leaf, PRINT_MAC(child_leaf->al_mac));
		delete_tree_from_current_leaf(ctx, child_leaf);

		free_leaf(ctx, child_leaf);
	}

	return 0;
}

int detect_deleted_leaf_from_current_leaf(struct p1905_managerd_ctx *ctx, struct leaf *current_leaf)
{
	struct leaf *child_leaf = NULL, *child_leaf_next = NULL;
	struct topology_response_db *current_rsp = NULL, *child_rpdb = NULL;

	current_rsp = lookup_tprdb_by_almac(ctx, current_leaf->al_mac);

	if (!current_rsp) {
		debug(DEBUG_ERROR, "BUG here!!!current leaf for almac(%02x:%02x:%02x:%02x:%02x:%02x) not exist in tpr db\n",
			PRINT_MAC(current_leaf->al_mac));
		return -1;
	}

	dl_list_for_each_safe(child_leaf, child_leaf_next, &current_leaf->children, struct leaf, entry) {
		child_rpdb = lookup_tprdb_by_almac(ctx, child_leaf->al_mac);
		/*maybe this condition should be add more, like check neighbor of child rsp db to see if it is current device*/
		if (child_rpdb && is_neighbor(current_rsp, child_leaf->al_mac)) {
			detect_deleted_leaf_from_current_leaf(ctx, child_leaf);
		} else {
			/*if child leaf is not my neighbor, should delete this child leaf and all its children leaves of this child leaf*/
			debug(DEBUG_OFF, "\t\033[1m\033[40;31m leaf delete from current(%p)(%02x:%02x:%02x:%02x:%02x:%02x)\033[0m\n",
				child_leaf, PRINT_MAC(child_leaf->al_mac));
			delete_tree_from_current_leaf(ctx, child_leaf);
			free_leaf(ctx, child_leaf);
		}
	}

	return 0;
}

/*init the root leaf of the tree, the root leaf always should represent current device*/
int create_topology_tree(struct p1905_managerd_ctx* ctx)
{
	struct leaf *root_leaf = NULL;

	if (ctx->root_leaf) {
		debug(DEBUG_ERROR, "root leaf already exist\n");
		goto fail;
	}

	root_leaf = create_leaf(ctx->p1905_al_mac_addr);
	if (!root_leaf)
		goto fail;

	ctx->root_leaf = root_leaf;
	return 0;
fail:
	return -1;
}

int update_topology_tree(struct p1905_managerd_ctx* ctx)
{
	struct p1905_neighbor_info *neighor_info = NULL;
	struct topology_response_db *child_rpdb = NULL;
	struct leaf *child_leaf = NULL, *child_leaf_next = NULL;
	int i = 0;
	unsigned char found = 0;

	if (!ctx->root_leaf) {
		debug(DEBUG_ERROR, "root leaf not exist, could not update topology tree\n");
		goto fail;
	}

	for (i = 0; i < ctx->itf_number; i++) {
		/*check the leaves that should be added to topology tree*/
		LIST_FOREACH(neighor_info, &ctx->p1905_neighbor_dev[i].p1905nbr_head, p1905nbr_entry) {
			debug(DEBUG_TRACE, "%s, %d\n", __FUNCTION__, __LINE__);
			child_rpdb = NULL;
			child_leaf = NULL;
			/*1. if neighbor exist in topology response db
			   2. if neighbor's 1905 neighbor tlv match the current devices's al mac and no leaf exist in current node
			       then create the child leaf
			   2.5 check whether this leaf exist, if yes, do not add this one until the exist one has deleted
			   3. and from this new/existed child leaf, rebuld its child leaves*/
			/*1.*/
			child_rpdb = lookup_tprdb_by_almac(ctx, neighor_info->al_mac_addr);
			if (!child_rpdb) {
				debug(DEBUG_TRACE, "almac(%02x:%02x:%02x:%02x:%02x:%02x) not exist in topology rsp db\n",
					PRINT_MAC(neighor_info->al_mac_addr));
				continue;
			}
			debug(DEBUG_TRACE, "%s, %d\n", __FUNCTION__, __LINE__);

			/*2.*/
			if (!is_neighbor(child_rpdb, ctx->root_leaf->al_mac)) {
				debug(DEBUG_ERROR, "current device is not a neighbor of almac(%02x:%02x:%02x:%02x:%02x:%02x)\n",
					PRINT_MAC(child_rpdb->al_mac_addr));
				continue;
			}
			debug(DEBUG_TRACE, "%s, %d\n", __FUNCTION__, __LINE__);

			child_leaf = lookup_childleaf_by_almac(ctx->root_leaf, neighor_info->al_mac_addr);
			if (!child_leaf) {
				/*2.5*/
				if (is_leaf_exist(ctx, ctx->root_leaf, neighor_info->al_mac_addr)) {
					debug(DEBUG_ERROR, "current leaf(%02x:%02x:%02x:%02x:%02x:%02x) exist, do not create an new one\n",
						PRINT_MAC(neighor_info->al_mac_addr));
					continue;
				}

				child_leaf = create_leaf(neighor_info->al_mac_addr);
				if (!child_leaf)
					continue;
				debug(DEBUG_OFF, "\t\033[1m\033[45;33m leaf add!root(%p)(%02x:%02x:%02x:%02x:%02x:%02x)----->child(%p)(%02x:%02x:%02x:%02x:%02x:%02x)\033[0m\n",
					ctx->root_leaf, PRINT_MAC(ctx->root_leaf->al_mac), child_leaf, PRINT_MAC(child_leaf->al_mac));
				dl_list_add(&ctx->root_leaf->children, &child_leaf->entry);

				debug(DEBUG_WARN, "%s, %d\n", __FUNCTION__, __LINE__);
			}
			/*3.*/

			debug(DEBUG_WARN, "%s, %d\n", __FUNCTION__, __LINE__);
			build_tree_from_current_leaf(ctx, child_rpdb, child_leaf, ctx->root_leaf);
		}
	}

	/*check the leaves that should be deleted from topology tree*/
	dl_list_for_each_safe(child_leaf, child_leaf_next, &ctx->root_leaf->children, struct leaf, entry) {
		found = 0;
		for (i = 0; i < ctx->itf_number; i++) {
			LIST_FOREACH(neighor_info, &ctx->p1905_neighbor_dev[i].p1905nbr_head, p1905nbr_entry) {
				if (!os_memcmp(child_leaf->al_mac, neighor_info->al_mac_addr, ETH_ALEN)) {
					found = 1;
					break;
				}
			}
			if (found)
				break;
		}

		child_rpdb = lookup_tprdb_by_almac(ctx, child_leaf->al_mac);

		if (!found || !child_rpdb) {
			debug(DEBUG_OFF, "\t\033[1m\033[40;31m leaf delete from current(%p)(%02x:%02x:%02x:%02x:%02x:%02x)\033[0m\n",
				child_leaf, PRINT_MAC(child_leaf->al_mac));
			delete_tree_from_current_leaf(ctx, child_leaf);
			free_leaf(ctx, child_leaf);
		} else {
			detect_deleted_leaf_from_current_leaf(ctx, child_leaf);
		}
	}

	return 0;
fail:
	return -1;
}

int clear_topology_tree(struct p1905_managerd_ctx* ctx)
{
	delete_tree_from_current_leaf(ctx, ctx->root_leaf);
	os_free(ctx->root_leaf);
	ctx->root_leaf = NULL;

	return 0;
}

int trace_topology_tree_cb(struct p1905_managerd_ctx* ctx, struct leaf *current_leaf, topology_tree_cb_func cb, void *data)
{
	struct leaf *child_leaf = NULL, *child_leaf_next = NULL;

	if (!current_leaf)
		return 0;

	dl_list_for_each_safe(child_leaf, child_leaf_next, &current_leaf->children, struct leaf, entry) {
		cb((void*)ctx, (void*)current_leaf, (void*)child_leaf, data);
		trace_topology_tree_cb(ctx, child_leaf, cb, data);
	}

	return 0;
}

int topology_tree_travel_preorder(struct p1905_managerd_ctx* ctx, struct leaf *current_leaf)
{
	struct leaf *child_leaf = NULL, *child_leaf_next = NULL;
	leaf_info leaf_in;
	struct agent_list_db *agent_info = NULL;

	if (!current_leaf)
		return 0;

	os_memset(&leaf_in, 0, sizeof(leaf_in));

	/*push to stack*/
	debug(DEBUG_OFF, "push to stack:(%02x:%02x:%02x:%02x:%02x:%02x)\n",
		PRINT_MAC(current_leaf->al_mac));
	os_memcpy(leaf_in.al_mac, current_leaf->al_mac, ETH_ALEN);
	if (os_memcmp(current_leaf->al_mac, ctx->p1905_al_mac_addr, ETH_ALEN)) {
		find_agent_info(ctx, current_leaf->al_mac, &agent_info);
		if (!agent_info) {
			debug(DEBUG_ERROR, "error! no agent info exist for (%02x:%02x:%02x:%02x:%02x:%02x)\n",
				PRINT_MAC(current_leaf->al_mac));
			leaf_in.band_cap = (BAND_2G_CAP | BAND_5GL_CAP | BAND_5GH_CAP);
			leaf_in.radio_cap = (RADIO_2G_CAP | RADIO_5GL_CAP | RADIO_5GH_CAP);
		} else {
			leaf_in.band_cap = agent_info->band_cap;
			leaf_in.radio_cap = agent_info->radio_cap;
		}
	}
	push(&ctx->topo_stack, leaf_in);

	dl_list_for_each_safe(child_leaf, child_leaf_next, &current_leaf->children, struct leaf, entry) {
		topology_tree_travel_preorder(ctx, child_leaf);
	}

	return 0;
}

